/*   EXTRAITS DE LA LICENCE
     Copyright CEA, contributeurs : Luc BILLARD et Damien
     CALISTE, laboratoire L_Sim, (2001-2005)
  
     Adresse ml :
     BILLARD, non joignable par ml ;
     CALISTE, damien P caliste AT cea P fr.

     Ce logiciel est un programme informatique servant  visualiser des
     structures atomiques dans un rendu pseudo-3D. 

     Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
     respectant les principes de diffusion des logiciels libres. Vous pouvez
     utiliser, modifier et/ou redistribuer ce programme sous les conditions
     de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
     sur le site "http://www.cecill.info".

     Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
     pris connaissance de la licence CeCILL, et que vous en avez accept les
     termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
     Copyright CEA, contributors : Luc BILLARD et Damien
     CALISTE, laboratoire L_Sim, (2001-2005)

     E-mail address:
     BILLARD, not reachable any more ;
     CALISTE, damien P caliste AT cea P fr.

     This software is a computer program whose purpose is to visualize atomic
     configurations in 3D.

     This software is governed by the CeCILL  license under French law and
     abiding by the rules of distribution of free software.  You can  use, 
     modify and/ or redistribute the software under the terms of the CeCILL
     license as circulated by CEA, CNRS and INRIA at the following URL
     "http://www.cecill.info". 

     The fact that you are presently reading this means that you have had
     knowledge of the CeCILL license and that you accept its terms. You can
     find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <gtk/gtk.h>

#include <ab6_symmetry.h>

#include <visu_object.h>
#include <gtk_main.h>
#include <renderingBackend/visu_windowInterface.h>
#include <gtk_renderingWindowWidget.h>
#include <openGLFunctions/interactive.h>

#define ABINIT_SYMMETRIES_INFO \
  _("left-button\t\t: select one atom to get the equivalent ones\n"		\
    "right-button\t\t: switch to observe")

static gulong onSelection_id, onSpin_id;
static Ab6Symmetry *sym;
static GtkWidget *lblSymName, *lblSymId, *lblSymCod, *lblSymWarning;
static GtkWidget *spinNode, *vbox, *vboxSym;
static guint timeout = 0;

static void onSelection(VisuInteractive *inter _U_,
			PickMesure *pickMesure, gpointer data);
static void onSymmetryToggled(GtkToggleButton *toggle, gpointer data);
static void onSymmetryClicked(GtkButton *button, gpointer data);
static void onVisuDataChanged(VisuObject *visu, VisuData *dataObj, gpointer data);
static void onSpinNode(GtkSpinButton *button, gpointer data);

static void updateSymmetries(VisuData *dataObj);
static void getEquivalents(VisuData *dataObj, PickMesure *pick, VisuNode *node);

GtkWidget* buildTab(GtkMain *main _U_, gchar **label,
		    gchar **help, GtkWidget **radio)
{
  GtkWidget *wd, *hbox;
  RenderingWindow *window;
  VisuData *data;

  window = visuRenderingWindowGet_current();
  g_return_val_if_fail(window, (GtkWidget*)0);
  data = renderingWindowGet_visuData(window);
  g_return_val_if_fail(data, (GtkWidget*)0);

  *label = _("Symmetries");
  *help  = g_strdup(ABINIT_SYMMETRIES_INFO);

  vbox = gtk_vbox_new(FALSE, 0);

  sym = (Ab6Symmetry*)0;

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

  *radio = gtk_radio_button_new_with_mnemonic
    (NULL, _("Analyse the symmetries of the system"));
  gtk_box_pack_start(GTK_BOX(hbox), *radio, FALSE, FALSE, 0);
  gtk_widget_set_name(*radio, "message_radio");
  g_signal_connect(G_OBJECT(*radio), "toggled",
		   G_CALLBACK(onSymmetryToggled), (gpointer)0);
  wd = gtk_button_new_with_mnemonic(_("Compute symmetries"));
  g_signal_connect(G_OBJECT(wd), "clicked",
		   G_CALLBACK(onSymmetryClicked), (gpointer)0);
  gtk_box_pack_end(GTK_BOX(hbox), wd, FALSE, FALSE, 0);

  vboxSym = gtk_vbox_new(FALSE, 0);
  gtk_widget_set_sensitive(vboxSym, FALSE);
  gtk_box_pack_start(GTK_BOX(vbox), vboxSym, TRUE, TRUE, 0);

  /* A message for ABINIT. */
  wd = gtk_label_new(_("<span size=\"smaller\">The symmetry routines"
		       " are provided by ABINIT ("
		       "<span font_desc=\"courier\" color=\"blue\">"
		       "http://www.abinit.org</span>).</span>"));
  gtk_misc_set_alignment(GTK_MISC(wd), 1.0, 0.5);
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_box_pack_end(GTK_BOX(vbox), wd, FALSE, FALSE, 5);

  /* The labels showing the symmetry. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxSym), hbox, FALSE, FALSE, 10);
  wd = gtk_label_new(_("<b>Space group:</b>"));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_misc_set_padding(GTK_MISC(wd), 10, 0);
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  wd = gtk_label_new(_("<span font_desc=\"courier\" color=\"blue\">"
		       "http://en.wikipedia.org/wiki/Space_group</span>"));
  gtk_label_set_selectable(GTK_LABEL(wd), TRUE);
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_box_pack_end(GTK_BOX(hbox), wd, FALSE, FALSE, 10);
  wd = gtk_image_new_from_stock(GTK_STOCK_HELP, GTK_ICON_SIZE_MENU);
  gtk_box_pack_end(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxSym), hbox, FALSE, FALSE, 0);
  wd = gtk_label_new(_("Crystal system:"));
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  lblSymName = gtk_label_new("");
  gtk_misc_set_alignment(GTK_MISC(lblSymName), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), lblSymName, TRUE, TRUE, 5);
  wd = gtk_label_new(_("space group:"));
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  lblSymId = gtk_label_new("");
  gtk_label_set_use_markup(GTK_LABEL(lblSymId), TRUE);
  gtk_misc_set_alignment(GTK_MISC(lblSymId), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), lblSymId, TRUE, TRUE, 5);
  wd = gtk_label_new(_("point group:"));
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  lblSymCod = gtk_label_new("");
  gtk_misc_set_alignment(GTK_MISC(lblSymCod), 0., 0.5);
  gtk_box_pack_start(GTK_BOX(hbox), lblSymCod, TRUE, TRUE, 5);
  lblSymWarning = gtk_label_new(_("<span color=\"red\">Warning:</span> the Bravais lattice determined from the primitive vectors is more symmetric than the real one obtained from coordinates (printed)."));
  gtk_label_set_use_markup(GTK_LABEL(lblSymWarning), TRUE);
  gtk_label_set_line_wrap(GTK_LABEL(lblSymWarning), TRUE);
  gtk_label_set_line_wrap_mode(GTK_LABEL(lblSymWarning), PANGO_WRAP_WORD);
  gtk_box_pack_start(GTK_BOX(vboxSym), lblSymWarning, FALSE, FALSE, 0);

  /* The interface to choose one atom to select. */
  wd = gtk_label_new(_("<b>Equivalent atoms:</b>"));
  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
  gtk_misc_set_padding(GTK_MISC(wd), 10, 0);
  gtk_box_pack_start(GTK_BOX(vboxSym), wd, FALSE, FALSE, 10);
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxSym), hbox, FALSE, FALSE, 0);
  wd = gtk_label_new(_("Visualise the equivalent nodes of node:"));
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
  spinNode = gtk_spin_button_new_with_range(0, 1, 1);
  onSpin_id = g_signal_connect(G_OBJECT(spinNode), "value-changed",
			       G_CALLBACK(onSpinNode), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hbox), spinNode, FALSE, FALSE, 5);
  wd = gtk_label_new(_(" or pick directly."));
  gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);

  gtk_widget_show_all(vbox);
  gtk_widget_hide(lblSymWarning);
    
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onVisuDataChanged), (gpointer)0);

  onVisuDataChanged((VisuObject*)0, data, (gpointer)0);

  return vbox;
}

void startSelect(RenderingWindow *window)
{
  VisuInteractive *inter;

  inter = renderingWindowGet_interactive(window);
  visuInteractiveSet_type(inter, interactive_pick);
  onSelection_id = g_signal_connect(G_OBJECT(inter),
				    "selection", G_CALLBACK(onSelection),
				    (gpointer)renderingWindowGet_visuData(window));
  pickMesureSet_formatedOutput(visuInteractiveGet_pickMesure(inter), FALSE);
}
void stopSelect(RenderingWindow *window)
{
  VisuInteractive *inter;

  inter = renderingWindowGet_interactive(window);
  g_signal_handler_disconnect(G_OBJECT(inter), onSelection_id);
  visuInteractiveSet_type(inter, interactive_none);
}

static void onSelection(VisuInteractive *inter _U_,
			PickMesure *pickMesure, gpointer data)
{
  PickMesureType type;
  VisuNode *node;
  
  if (pickMesureGet_newsAvailable(pickMesure, &type) && type == PICK_SELECTED)
    {
      node = pickMesureGet_selectedNode(pickMesure);
      getEquivalents(VISU_DATA(data), pickMesure, node);
      g_signal_handler_block(G_OBJECT(spinNode), onSpin_id);
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinNode), node->number + 1);
      g_signal_handler_unblock(G_OBJECT(spinNode), onSpin_id);
    }
}

static void onSymmetryToggled(GtkToggleButton *toggle _U_, gpointer data _U_)
{
}
static void onSymmetryClicked(GtkButton *button _U_, gpointer data _U_)
{
  RenderingWindow *window;
  VisuData *dataObj;

  /* Get the current VisuData object. */
  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  dataObj = renderingWindowGet_visuData(window);

  updateSymmetries(dataObj);
}
static void onVisuDataChanged(VisuObject *visu _U_, VisuData *dataObj,
			      gpointer data _U_)
{
  VisuDataIter iter;

  gtk_widget_set_sensitive(vbox, (dataObj !=(VisuData*)0));

  updateSymmetries((VisuData*)0);
  
  if (dataObj)
    {
      visuDataIter_new(dataObj, &iter);
      gtk_spin_button_set_range(GTK_SPIN_BUTTON(spinNode), 0, iter.idMax + 1);
    }
}
static void onSpinNode(GtkSpinButton *button, gpointer data _U_)
{
  RenderingWindow *window;
  VisuData *dataObj;
  PickMesure *pick;
  VisuNode *node;

  if (gtk_spin_button_get_value(button) == 0)
    return;

  /* Get the current VisuData object. */
  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  dataObj = renderingWindowGet_visuData(window);
  pick = visuInteractiveGet_pickMesure(renderingWindowGet_interactive(window));
  node = visuDataGet_nodeFromNumber(dataObj,
				    (int)gtk_spin_button_get_value(button) - 1);

  getEquivalents(dataObj, pick, node);
}
static gboolean onRemoveEquivalents(gpointer data _U_)
{
  return FALSE;
}

static void removeEquivalents(gpointer data)
{
  int *ids, i;
  RenderingWindow *window;
  VisuData *dataObj;
  PickMesure *pick;

  ids = (int*)data;

  /* Get the current VisuData object. */
  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  dataObj = renderingWindowGet_visuData(window);
  pick = visuInteractiveGet_pickMesure(renderingWindowGet_interactive(window));

  for (i = 0; ids[i] >= 0; i++)
    pickMesureSet_highlight(pick, (guint)ids[i], FALSE);

  pickMesureRebuild_classicalList(dataObj);
  g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));

  g_free(data);
}
static void getEquivalents(VisuData *dataObj, PickMesure *pick, VisuNode *node)
{
  int i, j, p, nSym;
  int *nodes, *ids;
  gboolean found;
#if GLIB_MINOR_VERSION > 13
#define fact 1
#define G_TIMEOUT_ADD_FULL g_timeout_add_seconds_full
#else
#define fact 1000
#define G_TIMEOUT_ADD_FULL g_timeout_add_full
#endif

  if (!sym)
    updateSymmetries(dataObj);
  g_return_if_fail(sym);

  if (ab6_symmetry_get_equivalent_atom(sym, &nodes, &nSym, node->number + 1) ==
      AB6_NO_ERROR)
    {
      ids = g_malloc(sizeof(guint) * (nSym + 1));
      p = 0;
      for (i = 0; i < nSym; i++)
	{
	  found = FALSE;
	  for ( j = 0; (j < p) && !found; j++)
	    found = (nodes[i * 4 + 3] == (ids[j] + 1));
	  if (!found)
	    {
	      ids[p] = nodes[i * 4 + 3] - 1;
	      p += 1;
	    }
	}
      ids[p] = -1;
      g_free(nodes);

      /* We remove possible earlier timeout. */
      if (timeout > 0)
	g_source_remove(timeout);

      /* Set the new highlights. */
      for (i = 0; ids[i] >= 0; i++)
	{
	  DBG_fprintf(stderr, " | %d\n", ids[i]);
	  pickMesureSet_highlight(pick, (guint)ids[i], TRUE);
	}
      /* Add the new timeout. */
      timeout = G_TIMEOUT_ADD_FULL(G_PRIORITY_DEFAULT, 3 * fact,
				   onRemoveEquivalents, ids, removeEquivalents);
				 
      pickMesureRebuild_classicalList(dataObj);
      g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));
    }
}

static void updateSymmetries(VisuData *dataObj)
{
  float xred0[3];
  double box[3][3], *xred, genAfm[3];
  int *typat, i, grpId, grpMagnId, centerId, nbrv;
  VisuDataIter iter;
  char *ptGrp, *spGrp;
  gchar *str;
  Ab6SymmetryMat *brvSyms;
  int brvMat[3][3];
  gchar *bravais[7] = {"triclinic", "monoclinic", "orthorhombic",
		       "tetragonal", "trigonal", "hexagonal", "cubic"};
  gchar *center[7] = {"F", "F", "I", "P", "A", "B", "C"};
  Ab6Error errno;

  DBG_fprintf(stderr, "Abinit: upadte symmetries from %p.\n", (gpointer)dataObj);

  if (sym)
    ab6_symmetry_free(sym);
  sym = (Ab6Symmetry*)0;

  if (dataObj)
    {
      sym = ab6_symmetry_new();
      ab6_symmetry_set_tolerance(sym, 1e-5);
      visuDataGet_boxMatrixD0(dataObj, box);
      ab6_symmetry_set_lattice(sym, box);
      
      visuDataIter_new(dataObj, &iter);
      i = 0;
      typat = g_malloc(sizeof(int) * iter.nAllStoredNodes);
      xred = g_malloc(sizeof(double) * 3 * iter.nAllStoredNodes);
      for (visuDataIter_start(dataObj, &iter); iter.node;
	   visuDataIter_next(dataObj, &iter))
	{
	  typat[i] = iter.iElement;
	  visuDataGet_reducedNodePosition(dataObj, iter.node, xred0);
	  xred[3 * i + 0] = (double)xred0[0];
	  xred[3 * i + 1] = (double)xred0[1];
	  xred[3 * i + 2] = (double)xred0[2];
	  i += 1;
	}
      ab6_symmetry_set_structure(sym, iter.nAllStoredNodes, typat, xred);
      g_free(typat);
      g_free(xred);

      /* We get then the space group. */
      errno = ab6_symmetry_get_group(sym, &ptGrp, &spGrp, &grpId,
				     &grpMagnId, genAfm);
      if (errno == AB6_NO_ERROR || errno == AB6_ERROR_SYM_BRAVAIS_XRED)
	{
	  str = g_strdup_printf("%s (#%d)", spGrp, grpId);
	  gtk_label_set_text(GTK_LABEL(lblSymId), str);
	  g_free(str);
	  gtk_label_set_text(GTK_LABEL(lblSymCod), ptGrp);
	  
	  g_free(ptGrp);
	  g_free(spGrp);
	}
      else
	{
	  gtk_label_set_markup(GTK_LABEL(lblSymId), _("<i>not primitive</i>"));
	  gtk_label_set_text(GTK_LABEL(lblSymCod), "");
	}
      if (ab6_symmetry_get_bravais(sym, brvMat, &grpId,
				   &centerId, &nbrv, &brvSyms) == AB6_NO_ERROR)
	{
	  g_free(brvSyms);
	  str = g_strdup_printf("%s (%s)", _(bravais[grpId - 1]), center[centerId+3]);
	  gtk_label_set_text(GTK_LABEL(lblSymName), str);
	  g_free(str);
	}
      else
	gtk_label_set_text(GTK_LABEL(lblSymName), "!");

      /* If the bravais lattice doesn't match with the xred bravais
	 lattice, we print a message. */
      if (errno == AB6_ERROR_SYM_BRAVAIS_XRED)
	gtk_widget_show(lblSymWarning);
      else
	gtk_widget_hide(lblSymWarning);
    }
  else
    {
      gtk_label_set_text(GTK_LABEL(lblSymName), "");
      gtk_label_set_text(GTK_LABEL(lblSymId), "");
      gtk_label_set_text(GTK_LABEL(lblSymCod), "");
    }
  gtk_widget_set_sensitive(vboxSym, (dataObj != (VisuData*)0));
}
