#!/bin/bash

# Copyright (c) 2012 The Chromium OS Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

# Abort on error.
set -e

# Load common constants and variables.
. "$(dirname "$0")/common.sh"

usage() {
    echo "Usage $PROG image [config]"
}

main() {
    # We want to catch all the discrepancies, not just the first one.
    # So, any time we find one, we set testfail=1 and continue.
    # When finished we will use testfail to determine our exit value.
    local testfail=0

    if [ $# -ne 1 ] && [ $# -ne 2 ]; then
        usage
        exit 1
    fi

    local image="$1"

    # Default config location: same name/directory as this script,
    # with a .config file extension, ie ensure_no_nonrelease_files.config.
    local configfile="$(dirname "$0")/${0/%.sh/.config}"
    # Or, maybe a config was provided on the command line.
    if [ $# -eq 2 ]; then
        configfile="$2"
    fi
    # Either way, load test-expectations data from config.
    . "$configfile" || return 1

    local rootfs=$(make_temp_dir)
    mount_image_partition_ro "$image" 3 "$rootfs"

    for file in ${RELEASE_FILE_BLACKLIST[@]}; do
        if [ -e "$rootfs/$file" ]; then
            echo "FAIL: $file exists in this image!"
            ls -al "$rootfs/$file"
            testfail=1
        fi
    done

    # Some things which used to be flag-files, checked-for by this
    # test, are now tracked as use-flags.
    local useflag_path="$rootfs/etc/session_manager_use_flags.txt"
    for prefix in dangerous_ test_; do
        local matches=$(grep "^$prefix" "$useflag_path")
        if [ -n "$matches" ]; then
            echo "FAIL: Found non-release use flags in $useflag_path:"
            echo "$matches"
            testfail=1
        fi
    done

    exit $testfail
}
main "$@"
