/*
 * VCD Player plugin for VDR
 * vcd_menu_control.c: Menu control of VCD replay
 *
 * See the README file for copyright information and how to reach the author.
 *
 * This code is distributed under the terms and conditions of the
 * GNU GENERAL PUBLIC LICENSE. See the file COPYING for details.
 *
 */


#include <stdio.h>
#include <typeinfo>

#include <vdr/interface.h>
#include <vdr/status.h>
#include "vcd_i18n.h"
#include "vcd_menu.h"
#include "vcd_func.h"
#include "vcd_setup.h"
#include "vcd_menu_control.h"


#define MODETIMEOUT       3 // seconds


// --- cMenuSpiControl -------------------------------------------------------

int cMenuSpiControl::spi = 0;
cVcd *cMenuSpiControl::vcd = 0;

cMenuSpiControl::cMenuSpiControl(void)
: cVcdViewerControl(spi, vcd)
{
  visible = modeOnly = false;
}

cMenuSpiControl::~cMenuSpiControl()
{
  cStatus::MsgReplaying(this, NULL);
  Stop();
}

void cMenuSpiControl::SetItem(int Spi, cVcd *Vcd)
{
  cControl *control = cControl::Control();
  if (control && typeid(*control)==typeid(cMenuVcdControl))
     static_cast<cMenuVcdControl*>(control)->Stop();
  if (control && typeid(*control)==typeid(cMenuSpiControl)
        && strcmp(Vcd->DeviceName(),static_cast<cMenuSpiControl*>(control)->DeviceName())==0) {
     static_cast<cMenuSpiControl*>(control)->GotoItem(Spi);
     return;
     }

  spi = Spi;
  vcd = Vcd;
  cControl::Launch(new cMenuSpiControl);
}

void cMenuSpiControl::Hide(void)
{
  if (visible) {
     Interface->Close();
     needsFastResponse = visible = false;
     modeOnly = false;
     }
}

eOSState cMenuSpiControl::ProcessKey(eKeys Key)
{
  if (!Active())
     return osEnd;
  switch (Key) {
     // Positioning:
     case k4:       SkipItems(-1); break;
     case k5:       ToggleStillRes(); break;
     case k6:       SkipItems(1); break;
     case kBlue:    Hide();
                    Stop();
                    vcd->Close();
                    return osEnd;
     default: {
        switch (Key) {
           // Editing: (?)
           default: {
              switch (Key) {
                 // Menu control:
                 case kOk:   break;
                 case kBack: Hide();
                             Stop();
                             vcd->Close();
                             return osEnd;
                 default:    return osUnknown;
                 }
              }
           }
        }
     }
  return osContinue;
}


// --- cVcdProgressBar -------------------------------------------------------

class cVcdProgressBar : public cBitmap {
 protected:
    int total;
    int Pos(int p) { return p * width / total; }
 public:
    cVcdProgressBar(int Width, int Height, int Current, int Total);
};

cVcdProgressBar::cVcdProgressBar(int Width, int Height, int Current, int Total)
:cBitmap(Width, Height, 2)
{
  total = Total;
  if (total > 0) {
     int p = Pos(Current);
     Fill(0, 0, p, Height - 1, clrGreen);
     Fill(p + 1, 0, Width - 1, Height - 1, clrWhite);
  }
}



// --- cMenuVcdControl ------------------------------------------------------

int cMenuVcdControl::track = 0;
cVcd *cMenuVcdControl::vcd = 0;
char *cMenuVcdControl::title = 0;

cMenuVcdControl::cMenuVcdControl(void)
: cVcdPlayerControl(track, vcd)
{
  visible = modeOnly = shown = displayFrames = false;
  timeoutShow = 0;
  timeSearchActive = false;
  lastCurrent = lastTotal = -1;
  cStatus::MsgReplaying(this, "VCD");
}

cMenuVcdControl::~cMenuVcdControl()
{
  Hide();
  cStatus::MsgReplaying(this, NULL);
  Stop();
}

void cMenuVcdControl::SetTrack(int Track, cVcd *Vcd)
{
  cControl *control = cControl::Control();
  if (control && typeid(*control)==typeid(cMenuSpiControl))
     static_cast<cMenuSpiControl*>(control)->Stop();
  if (control && typeid(*control)==typeid(cMenuVcdControl)
        && strcmp(Vcd->DeviceName(),static_cast<cMenuVcdControl*>(control)->DeviceName())==0) {
     static_cast<cMenuVcdControl*>(control)->GotoTrack(Track);
     return;
     }

  free(title);
  track = Track;
  vcd = Vcd;
  title = NULL;
  cControl::Launch(new cMenuVcdControl);
}

const int cMenuVcdControl::LastReplayed(void)
{
  return track;
}

void cMenuVcdControl::Show(int Seconds)
{
  if (modeOnly)
     Hide();
  if (!visible) {
     shown = ShowProgress(true);
     timeoutShow = (shown && Seconds>0) ? time(0) + Seconds : 0;
     }
}

void cMenuVcdControl::Hide(void)
{
  if (visible) {
     Interface->Close();
     needsFastResponse = visible = false;
     modeOnly = false;
     }
}

void cMenuVcdControl::DisplayAtBottom(const char *s)
{
  if (s) {
     int w = cOsd::WidthInCells(s);
     int d = max(Width() - w, 0) / 2;
     if (modeOnly) //XXX remove when displaying replay mode differently
        Interface->Fill(0, -1, Interface->Width(), 1, clrTransparent);
     Interface->Write(d, -1, s);
     Interface->Flush();
     }
  else
     Interface->Fill(12, 2, Width() - 22, 1, clrBackground);
}

void cMenuVcdControl::ShowMode(void)
{
  if (Setup.ShowReplayMode && !timeSearchActive) {
     bool Play, Forward;
     int Speed;
     if (GetReplayMode(Play, Forward, Speed)) {
        bool NormalPlay = (Play && Speed == -1);

        if (!visible) {
           if (NormalPlay)
              return; // no need to do indicate ">" unless there was a different mode displayed before
           // open small display
           /*XXX change when displaying replay mode differently
           Interface->Open(9, -1);
           Interface->Clear();
           XXX*/
           Interface->Open(0, -1); //XXX remove when displaying replay mode differently
           visible = modeOnly = true;
           }

        if (modeOnly && !timeoutShow && NormalPlay)
           timeoutShow = time(NULL) + MODETIMEOUT;
        const char *Mode;
        if (Speed == -1) Mode = Play    ? "  >  " : " ||  ";
        else if (Play)   Mode = Forward ? " X>> " : " <<X ";
        else             Mode = Forward ? " X|> " : " <|X ";
        char buf[16];
        strncpy(buf, Mode, sizeof(buf));
        char *p = strchr(buf, 'X');
        if (p)
           *p = Speed > 0 ? '1' + Speed - 1 : ' ';

        eDvbFont OldFont = Interface->SetFont(fontFix);
        DisplayAtBottom(buf);
        Interface->SetFont(OldFont);
        }
     }
}

bool cMenuVcdControl::ShowProgress(bool Initial)
{
  int Current, Total;

  if (GetIndex(Current, Total) && Total > 0) {
     if (!visible) {
        Interface->Open(Setup.OSDwidth, -3);
        needsFastResponse = visible = true;
        }
     if (Initial) {
        Interface->Clear();
        lastCurrent = lastTotal = -1;
        }
     if (title) 
        Interface->Write(0, 0, title);
     if (Total != lastTotal) {
        Interface->Write(-7, 2, IndexToHMSF(Total));
        if (!Initial)
           Interface->Flush();
        }
     if (Current != lastCurrent || Total != lastTotal) {
#ifdef DEBUG_OSD
        int p = Width() * Current / Total;
        Interface->Fill(0, 1, p, 1, clrGreen);
        Interface->Fill(p, 1, Width() - p, 1, clrWhite);
#else
        cVcdProgressBar ProgressBar(Width() * cOsd::CellWidth(), cOsd::LineHeight(), Current, Total);
        Interface->SetBitmap(0, cOsd::LineHeight(), ProgressBar);
        if (!Initial)
           Interface->Flush();
#endif
        Interface->Write(0, 2, IndexToHMSF(Current, displayFrames));
        Interface->Flush();
        lastCurrent = Current;
        }
     lastTotal = Total;
     ShowMode();
     return true;
     }
  return false;
}

void cMenuVcdControl::TimeSearchDisplay(void)
{
  char buf[64];
  strcpy(buf, tr("Jump: "));
  int len = strlen(buf);
  char h10 = '0' + (timeSearchTime >> 24);
  char h1  = '0' + ((timeSearchTime & 0x00FF0000) >> 16);
  char m10 = '0' + ((timeSearchTime & 0x0000FF00) >> 8);
  char m1  = '0' + (timeSearchTime & 0x000000FF);
  char ch10 = timeSearchPos > 3 ? h10 : '-';
  char ch1  = timeSearchPos > 2 ? h1  : '-';
  char cm10 = timeSearchPos > 1 ? m10 : '-';
  char cm1  = timeSearchPos > 0 ? m1  : '-';
  sprintf(buf + len, "%c%c:%c%c", ch10, ch1, cm10, cm1);
  DisplayAtBottom(buf);
}

void cMenuVcdControl::TimeSearchProcess(eKeys Key)
{
#define STAY_SECONDS_OFF_END 10
  int Seconds = (timeSearchTime >> 24) * 36000 + ((timeSearchTime & 0x00FF0000) >> 16) * 3600 + ((timeSearchTime & 0x0000FF00) >> 8) * 600 + (timeSearchTime & 0x000000FF) * 60;
  int Current = (lastCurrent / FRAMESPERSEC);
  int Total = (lastTotal / FRAMESPERSEC);
  switch (Key) {
    case k0 ... k9:
         if (timeSearchPos < 4) {
            timeSearchTime <<= 8;
            timeSearchTime |= Key - k0;
            timeSearchPos++;
            TimeSearchDisplay();
            }
         break;
    case kFastRew:
    case kLeft:
    case kFastFwd:
    case kRight: {
         int dir = ((Key == kRight || Key == kFastFwd) ? 1 : -1);
         if (dir > 0)
            Seconds = min(Total - Current - STAY_SECONDS_OFF_END, Seconds);
         SkipSeconds(Seconds * dir);
         timeSearchActive = false;
         }
         break;
    case kPlay:
    case kUp:
    case kPause:
    case kDown:
         Seconds = min(Total - STAY_SECONDS_OFF_END, Seconds);
         Goto(Seconds * FRAMESPERSEC, Key == kDown || Key == kPause);
         timeSearchActive = false;
         break;
    default:
         timeSearchActive = false;
         break;
    }
  if (!timeSearchActive) {
     if (timeSearchHide)
        Hide();
     else
        DisplayAtBottom();
     ShowMode();
     }
}

void cMenuVcdControl::TimeSearch(void)
{
  timeSearchTime = timeSearchPos = 0;
  timeSearchHide = false;
  if (modeOnly)
     Hide();
  if (!visible) {
     Show();
     if (visible)
        timeSearchHide = true;
     else
        return;
     }
  timeoutShow = 0;
  TimeSearchDisplay();
  timeSearchActive = true;
}

eOSState cMenuVcdControl::ProcessKey(eKeys Key)
{
  if (!Active())
     return osEnd;
  if (visible) {
     if (timeoutShow && time(0) > timeoutShow) {
        Hide();
        ShowMode();
        timeoutShow = 0;
        }
     else if (modeOnly)
        ShowMode();
     else
        shown = ShowProgress(!shown) || shown;
     }
  bool DisplayedFrames = displayFrames;
  displayFrames = false;
  if (timeSearchActive && Key != kNone) {
     TimeSearchProcess(Key);
     return osContinue;
     }
  switch (Key) {
     case k1: SkipEntries(-1); break;
     case k3: SkipEntries(1); break;
     case k4: SkipTracks(-1); break;
     case k5: SkipTracks(0); break;
     case k6: SkipTracks(1); break;
     default: break;
     }
  asprintf(&title, "VideoCD Track %02d", GetTrack());
  bool DoShowMode = true;
  switch (Key) {
     // Positioning:
     case kPlay:
     case kUp:      Play(); break;
     case kPause:
     case kDown:    Pause(); break;
     case kFastRew|k_Release:
     case kLeft|k_Release:
                    if (Setup.MultiSpeedMode) break;
     case kFastRew:
     case kLeft:    Backward(); break;
     case kFastFwd|k_Release:
     case kRight|k_Release:
                    if (Setup.MultiSpeedMode) break;
     case kFastFwd:
     case kRight:   Forward(); break;
     case kRed:     TimeSearch(); break;
     case kGreen|k_Repeat:
     case kGreen:   SkipSeconds(-60); break;
     case kYellow|k_Repeat:
     case kYellow:  SkipSeconds(60); break;
     case kStop:
     case kBlue:    Hide();
                    Stop();
                    vcd->Close();
                    return osEnd;
     default: {
        DoShowMode = false;
        switch (Key) {
           // Editing: (?)

           default: {
              displayFrames = DisplayedFrames;
              switch (Key) {
                 // Menu control:
                 case kOk:   if (visible && !modeOnly) {
                                Hide();
                                DoShowMode = true;
                                }
                             else
                                Show();
                             break;
                 case kBack: Hide();
                             Stop();
                             vcd->Close();
                             return osEnd;
                 default:    return osUnknown;
                 }
              }
           }
        }
     }
  if (DoShowMode)
     ShowMode();
  if (DisplayedFrames && !displayFrames)
     Interface->Fill(0, 2, 11, 1, clrBackground);
  return osContinue;
}

