#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "../include/fio.h"
#include "../include/string.h"

#include "cdialog.h"

#include "scratchpad.h"
#include "scratchpadfio.h"

#include "vma.h"
#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


gint ScratchPadLoadFromFile(
	vma_scratch_pad_struct *sp, const gchar *filename
);
gint ScratchPadSaveToFile(
	vma_scratch_pad_struct *sp, const gchar *filename
);

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))


/*
 *	Loads scratchpad data from the specified file into the given
 *	scratchpad sp.
 *
 *	Any loaded data in the scratchpad sp will be deallocated first.
 *
 *	Returns non-zero on error.
 */
gint ScratchPadLoadFromFile(
	vma_scratch_pad_struct *sp, const gchar *filename
)
{
	GtkCList *clist;
	FILE *fp;
	struct stat stat_buf;
	gchar *parm;
	vma_scratch_pad_row_struct rd;


	if((sp == NULL) || (filename == NULL))
	    return(-1);

	/* Apply and data from widgets into the clist. */
	ScratchPadSyncData(sp);

	clist = (GtkCList *)sp->clist;
	if(clist == NULL)
	    return(-1);

	/* Freeze clist and deallocate any loaded data on the
	 * scratchpad.
	 */
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);


	/* Check if given file exists. */
	if(stat(filename, &stat_buf))
	{
/*
	    fprintf(
		stderr,
		"%s: No such file.\n",
		filename
	    );
 */
	    gtk_clist_thaw(clist);
	    return(-1);
	}
	if(!S_ISREG(stat_buf.st_mode))
	{
	    fprintf(
		stderr,
		"%s: Not a file.\n",
		filename
	    );
	    gtk_clist_thaw(clist);
	    return(-1);
	}

	/* Open file. */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	{
/*
	    fprintf(
		stderr,
		"%s: Cannot open.\n",
		filename
	    );
 */
	    gtk_clist_thaw(clist);
	    return(-1);
	}

	/* Reset contexts and begin reading file. */
	parm = NULL;
	memset(&rd, 0x00, sizeof(vma_scratch_pad_row_struct));
	while(1)
	{
	    parm = FSeekNextParm(
		fp, parm,
		VMA_CFG_COMMENT_CHAR, VMA_CFG_DELIMINATOR_CHAR
	    );
	    if(parm == NULL)
		break;

	    /* Handle by parm. */

/* Macro to deallocate contexts and reset their values. */
#define RESET_CONTEXT	\
{ \
 free(rd.comment); \
 rd.comment = NULL; \
 memset(&rd, 0x00, sizeof(vma_scratch_pad_row_struct)); \
}

	    /* Create new vertex data block. */
	    if(!strcasecmp(parm, "BeginVertex"))
	    {
		RESET_CONTEXT
	    }
	    /* End vertex data block. */
	    else if(!strcasecmp(parm, "EndVertex"))
	    {
		ScratchPadRowAppend(
		    sp, &rd.v, &rd.n, &rd.tc, rd.comment
		);
	    }

	    /* Vertex. */
	    else if(!strcasecmp(parm, "Vertex"))
	    {
		double vd[3];

		FGetValuesF(fp, vd, 3);

		rd.v.x = vd[0];
		rd.v.y = vd[1];
		rd.v.z = vd[2];
	    }
	    /* Normal. */
	    else if(!strcasecmp(parm, "Normal"))
	    {
		double vd[3];

		FGetValuesF(fp, vd, 3);

		rd.n.x = vd[0];
		rd.n.y = vd[1];
		rd.n.z = vd[2];
	    }
	    /* TexCoord. */
	    else if(!strcasecmp(parm, "TexCoord"))
	    {
		double vd[3];

		FGetValuesF(fp, vd, 3);

		rd.tc.x = vd[0];
		rd.tc.y = vd[1];
		rd.tc.z = vd[2];
	    }
	    /* Comment. */
	    else if(!strcasecmp(parm, "Comment"))
	    {
		gchar *strptr = FGetStringLined(fp);

		free(rd.comment);
		rd.comment = strptr;
	    }

	    /* Unsupported parameter. */
	    else
	    {
		fprintf(
		    stderr,
 "%s: Unsupported parameter `%s'.\n",
		    filename, parm
		);
		FSeekNextLine(fp);
	    }
	}

	/* Reset context. */
	RESET_CONTEXT

	/* Close file. */
	FClose(fp);
	fp = NULL;

	gtk_clist_thaw(clist);

#undef RESET_CONTEXT

	return(0);
}


/*
 *	Saves contents of the given scratchpad to the given file.
 */
gint ScratchPadSaveToFile(
	vma_scratch_pad_struct *sp, const gchar *filename
)
{
	gint row;
	GtkCList *clist;
	vma_scratch_pad_row_struct *rd_ptr;
	mp_vertex_struct *v;
	const gchar *cstrptr;
	FILE *fp;
	struct stat stat_buf;


	if((sp == NULL) || (filename == NULL))
	    return(-1);

	/* Apply and data from widgets into the clist. */
	ScratchPadSyncData(sp);

	clist = (GtkCList *)sp->clist;
	if(clist == NULL)
	    return(-1);

	if(!stat(filename, &stat_buf))
	{
	    if(S_ISDIR(stat_buf.st_mode))
	    {
		fprintf(
		    stderr,
		    "%s: Cannot save directory as a file.\n",
		    filename
		);
		return(-1);
	    }
	}

	/* Open file for writing. */
	fp = FOpen(filename, "wb");


	/* Itterate through each row on the clist and save each row data
	 * to file.
	 */
	for(row = 0; row < clist->rows; row++)
	{
	    rd_ptr = (vma_scratch_pad_row_struct *)gtk_clist_get_row_data(
		clist, row
	    );
	    if(rd_ptr == NULL)
		continue;

	    /* Begin writing data for this row. */

	    /* Begin vertex data block. */
	    fprintf(
		fp,
		"BeginVertex\n"
	    );

	    /* Vertex. */
	    v = &rd_ptr->v;
	    fprintf(
		fp,
		"    Vertex = %f %f %f\n",
		v->x, v->y, v->z
	    );

	    /* Normal. */
	    v = &rd_ptr->n;
	    fprintf(
		fp,
		"    Normal = %f %f %f\n",
		v->x, v->y, v->z
	    );

	    /* TexCoord. */
	    v = &rd_ptr->tc;
	    fprintf(
		fp,
		"    TexCoord = %f %f %f\n",
		v->x, v->y, v->z
	    );


	    /* Begin writing comment, we need to change any occurance
	     * of \r or \n into a the two characters '\\' and '\n'
	     * occurance.
	     */
	    cstrptr = (const gchar *)rd_ptr->comment;
	    if(cstrptr != NULL)
	    {
		fprintf(
		    fp,
		    "    Comment = "
		);

		while((*cstrptr) != '\0')
		{
		    if(ISCR(*cstrptr))
		    {
			fputc('\\', fp);
			fputc('\n', fp);
		    }
		    else
		    {
			fputc(*cstrptr, fp);
		    }
		    cstrptr++;
		}

		fputc('\n', fp);
	    }

	    /* End vertex data block. */
	    fprintf(
		fp,
		"EndVertex\n"
	    );
	}

	/* Close file. */
	FClose(fp);
	fp = NULL;

	return(0);
}
