/*
 * Copyright (C) 2003  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "ExtendedImageView.h"

#include <Foundation/NSString.h>
#include <Foundation/NSException.h>
#include <Foundation/NSNotification.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSDictionary.h>
#include <AppKit/NSAffineTransform.h>
#include <AppKit/NSColor.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSImageRep.h>
#include <AppKit/NSBezierPath.h>
#include <AppKit/NSCursor.h>
#include <AppKit/NSScrollView.h>
#include <AppKit/NSWindow.h>


/*
 * Initialize constants.
 */
NSString* N_SelectionChanged     = @"N_SelectionChanged";
NSString* N_SelectionRemoved     = @"N_SelectionRemoved";
NSString* N_ScrollingRequested   = @"N_ScrollingRequested";
NSString* UserInfoKeySelection    = @"Selection";
NSString* UserInfoKeyScrollAmount = @"ScrollAmount";


/*
 * Non-Public methods.
 */
@interface ExtendedImageView(Private)
- (void) _adjustScaling;
- (void) _scrollMouseDown: (NSEvent*)event;
- (void) _scrollMouseDragged: (NSEvent*)event;
- (void) _scrollMouseUp: (NSEvent*)event;
- (void) _selectionMouseDown: (NSEvent*)event;
- (void) _selectionMouseDragged: (NSEvent*)event;
- (void) _selectionMouseUp: (NSEvent*)event;
- (void) _setDocumentCursor: (NSCursor*)cursor;
@end


/*
 */
@implementation ExtendedImageView

/*
 * Designated initializer.
 */
- (id) initWithFrame: (NSRect)frame
{
   if ((self = [super initWithFrame: frame]))
   {
      selection = NSMakeRect(0, 0, 0, 0);
      drawSelection = NO;
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc ExtendedImageView, retain count is %d", [self retainCount]);

   [super dealloc];
}


- (void) setSelection: (NSRect)aSelection
{
   /*
   NSLog(@"set selection to x:%f, y:%f, w:%f, h:%f",
         aSelection.origin.x, aSelection.origin.y,
         aSelection.size.width, aSelection.size.height);
   */
   NSRect updateRect = NSUnionRect(aSelection, selection);
   selection = aSelection;
   drawSelection = YES;
   [self setNeedsDisplayInRect: updateRect];
}


- (void) removeSelection
{
   [self setNeedsDisplayInRect: selection];
   drawSelection = NO;
}


- (void) drawRect: (NSRect)aRect
{
   [super drawRect: aRect];

   if (drawSelection)
   {
      [[[NSColor blueColor] colorWithAlphaComponent: 0.5] set];
      [NSBezierPath fillRect: selection];
   }
}


/*
 * Setup a new selection rectangle. The old selection
 * will be deleted if present.
 */
- (void) mouseDown: (NSEvent*)event
{
   if ([event modifierFlags] & NSShiftKeyMask)
   {
      [self _selectionMouseDown: event];
   }
   else
   {
      [self _scrollMouseDown: event];
   }
}


/*
 *When the mouse is dragged, the selection will
 * be extended to where the user drags.
 */
- (void) mouseDragged: (NSEvent*)event
{
   if ([event modifierFlags] & NSShiftKeyMask)
   {
      [self _selectionMouseDragged: event];
   }
   else
   {
      [self _scrollMouseDragged: event];
   }
}


/*
 * Finish selection.
 */
- (void) mouseUp: (NSEvent*)event
{
   if ([event modifierFlags] & NSShiftKeyMask)
   {
      [self _selectionMouseUp: event];
   }
   else
   {
      [self _scrollMouseUp: event];
   }
}

@end


@implementation ExtendedImageView(Private)

/*
 * The following methods implement scrolling by dragging
 * the mouse.
 */
- (void) _scrollMouseDown: (NSEvent*)event
{
   // TODO: set grab mouse cursor
   return;
}


- (void) _scrollMouseDragged: (NSEvent*)event
{
   NSSize scrollAmount = NSMakeSize(-[event deltaX], -[event deltaY]);

   [[NSNotificationCenter defaultCenter]
      postNotificationName: N_ScrollingRequested
      object: self
      userInfo: [NSDictionary dictionaryWithObject: [NSValue valueWithSize: scrollAmount]
                              forKey: UserInfoKeyScrollAmount]];
}


- (void) _scrollMouseUp: (NSEvent*)event
{
   // TODO: reset mouse cursor
   return;
}


/*
 * The following methods implement text selection with the mouse
 * (dragging while holding the shift key)
 */
- (void) _selectionMouseDown: (NSEvent*)event
{
   [self removeSelection];

   // post a notification to observers that the selection has changed
   // (doing this in removeSelection will cause unwanted loops because
   // removeSelection can also be called from outside).
   // post a notification to observers that the selection has been removed
   [[NSNotificationCenter defaultCenter]
      postNotificationName: N_SelectionRemoved
      object: self];
      
   selectionStart = [self convertPoint: [event locationInWindow] fromView: nil];
}


- (void) _selectionMouseDragged: (NSEvent*)event
{
   NSPoint mouseLocation = [self convertPoint: [event locationInWindow] fromView: nil];

   NSRect newSelection;

   if (mouseLocation.x < selectionStart.x)
   {
      newSelection.origin.x = mouseLocation.x;
      newSelection.size.width = selectionStart.x - mouseLocation.x;
   }
   else
   {
      newSelection.origin.x = selectionStart.x;
      newSelection.size.width = mouseLocation.x - selectionStart.x;
   }

   if (mouseLocation.y < selectionStart.y)
   {
      newSelection.origin.y = mouseLocation.y;
      newSelection.size.height = selectionStart.y - mouseLocation.y;
   }
   else
   {
      newSelection.origin.y = selectionStart.y;
      newSelection.size.height = mouseLocation.y - selectionStart.y;
   }

   [self setSelection: newSelection];

   // post a notification to observers that the selection has changed
   // (doing this in setSelection will cause unwanted loops because
   // setSelection can also be called from outside).
   [[NSNotificationCenter defaultCenter]
      postNotificationName: N_SelectionChanged
      object: self
      userInfo: [NSDictionary dictionaryWithObject: [NSValue valueWithRect: newSelection]
                              forKey: UserInfoKeySelection]];
}


- (void) _selectionMouseUp: (NSEvent*)event
{
   [self setNeedsDisplay: YES];
}


/*
 * Adjust the scale factor for the image so that the
 * image will completely fill the image view.
 */
- (void) _adjustScaling
{
   float   wScale;
   float   hScale;
   id      scaleTransform;
   NSSize  imageSize;
   NSSize  viewSize;

   imageSize = [[self image] size];
   viewSize  = [self frame].size;
      
   wScale = viewSize.width / imageSize.width;
   hScale = viewSize.height / imageSize.height;

   scaleTransform = [NSAffineTransform transform];
   [scaleTransform scaleBy: (wScale < hScale ? wScale : hScale)];
   [scaleTransform concat];
}


/*
 * Look for a superview that responds to setDocumentCursor
 * and set the document cursor of this view to the specified
 * cursor.
 */
- (void) _setDocumentCursor: (NSCursor*)cursor
{
   id view;

   view = [self superview];
   while(view)
   {
      if ([view respondsToSelector: @selector(setDocumentCursor:)])
      {
         [view setDocumentCursor: cursor];
         [[view window] resetCursorRects];
         NSLog(@"document cursor set");
         break;
      }
      view = [view superview];
   }
  
   if (!view)
   {
      NSLog(@"Warning: Unable to set document cursor");
   }
}

@end


