/* TIFF parts: Copyright (c) 1988, 1990 by Sam Leffler.
 * All rights reserved.
 *
 * This file is provided for unrestricted use provided that this
 * legend is included on all tape media and as a part of the
 * software program in whole or part.  Users may copy, modify or
 * distribute this file at will.
 * -----------------------------
 * Modifications for VIPS:  Kirk Martinez 1994
 * 22/11/94 JC
 *	- more general
 *	- memory leaks fixed
 * 20/3/95 JC
 *	- TIFF error handler added
 *	- read errors detected correctly
 *
 * Modified to handle LAB in tiff format.
 * It convert LAB-tiff format to IM_TYPE_LABQ in vips format.
 *  Copyright July-1995 Ahmed Abbood.
 *
 *
 * 19/9/95 JC
 *	- now calls TIFFClose ... stupid
 * 25/1/96 JC
 *	- typo on MINISBLACK ...
 * 7/4/97 JC
 *	- completely redone for TIFF 6
 *	- now full baseline TIFF 6 reader, and does CIELAB as well
 * 11/4/97 JC
 *	- added partial read for tiled images
 * 23/4/97 JC
 *	- extra subsample parameter
 *	- im_istiffpyramid() added
 * 5/12/97 JC
 *	- if loading YCbCr, convert to IM_CODING_LABQ
 * 1/5/98 JC
 *	- now reads 16-bit greyscale and RGB
 * 26/10/98 JC
 *	- now used "rb" mode on systems that need binary open
 * 12/11/98 JC
 *	- no sub-sampling if sub == 1
 * 26/2/99 JC
 *	- ooops, else missing for subsample stuff above
 * 2/10/99 JC
 *	- tiled 16-bit greyscale read was broken
 *	- added mutex for TIFF*() calls
 * 11/5/00 JC
 *	- removed TIFFmalloc/TIFFfree usage
 * 23/4/01 JC
 *	- HAVE_TIFF turns on TIFF goodness
 * 24/5/01 JC
 *	- im_tiff2vips_header() added
 * 11/7/01 JC
 *	- subsample now in input filename
 *	- ... and it's a page number (from 0) instead
 * 21/8/02 JC
 *	- now reads CMYK
 *	- hmm, dpi -> ppm conversion was wrong!
 * 10/9/02 JC
 *	- oops, handle TIFF errors better
 * 2/12/02 JC
 *	- reads 8-bit RGBA
 * 12/12/02 JC
 *	- reads 16-bit LAB
 * 13/2/03 JC
 *	- pixels/cm res read was wrong
 * 17/11/03 Andrey Kiselev
 *	- read 32-bit float greyscale and rgb
 * 5/4/04
 *	- better handling of edge tiles (thanks Ruven)
 * 16/4/04
 *	- cleanup
 *	- added broken tile read mode
 * 18/5/04 Andrey Kiselev
 *	- better no resolution diagnostic
 * 26/5/04
 *	- reads 16 bit RGBA
 * 28/7/04
 *	- arrg, 16bit RGB was broken, thanks haida
 * 26/11/04
 *	- add a TIFF warning handler, stops occasional libMagick exceptions
 * 9/3/05
 *	- load 32-bit float LAB
 * 8/4/05
 *	- onebit read no longer reads one byte too many on multiple of 8 wide
 *	  images
 * 22/6/05
 *	- 16 bit LAB read was broken
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* Turn on debugging output.
#define DEBUG
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG_H*/
#include <vips/intl.h>

#ifndef HAVE_TIFF

#include <vips/vips.h>

int
im_tiff2vips( const char *tiffile, IMAGE *im )
{
	im_error( "im_tiff2vips", _( "TIFF support disabled" ) );
	return( -1 );
}

int
im_istiffpyramid( const char *name )
{
	return( 0 );
}

int
im_tiff2vips_header( const char *tiffile, IMAGE *im )
{
	im_error( "im_tiff2vips", _( "TIFF support disabled" ) );
	return( -1 );
}

#else /*HAVE_TIFF*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <vips/vips.h>
#include <vips/thread.h>

#include <tiffio.h>

#ifdef WITH_DMALLOC
#include <dmalloc.h>
#endif /*WITH_DMALLOC*/

/* Lower and upper bounds for tile cache size. Choose an exact number based on
 * tile size.
 */
#define IM_MAX_TILE_CACHE (250)
#define IM_MIN_TILE_CACHE (5)

/* Scanline-type process function.
 */
typedef PEL *(*scanline_process_fn)( PEL *q, PEL *p, int n, void *user );

/* A tile in our cache.
 */
typedef struct {
	REGION *tile;			/* REGION with private mem for data */
	int time;			/* Time of last use for flush */
} CachedTile;

/* Stuff we track during a read.
 */
typedef struct {
	/* Parameters.
	 */
	char *filename;
	IMAGE *out;

	/* From filename.
	 */
	int page;
	int broken;

	/* The TIFF we read.
	 */
	TIFF *tiff;

	/* Process for this image type.
	 */
	scanline_process_fn sfn;
	void *client;

	/* Geometry.
	 */
	int twidth, theight;		/* Tile size */
	int csize;			/* Cache size */
	int tls;			/* Tile line size */

	/* Only need one of these, since we mutex around TIFF*().
	 */
	GMutex *tlock;			/* Lock for TIFF*() calls */
	tdata_t *buf;			/* TIFF read buffer */
} ReadTiff;

/* Per-thread stuff - give each one a separate cache for simplicity.
 */
typedef struct {
	/* Parent.
	 */
	ReadTiff *rtiff;

	/* Cache. 
	 */
	CachedTile tcache[IM_MAX_TILE_CACHE];
	int time;			/* Increment and mark tiles on use */
} SeqInfo;

/* Reading a YCbCr image ... parameters we use for conversion.
 */
typedef struct {
	/* Input and output.
	 */
	TIFF *tif;			/* From here */
	IMAGE *im;			/* To here */

	/* RGB <-> YCbCr conversion.
	 */
	float LumaRed, LumaGreen, LumaBlue;

	/* RGB -> LAB conversion.
	 */
	void *table;
} YCbCrParams;

/* Handle TIFF errors here. 
 */
static void 
thandler_error( char *module, char *fmt, va_list ap )
{
	im_verror( module, fmt, ap );
}

static void 
thandler_warning( char *module, char *fmt, va_list ap )
{
	char buf[256];

	im_vsnprintf( buf, 256, fmt, ap );
	im_warn( module, "%s", buf );
}

/* Test for field exists.
 */
static int
tfexists( TIFF *tif, ttag_t tag )
{
	uint32 a, b;

	if( TIFFGetField( tif, tag, &a, &b ) ) 
		return( 1 );
	else 
		return( 0 );
}

/* Test a uint16 field. Field must be defined and equal to the value.
 */
static int
tfequals( TIFF *tif, ttag_t tag, uint16 val )
{
	uint16 fld;

	if( !TIFFGetFieldDefaulted( tif, tag, &fld ) ) {
		im_error( "im_tiff2vips", 
			_( "required field %d missing" ), tag );
		return( 0 );
	}
	if( fld != val ) {
		im_error( "im_tiff2vips", _( "required field %d=%d, not %d" ),
			tag, fld, val );
		return( 0 );
	}

	/* All ok.
	 */
	return( 1 );
}

/* Get a uint32 field.
 */
static int
tfget32( TIFF *tif, ttag_t tag, int *out )
{
	uint32 fld;

	if( !TIFFGetFieldDefaulted( tif, tag, &fld ) ) {
		im_error( "im_tiff2vips", 
			_( "required field %d missing" ), tag );
		return( 0 );
	}

	/* All ok.
	 */
	*out = fld;
	return( 1 );
}

/* Get a uint16 field.
 */
static int
tfget16( TIFF *tif, ttag_t tag, int *out )
{
	uint16 fld;

	if( !TIFFGetFieldDefaulted( tif, tag, &fld ) ) {
		im_error( "im_tiff2vips", 
			_( "required field %d missing" ), tag );
		return( 0 );
	}

	/* All ok.
	 */
	*out = fld;
	return( 1 );
}

/* Make a SeqInfo.
 */
static void *
seq_build( IMAGE *out, ReadTiff *rtiff )
{
	int i;
	SeqInfo *seq = IM_NEW( rtiff->out, SeqInfo );

	if( !seq )
		return( NULL );
	seq->rtiff = rtiff;

	for( i = 0; i < rtiff->csize; i++ )
		seq->tcache[i].tile = NULL;

	seq->time = 0;

	return( (void *) seq );
}

/* Touch a cache entry.
 */
static void
cache_touch( SeqInfo *seq, int i )
{
	assert( i >=0 && i < seq->rtiff->csize );

	seq->tcache[i].time = seq->time++;
}

/* Find an empty cache slot and fill it, or find the LRU and reuse that.
 * Return <0 for error.
 */
static int
cache_find( SeqInfo *seq )
{
	ReadTiff *rtiff = seq->rtiff;
	int i;
	int min, minpos;

	/* Look for empty.
	 */
	for( i = 0; i < rtiff->csize; i++ )
		if( !seq->tcache[i].tile ) {
			if( !(seq->tcache[i].tile = 
				im_region_create( rtiff->out )) )
				return( -1 );
			return( i );
		}

	/* Find LRU.
	 */
	min = seq->tcache[0].time;
	minpos = 0;
	for( i = 1; i < rtiff->csize; i++ )
		if( seq->tcache[i].time < min ) {
			min = seq->tcache[i].time;
			minpos = i;
		}

	return( minpos );
}

/* Run over a TIFF tile with the scanline process function, filling a
 * region.
 */
static void
vips_tile( ReadTiff *rtiff, tdata_t *td, REGION *out )
{
	Rect *r = &out->valid;
	PEL *p = (PEL *) td;
	int y;

	if( rtiff->broken ) {
		/* Old and broken code, here so we can still read old tiled
		 * tiffs.
		 */
		for( y = r->top; y < IM_RECT_BOTTOM( r ); y++ ) {
			PEL *q = (PEL *) IM_REGION_ADDR( out, r->left, y );

			p = rtiff->sfn( q, p, r->width, rtiff->client );
		}
	}
	else {
		for( y = r->top; y < IM_RECT_BOTTOM( r ); y++ ) {
			PEL *q = (PEL *) IM_REGION_ADDR( out, r->left, y );

			(void) rtiff->sfn( q, p, r->width, rtiff->client );
			p += rtiff->tls;
		}
	}
}

/* Find a tile ... if we don't have it in cache, get it. Return the cache
 * index, or <0 for error.
 */
static int
cache_fill( SeqInfo *seq, int ix, int iy )
{
	ReadTiff *rtiff = seq->rtiff;
	int i;
	REGION *reg;
	Rect area;

	/* Do we have this tile? Should really use a hashtable I guess.
	 */
	for( i = 0; i < rtiff->csize; i++ ) {
		reg = seq->tcache[i].tile;

		if( reg && im_rect_includespoint( &reg->valid, ix, iy ) ) {
			/* In cache .. note use.
			 */
			cache_touch( seq, i );

			return( i );
		}
	}

	/* Not there ... pick a slot to use.
	 */
	if( (i = cache_find( seq )) < 0 )
		return( -1 );
	cache_touch( seq, i );
	reg = seq->tcache[i].tile;
	area.left = ix; 
	area.top = iy;
	area.width = rtiff->twidth; 
	area.height = rtiff->theight;
	if( im_region_local( reg, &area ) )
		return( -1 );

	/* Read that tile.
	 */
	g_mutex_lock( rtiff->tlock );
	if( TIFFReadTile( rtiff->tiff, rtiff->buf, ix, iy, 0, 0 ) < 0 ) {
		g_mutex_unlock( rtiff->tlock );
		return( -1 );
	}
	vips_tile( rtiff, rtiff->buf, reg );
	g_mutex_unlock( rtiff->tlock );

	return( i );
}

/* Copy useful parts of from to to.
 */
static void
copy_region( REGION *from, REGION *to )
{
	Rect ovlap;
	int y;

	/* Find common pixels.
	 */
	im_rect_intersectrect( &from->valid, &to->valid, &ovlap );
	assert( !im_rect_isempty( &ovlap ) );

	/* Loop down common area, copying.
	 */
	for( y = ovlap.top; y < IM_RECT_BOTTOM( &ovlap ); y++ ) {
		PEL *p = (PEL *) IM_REGION_ADDR( from, ovlap.left, y );
		PEL *q = (PEL *) IM_REGION_ADDR( to, ovlap.left, y );

		memcpy( q, p, IM_IMAGE_SIZEOF_PEL( from->im ) * ovlap.width );
	}
}

/* Loop over the output region, filling with data from cache.
 */
static int
fill_region( REGION *out, SeqInfo *seq )
{
	ReadTiff *rtiff = seq->rtiff;
	int x, y;
	Rect *r = &out->valid;

	/* Find top left of tiles we need.
	 */
	int xs = (r->left / rtiff->twidth) * rtiff->twidth;
	int ys = (r->top / rtiff->theight) * rtiff->theight;

	for( y = ys; y < IM_RECT_BOTTOM( r ); y += rtiff->theight )
		for( x = xs; x < IM_RECT_RIGHT( r ); x += rtiff->twidth ) {
			int i;

			if( (i = cache_fill( seq, x, y )) < 0 )
				return( -1 );
			copy_region( seq->tcache[i].tile, out );
		}

	return( 0 );
}

/* Free a cache entry.
 */
static void
cache_free( SeqInfo *seq, int i )
{
	if( seq->tcache[i].tile ) {
		(void) im_region_free( seq->tcache[i].tile );
		seq->tcache[i].tile = NULL;
	}
}

/* Free a SeqInfo.
 */
static int
seq_free( SeqInfo *seq )
{
	int i;

	for( i = 0; i < seq->rtiff->csize; i++ )
		cache_free( seq, i );
	
	return( 0 );
}

/* Tile-type TIFF reader core - pass in a per-tile transform. Generate into
 * the im and do it all partially.
 */
static int
read_tilewise( ReadTiff *rtiff )
{
	int tpl;

	/* Get tiling geometry.
	 */
	if( !tfget32( rtiff->tiff, TIFFTAG_TILEWIDTH, &rtiff->twidth ) ||
		!tfget32( rtiff->tiff, TIFFTAG_TILELENGTH, &rtiff->theight ) )
		return( -1 );

	/* Find upper bound for cache size ... aim for enough tiles to cover a
	 * line of a 20,000 pixel wide image, plus 50%.
	 */
	tpl = (3 * 20000 / rtiff->twidth) / 2;
	rtiff->csize = IM_CLIP( IM_MIN_TILE_CACHE, tpl, IM_MAX_TILE_CACHE );
#ifdef DEBUG
	printf( "im_tiff2vips: caching %d %dx%d tiles\n", 
		rtiff->csize, rtiff->twidth, rtiff->theight );
#endif /*DEBUG*/

	if( !(rtiff->buf = im_malloc( NULL, TIFFTileSize( rtiff->tiff ) )) )
		return( -1 );

	/* Sizeof a line of bytes in the TIFF tile.
	 */
	rtiff->tls = TIFFTileSize( rtiff->tiff ) / rtiff->theight;

	/* Make sure we can write PIO-style.
	 */
	if( im_poutcheck( rtiff->out ) )
		return( -1 );

	if( rtiff->broken )
		im_warning( "im_tiff2vips: using broken tile read\n" );

	/* Process and save as VIPS.
	 */
	if( im_demand_hint( rtiff->out, IM_SMALLTILE, NULL ) ||
		im_generate( rtiff->out, 
			seq_build, fill_region, seq_free, rtiff, NULL ) )
		return( -1 );

	return( 0 );
}

/* Scanline-type TIFF reader core - pass in a per-scanline transform.
 */
static int
read_scanlinewise( ReadTiff *rtiff )
{
	PEL *vbuf;
	tdata_t tbuf;
	int y;

	/* Make sure we can write WIO-style.
	 */
	if( im_outcheck( rtiff->out ) || im_setupout( rtiff->out ) )
		return( -1 );

	/* Make VIPS output buffer.
	 */
	if( !(vbuf = IM_ARRAY( rtiff->out, 
		IM_IMAGE_SIZEOF_LINE( rtiff->out ), PEL )) )
		return( -1 );

	/* Make TIFF line input buffer.
	 */
	if( !(tbuf = im_malloc( rtiff->out, 
		TIFFScanlineSize( rtiff->tiff ) )) ) 
		return( -1 );

	for( y = 0; y < rtiff->out->Ysize; y++ ) {
		/* Read TIFF scanline.
		 */
		if( TIFFReadScanline( rtiff->tiff, tbuf, y, 0 ) < 0 ) {
			im_error( "im_tiff2vips", _( "read error" ) );
			return( -1 );
		}

		/* Process and save as VIPS.
		 */
		(void) rtiff->sfn( vbuf, tbuf, rtiff->out->Xsize, 
			rtiff->client );
		if( im_writeline( y, rtiff->out, vbuf ) ) 
			return( -1 );
	}

	return( 0 );
}

/* Per-scanline process function for IM_CODING_LABQ.
 */
static PEL *
labpack_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;

	for( x = 0; x < n; x++ ) {
		q[0] = p[0];
		q[1] = p[1];
		q[2] = p[2];
		q[3] = 0;

		q += 4;
		p += 3;
	}

	return( p );
}

/* Read an 8-bit LAB image.
 */
static int
parse_labpack( ReadTiff *rtiff )
{
	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 3 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 8 ) )
		return( -1 );

	rtiff->out->Bands = 4; 
	rtiff->out->Bbits = 8; 
	rtiff->out->BandFmt = IM_BANDFMT_UCHAR; 
	rtiff->out->Coding = IM_CODING_LABQ; 
	rtiff->out->Type = IM_TYPE_LAB; 

	rtiff->sfn = labpack_line;
	rtiff->client = NULL;

	return( 0 );
}

/* Per-scanline process function for IM_CODING_LABQ.
 */
static PEL *
labs_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;
	unsigned short *p1 = (unsigned short *) p;
	short *q1 = (short *) q;

	for( x = 0; x < n; x++ ) {
		q1[0] = p1[0] >> 1;
		q1[1] = p1[1];
		q1[2] = p1[2];

		q1 += 3;
		p1 += 3;
	}

	return( p );
}

/* Read a 16-bit LAB image.
 */
static int
parse_labs( ReadTiff *rtiff )
{
	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 3 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 16 ) )
		return( -1 );

	rtiff->out->Bands = 3; 
	rtiff->out->Bbits = 16; 
	rtiff->out->BandFmt = IM_BANDFMT_SHORT; 
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_LABS; 

	rtiff->sfn = labs_line;
	rtiff->client = NULL;

	return( 0 );
}

/* Per-scanline process function for IM_CODING_LABQ.
 */
static PEL *
ycbcr_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;

	for( x = 0; x < n; x++ ) {
		q[0] = p[0];
		q[1] = p[1];
		q[2] = p[2];
		q[3] = 0;

		q += 4;
		p += 3;
	}

	return( p );
}

/* Read a YCbCr image.
 */
static int
parse_ycbcr( ReadTiff *rtiff )
{
	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 3 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 8 ) )
		return( -1 );

	rtiff->out->Bands = 4; 
	rtiff->out->Bbits = 8; 
	rtiff->out->BandFmt = IM_BANDFMT_UCHAR; 
	rtiff->out->Coding = IM_CODING_LABQ; 
	rtiff->out->Type = IM_TYPE_LAB; 

	rtiff->sfn = ycbcr_line;
	rtiff->client = NULL;

	return( 0 );
}

/* Per-scanline process function for 1 bit images.
 */
static PEL *
onebit_line( PEL *q, PEL *p, int n, void *flg )
{
	/* Extract PHOTOMETRIC_INTERPRETATION.
	 */
	int pm = *((int *) flg);
	int x, i, z;
	PEL bits;

	int black = (pm == PHOTOMETRIC_MINISBLACK) ? 0 : 255;
	int white = black ^ -1;

	/* (sigh) how many times have I written this?
	 */
	for( x = 0, i = 0; i < (n >> 3); i++ ) {
		bits = (PEL) p[i];

		for( z = 0; z < 8; z++, x++ ) {
			q[x] = (bits & 128) ? white : black;
			bits <<= 1;
		}
	}

	/* Do last byte in line.
	 */
	if( n & 7 ) {
		bits = p[i];
		for( z = 0; z < (n & 7); z++ ) {
			q[x + z] = (bits & 128) ? white : black;
			bits <<= 1;
		}
	}

	/* Fits exactly? Don't need to skip padding.
	 */
	if( n & 7 )
		return( p + i + 1 );
	else
		return( p + i );
}

/* Read a 1-bit TIFF image. Pass in pixel values to use for black and white.
 */
static int
parse_onebit( ReadTiff *rtiff, int pm )
{
	int *ppm;

	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 1 ) )
		return( -1 );

	rtiff->out->Bands = 1; 
	rtiff->out->Bbits = 8; 
	rtiff->out->BandFmt = IM_BANDFMT_UCHAR; 
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_B_W; 

	/* Note pm for later.
	 */
	if( !(ppm = IM_ARRAY( rtiff->out, 1, int )) )
		return( -1 );
	*ppm = pm;

	rtiff->sfn = onebit_line;
	rtiff->client = ppm;

	return( 0 );
}

/* Per-scanline process function for 8-bit greyscale images.
 */
static PEL *
greyscale8_line( PEL *q, PEL *p, int n, void *flg )
{
	/* Extract swap mask.
	 */
	PEL mask = *((PEL *) flg);
	int x;

	/* Read bytes, swapping sense if necessary.
	 */
	for( x = 0; x < n; x++ ) 
		q[x] = p[x] ^ mask;

	return( p + x );
}

/* Read a 8-bit grey-scale TIFF image. 
 */
static int
parse_greyscale8( ReadTiff *rtiff, int pm )
{
	PEL *mask;

	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 8 ) )
		return( -1 );

	/* Eor each pel with this later.
	 */
	if( !(mask = IM_ARRAY( rtiff->out, 1, PEL )) )
		return( -1 );
	*mask = (pm == PHOTOMETRIC_MINISBLACK) ? 0 : 255;

	rtiff->out->Bands = 1; 
	rtiff->out->Bbits = 8; 
	rtiff->out->BandFmt = IM_BANDFMT_UCHAR; 
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_B_W; 

	rtiff->sfn = greyscale8_line;
	rtiff->client = mask;

	return( 0 );
}

/* Per-scanline process function for 16-bit greyscale images.
 */
static PEL *
greyscale16_line( PEL *q, PEL *p, int n, void *flg )
{
	/* Extract swap mask.
	 */
	unsigned short mask = *((unsigned short *) flg);
	unsigned short *p1 = (unsigned short *) p;
	unsigned short *q1 = (unsigned short *) q;
	int x;

	/* Read bytes, swapping sense if necessary.
	 */
	for( x = 0; x < n; x++ ) 
		q1[x] = p1[x] ^ mask;

	return( (PEL *) (p1 + x) );
}

/* Read a 16-bit grey-scale TIFF image. 
 */
static int
parse_greyscale16( ReadTiff *rtiff, int pm )
{
	unsigned short *mask;

	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 16 ) )
		return( -1 );

	/* Eor each pel with this later.
	 */
	if( !(mask = IM_ARRAY( rtiff->out, 1, unsigned short )) )
		return( -1 );
	mask[0] = (pm == PHOTOMETRIC_MINISBLACK) ? 0 : 65535;

	rtiff->out->Bands = 1; 
	rtiff->out->Bbits = 16; 
	rtiff->out->BandFmt = IM_BANDFMT_USHORT; 
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_B_W; 

	rtiff->sfn = greyscale16_line;
	rtiff->client = mask;

	return( 0 );
}

/* Per-scanline process function for 32-bit floating point greyscale images.
 */
static PEL *
greyscale32f_line( PEL *q, PEL *p, int n )
{
	float *p1 = (float *) p;
	float *q1 = (float *) q;
	int x;

	for( x = 0; x < n; x++ )
		q1[x] = p1[x];

	return( (PEL *) (p1 + x) );
}

/* Read a 32-bit floating point greyscale TIFF image. What do we do about
 * MINISWHITE/MINISBLACK (pm)? Not sure ... just ignore it.
 */
static int
parse_greyscale32f( ReadTiff *rtiff, int pm )
{
	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 32 ) )
		return( -1 );

	rtiff->out->Bands = 1; 
	rtiff->out->Bbits = 32;
	rtiff->out->BandFmt = IM_BANDFMT_FLOAT;
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_B_W; 

	rtiff->sfn = (scanline_process_fn) greyscale32f_line;
	rtiff->client = NULL;

	return( 0 );
}

/* Per-scanline process function for palette images.
 */
static PEL *
palette_line( PEL *q, PEL *p, int n, void *flg )
{
	/* Extract maps.
	 */
	PEL *red = ((PEL **) flg)[0];
	PEL *green = ((PEL **) flg)[1];
	PEL *blue = ((PEL **) flg)[2];
	int x;

	/* Read bytes, generating colour.
	 */
	for( x = 0; x < n; x++ ) {
		int i = *p++;

		q[0] = red[i];
		q[1] = green[i];
		q[2] = blue[i];

		q += 3;
	}

	return( p );
}

/* Read a palette-ised TIFF image. Again, we only allow 8-bits for now.
 */
static int
parse_palette( ReadTiff *rtiff )
{
	uint16 *tred, *tgreen, *tblue;
	PEL *red, *green, *blue;
	PEL **maps;
	int i;

	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 1 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 8 ) )
		return( -1 );
	
	/* Allocate mem for VIPS colour maps.
	 */
	if( !(red = IM_ARRAY( rtiff->out, 256, PEL )) ||
		!(green = IM_ARRAY( rtiff->out, 256, PEL )) ||
		!(blue = IM_ARRAY( rtiff->out, 256, PEL )) ||
		!(maps = IM_ARRAY( rtiff->out, 3, PEL * )) )
		return( -1 );

	/* Get maps, convert to 8-bit data.
	 */
	if( !TIFFGetField( rtiff->tiff, 
		TIFFTAG_COLORMAP, &tred, &tgreen, &tblue ) ) {
		im_error( "im_tiff2vips", _( "bad colormap" ) );
		return( -1 );
	}
	for( i = 0; i < 256; i++ ) {
		red[i] = tred[i] >> 8;
		green[i] = tgreen[i] >> 8;
		blue[i] = tblue[i] >> 8;
	}
	maps[0] = red; 
	maps[1] = green; 
	maps[2] = blue;

	rtiff->out->Bands = 3; 
	rtiff->out->Bbits = 8; 
	rtiff->out->BandFmt = IM_BANDFMT_UCHAR; 
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_sRGB; 

	rtiff->sfn = palette_line;
	rtiff->client = maps;

	return( 0 );
}

/* Per-scanline process function for 8-bit RGB/RGBA.
 */
static PEL *
rgb8_line( PEL *q, PEL *p, int n, IMAGE *im )
{
	int x, b;

	for( x = 0; x < n; x++ ) {
		for( b = 0; b < im->Bands; b++ )
			q[b] = p[b];

		q += im->Bands;
		p += im->Bands;
	}

	return( p );
}

/* Read an 8-bit RGB/RGBA image.
 */
static int
parse_rgb8( ReadTiff *rtiff )
{
	int bands;

	/* Check other TIFF fields to make sure we can read this. Can have 4
	 * bands for RGBA.
	 */
	if( !tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 8 ) ||
		!tfget16( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, &bands ) )
		return( -1 );
	if( bands != 3 && bands != 4 ) {
		im_error( "im_tiff2vips", _( "3 or 4 bands RGB TIFF only" ) );
		return( -1 );
	}

	rtiff->out->Bands = bands; 
	rtiff->out->Bbits = 8; 
	rtiff->out->BandFmt = IM_BANDFMT_UCHAR; 
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_sRGB; 

	rtiff->sfn = (scanline_process_fn) rgb8_line;
	rtiff->client = rtiff->out;

	return( 0 );
}

/* Per-scanline process function for RGB/RGBA 16.
 */
static PEL *
rgb16_line( PEL *q, PEL *p, int n, IMAGE *im )
{
	int x, b;
	unsigned short *p1 = (unsigned short *) p;
	unsigned short *q1 = (unsigned short *) q;

	for( x = 0; x < n; x++ ) {
		for( b = 0; b < im->Bands; b++ )
			q1[b] = p1[b];

		q1 += im->Bands;
		p1 += im->Bands;
	}

	return( (PEL *) p1 );
}

/* Read a 16-bit RGB/RGBA image.
 */
static int
parse_rgb16( ReadTiff *rtiff )
{
	int bands;

	/* Check other TIFF fields to make sure we can read this. Can have 4
	 * bands for RGBA.
	 */
	if( !tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 16 ) ||
		!tfget16( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, &bands ) )
		return( -1 );
	if( bands != 3 && bands != 4 ) {
		im_error( "im_tiff2vips", _( "3 or 4 bands RGB TIFF only" ) );
		return( -1 );
	}

	rtiff->out->Bands = bands; 
	rtiff->out->Bbits = 16; 
	rtiff->out->BandFmt = IM_BANDFMT_USHORT; 
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_sRGB; 

	rtiff->sfn = (scanline_process_fn) rgb16_line;
	rtiff->client = rtiff->out;

	return( 0 );
}

/* Per-scanline process function for 32f.
 */
static PEL *
r32f_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;
	float *p1 = (float *) p;
	float *q1 = (float *) q;

	for( x = 0; x < n; x++ ) {
		q1[0] = p1[0];
		q1[1] = p1[1];
		q1[2] = p1[2];

		q1 += 3;
		p1 += 3;
	}

	return( (PEL *) p1 );
}

/* Read a 32-bit float image. RGB or LAB, with or without alpha.
 */
static int
parse_32f( ReadTiff *rtiff, int pm )
{
	int bands;

	if( !tfget16( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, &bands ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 32 ) )
		return( -1 );

	/* Can be 4 for images with an alpha channel.
	 */
	assert( bands == 3 || bands == 4 );

	rtiff->out->Bands = bands; 
	rtiff->out->Bbits = 32; 
	rtiff->out->BandFmt = IM_BANDFMT_FLOAT; 
	rtiff->out->Coding = IM_CODING_NONE; 

	switch( pm ) {
	case PHOTOMETRIC_CIELAB:
		rtiff->out->Type = IM_TYPE_LAB; 
		break;

	case PHOTOMETRIC_RGB:
		rtiff->out->Type = IM_TYPE_sRGB; 
		break;

	default:
		assert( 0 );
	}

	rtiff->sfn = r32f_line;
	rtiff->client = NULL;

	return( 0 );
}

/* Per-scanline process function for CMYK8.
 */
static PEL *
cmyk8_line( PEL *q, PEL *p, int n, void *dummy )
{
	int x;

	for( x = 0; x < n; x++ ) {
		q[0] = p[0];
		q[1] = p[1];
		q[2] = p[2];
		q[3] = p[3];

		q += 4;
		p += 4;
	}

	return( p );
}

/* Read a CMYK image.
 */
static int
parse_cmyk( ReadTiff *rtiff )
{
	if( !tfequals( rtiff->tiff, TIFFTAG_SAMPLESPERPIXEL, 4 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, 8 ) ||
		!tfequals( rtiff->tiff, TIFFTAG_INKSET, INKSET_CMYK ) )
		return( -1 );

	rtiff->out->Bands = 4; 
	rtiff->out->Bbits = 8; 
	rtiff->out->BandFmt = IM_BANDFMT_UCHAR; 
	rtiff->out->Coding = IM_CODING_NONE; 
	rtiff->out->Type = IM_TYPE_CMYK; 

	rtiff->sfn = cmyk8_line;
	rtiff->client = NULL;

	return( 0 );
}

/* Read resolution from a TIFF image.
 */
static int
parse_resolution( TIFF *tiff, IMAGE *im )
{
	float x, y;
	int ru;

	if( TIFFGetFieldDefaulted( tiff, TIFFTAG_XRESOLUTION, &x ) &&
		TIFFGetFieldDefaulted( tiff, TIFFTAG_YRESOLUTION, &y ) &&
		tfget16( tiff, TIFFTAG_RESOLUTIONUNIT, &ru ) ) {
		switch( ru ) {
		case RESUNIT_NONE:
			break;

		case RESUNIT_INCH:
			/* In pixels-per-inch ... convert to mm.
			 */
			x /= 10.0 * 2.54;
			y /= 10.0 * 2.54;
			break;

		case RESUNIT_CENTIMETER:
			/* In pixels-per-centimetre ... convert to mm.
			 */
			x /= 10.0;
			y /= 10.0;
			break;

		default:
			im_error( "im_tiff2vips", 
				_( "unknown resolution unit" ) );
			return( -1 );
		}
	}
	else {
		im_warning( "im_tiff2vips: no resolution information for "
			"TIFF image \"%s\" -- defaulting to 1 pixel per mm", 
			TIFFFileName( tiff ) );
		x = 1.0;
		y = 1.0;
	}

	im->Xres = x;
	im->Yres = y;

	return( 0 );
}

/* Look at PhotometricInterpretation and BitsPerPixel, and try to figure out 
 * which of the image classes this is.
 */
static int
parse_header( ReadTiff *rtiff )
{
	int pm, bps, format;

	/* Ban separate planes, too annoying.
	 */
	if( tfexists( rtiff->tiff, TIFFTAG_PLANARCONFIG ) && 
		!tfequals( rtiff->tiff, 
			TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG ) ) 
		return( -1 );

	/* Always need dimensions.
	 */
	if( !tfget32( rtiff->tiff, TIFFTAG_IMAGEWIDTH, &rtiff->out->Xsize ) ||
		!tfget32( rtiff->tiff, 
			TIFFTAG_IMAGELENGTH, &rtiff->out->Ysize ) ||
		parse_resolution( rtiff->tiff, rtiff->out ) )
		return( -1 );

	/* Try to find out which type of TIFF image it is.
	 */
	if( !tfget16( rtiff->tiff, TIFFTAG_PHOTOMETRIC, &pm ) ||
		!tfget16( rtiff->tiff, TIFFTAG_BITSPERSAMPLE, &bps ) )
		return( -1 );

	/* Can set this now.
	 */
	rtiff->out->Hist = im_strdup( NULL, 
		"TIFF image imported by im_tiff2vips()\n" );

	switch( pm ) {
	case PHOTOMETRIC_CIELAB:
		switch( bps ) {
		case 8:
			if( parse_labpack( rtiff ) )
				return( -1 );
			break;

		case 16:
			if( parse_labs( rtiff ) )
				return( -1 );
			break;

		case 32:
			if( !tfget16( rtiff->tiff, 
				TIFFTAG_SAMPLEFORMAT, &format ) ) 
				return( -1 );

			if( format == SAMPLEFORMAT_IEEEFP ) {
				if( parse_32f( rtiff, pm ) )
					return( -1 );
			}
			else {
				im_error( "im_tiff2vips", 
					_( "unsupported sample "
					"format %d for lab image" ),
					format );
				return( -1 );
			}

			break;

		default:
			im_error( "im_tiff2vips", 
				_( "unsupported depth %d for LAB image" ), 
				bps );
			return( -1 );
		}

		break;

	case PHOTOMETRIC_YCBCR:
		/* Easy decision!
		 */
		if( parse_ycbcr( rtiff ) )
			return( -1 );

		break;

	case PHOTOMETRIC_MINISWHITE:
	case PHOTOMETRIC_MINISBLACK:
		switch( bps ) {
		case 1:
			if( parse_onebit( rtiff, pm ) )
				return( -1 );

			break;

		case 8:
			if( parse_greyscale8( rtiff, pm ) )
				return( -1 );

			break;

		case 16:
			if( parse_greyscale16( rtiff, pm ) )
				return( -1 );

			break;

		case 32:
			if( !tfget16( rtiff->tiff, 
				TIFFTAG_SAMPLEFORMAT, &format ) ) 
				return( -1 );

			if( format == SAMPLEFORMAT_IEEEFP ) {
				if( parse_greyscale32f( rtiff, pm ) )
					return( -1 );
			}
			else {
				im_error( "im_tiff2vips", 
					_( "unsupported sample format "
					"%d for greyscale image" ),
					format );
				return( -1 );
			}

			break;

		default:
			im_error( "im_tiff2vips", _( "unsupported depth %d "
				"for greyscale image" ), bps );
			return( -1 );
		}

		break;

	case PHOTOMETRIC_PALETTE:
		/* Full colour pallette.
		 */
		if( parse_palette( rtiff ) )
			return( -1 );

		break;

	case PHOTOMETRIC_RGB:
		/* Plain RGB.
		 */
		switch( bps ) {
		case 8:
			if( parse_rgb8( rtiff ) )
				return( -1 );
			break;

		case 16:
			if( parse_rgb16( rtiff ) )
				return( -1 );
			break;

		case 32:
			if( !tfget16( rtiff->tiff, 
				TIFFTAG_SAMPLEFORMAT, &format ) ) 
				return( -1 );

			if( format == SAMPLEFORMAT_IEEEFP ) {
				if( parse_32f( rtiff, pm ) )
					return( -1 );
			}
			else {
				im_error( "im_tiff2vips", 
					_( "unsupported sample "
					"format %d for rgb image" ),
					format );
				return( -1 );
			}

			break;

		default:
			im_error( "im_tiff2vips", _( "unsupported depth %d "
				"for RGB image" ), bps );
			return( -1 );
		}

		break;

	case PHOTOMETRIC_SEPARATED:
		if( parse_cmyk( rtiff ) )
			return( -1 );

		break;

	default:
		im_error( "im_tiff2vips", _( "unknown photometric "
			"interpretation %d" ), pm );
		return( -1 );
	}

	return( 0 );
}

/* Free a ReadTiff.
 */
static int
readtiff_destroy( ReadTiff *rtiff )
{
	IM_FREEF( TIFFClose, rtiff->tiff );
	IM_FREE( rtiff->buf );
	IM_FREEF( g_mutex_free, rtiff->tlock );

	return( 0 );
}

/* Make a ReadTiff.
 */
static ReadTiff *
readtiff_new( const char *filename, IMAGE *out )
{
	ReadTiff *rtiff;
	char name[FILENAME_MAX];
	char mode[FILENAME_MAX];
	char *p, *q;

	if( !(rtiff = IM_NEW( out, ReadTiff )) )
		return( NULL );
	rtiff->filename = NULL;
	im_filename_split( filename, name, mode );
	rtiff->filename = im_strdup( out, name );
	rtiff->out = out;
	rtiff->page = 0;
	rtiff->broken = 0;
	rtiff->tiff = NULL;
	rtiff->sfn = NULL;
	rtiff->client = NULL;
	rtiff->twidth = 0;
	rtiff->theight = 0;
	rtiff->csize = 0;
	rtiff->tls = 0;
	rtiff->tlock = g_mutex_new();
	rtiff->buf = NULL;

	if( im_add_close_callback( out, 
		(im_callback_fn) readtiff_destroy, rtiff, NULL ) ) {
		readtiff_destroy( rtiff );
		return( NULL );
	}

	/* Parse out params.
	 */
	p = &mode[0];
	if( (q = im_getnextoption( &p )) ) {
		rtiff->page = atoi( q );

		if( rtiff->page < 0 || rtiff->page > 1000 ) {
			im_error( "im_tiff2vips", _( "bad page number %d" ),
				rtiff->page );
			return( NULL );
		}
	}
	if( (q = im_getnextoption( &p )) ) {
		if( im_isprefix( "broken", q ) )
			rtiff->broken = 1;
	}

	return( rtiff );
}

/* Pull out the nth directory from a TIFF file.
 */
static TIFF *
get_directory( const char *filename, int page )
{
	TIFF *tif;
	int i;

#ifdef BINARY_OPEN
	if( !(tif = TIFFOpen( filename, "rb" )) ) {
#else /*BINARY_OPEN*/
	if( !(tif = TIFFOpen( filename, "r" )) ) {
#endif /*BINARY_OPEN*/
		im_error( "im_tiff2vips", 
			_( "unable to open \"%s\" for input" ),
			filename );
		return( NULL );
	}

	for( i = 0; i < page; i++ ) 
		if( !TIFFReadDirectory( tif ) ) {
			/* Run out of directories.
			 */
			TIFFClose( tif );
			return( NULL );
		}

	return( tif );
}

int
im_istiffpyramid( const char *name )
{
	TIFF *tif;

	TIFFSetErrorHandler( (TIFFErrorHandler) thandler_error );
	TIFFSetWarningHandler( (TIFFErrorHandler) thandler_warning );

	if( (tif = get_directory( name, 2 )) ) {
		/* We can see page 2 ... assume it is.
		 */
		TIFFClose( tif );
		return( 1 );
	}

	return( 0 );
}

int
im_tiff2vips( const char *filename, IMAGE *im )
{
	ReadTiff *rtiff;

#ifdef DEBUG
	printf( "im_tiff2vips: libtiff version is \"%s\"\n", TIFFGetVersion() );
#endif /*DEBUG*/

	TIFFSetErrorHandler( (TIFFErrorHandler) thandler_error );
	TIFFSetWarningHandler( (TIFFErrorHandler) thandler_warning );

	if( !(rtiff = readtiff_new( filename, im )) )
		return( -1 );

	if( !(rtiff->tiff = get_directory( rtiff->filename, rtiff->page )) ) {
		im_error( "im_tiff2vips", _( "TIFF file does not "
			"contain page %d" ), rtiff->page );
		return( -1 );
	}

	if( parse_header( rtiff ) )
		return( -1 );

	if( TIFFIsTiled( rtiff->tiff ) ) {
		if( read_tilewise( rtiff ) )
			return( -1 );
	}
	else {
		if( read_scanlinewise( rtiff ) )
			return( -1 );
	}

	return( 0 );
}

/* Just parse the header.
 */
int
im_tiff2vips_header( const char *filename, IMAGE *im )
{
	ReadTiff *rtiff;

	TIFFSetErrorHandler( (TIFFErrorHandler) thandler_error );
	TIFFSetWarningHandler( (TIFFErrorHandler) thandler_warning );

	if( !(rtiff = readtiff_new( filename, im )) )
		return( -1 );

	if( !(rtiff->tiff = get_directory( rtiff->filename, rtiff->page )) ) {
		im_error( "im_tiff2vips", _( "TIFF file does not "
			"contain page %d" ), rtiff->page );
		return( -1 );
	}

	if( parse_header( rtiff ) )
		return( -1 );

	return( 0 );
}

#endif /*HAVE_TIFF*/
