/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/

#include <iostream>
#include <fstream>
#include <map>
#include <set>
#include <queue>
/* -- */
#include "common/Errors.hpp"
/* -- */
#include "parser/Line.hpp"
#include "parser/Definition.hpp"
#include "parser/ParserDefinitionPaje.hpp"
/* -- */
using namespace std;

const int ParserDefinitionPaje::_OUT_A_DEFINITION = 0;
const int ParserDefinitionPaje::_IN_A_DEFINITION = 1;

ParserDefinitionPaje::ParserDefinitionPaje() : _state(_OUT_A_DEFINITION) {
}

int ParserDefinitionPaje::definitions_number() const {
    return _definitions.size();
}

void ParserDefinitionPaje::enter_definition(const Line &line){
    
    if(_state == _IN_A_DEFINITION){
        Error::set(Error::_EXPECT_END_DEF, line.get_line_count(), Error::_WARNING);
        leave_definition(line);
    }

    string definition_name;

    if(!line.item(2, definition_name)){
        Error::set(Error::_EXPECT_NAME_DEF, line.get_line_count(), Error::_ERROR);
        return;
    }
  
    unsigned int definition_identity;
    string definition_identity_string;
    
    if(!line.item(3, definition_identity_string)){
        Error::set(Error::_EXPECT_ID_DEF, line.get_line_count(), Error::_ERROR);
        return;
    }

    if(sscanf(definition_identity_string.c_str(), "%u", &definition_identity) != 1){
        Error::set(Error::_EXPECT_ID_DEF, line.get_line_count(), Error::_ERROR);
        return;
    }

    Definition current_definition(definition_name);
    _definitions.insert(pair<unsigned int,Definition>(definition_identity, current_definition));
    _current_definition_id = definition_identity;
    
    if(line.length() > 4){
        Error::set(Error::_EXTRA_TOKEN, line.get_line_count(), Error::_WARNING);
    }
  
    _state = _IN_A_DEFINITION;
    return;
}

void ParserDefinitionPaje::leave_definition(const Line &line){
    
    if(_state != _IN_A_DEFINITION){
        Error::set(Error::_EXPECT_EVENT_DEF, line.get_line_count(), Error::_WARNING);
        return;
    }
    
    _state = _OUT_A_DEFINITION;
    if(!_definitions[_current_definition_id].check_definition()){
        Error::set(Error::_UNKNOWN_EVENT_DEF + _definitions[_current_definition_id].print_string(), line.get_line_count(), Error::_ERROR);
       _definitions.erase(_current_definition_id);
        return;
    }
}

void ParserDefinitionPaje::add_field_to_definition(std::string& first_token, const Line& line){
  
    if(_state == _OUT_A_DEFINITION){
        Error::set(Error::_EXPECT_EVENT_DEF, line.get_line_count(), Error::_ERROR);
        return;
    }

    string field_type;
    if(!line.item(2, field_type)){
        Error::set(Error::_FIELD_TYPE_MISSING, line.get_line_count(), Error::_ERROR);
        return;
    }

    if(field_type != "string" && field_type != "int" && field_type != "hex" &&
       field_type != "date" && field_type != "double" && field_type != "color"){
        Error::set(Error::_FIELD_TYPE_UNKNOWN + field_type, line.get_line_count(), Error::_ERROR);
        return;
    }

    _definitions[_current_definition_id].store(first_token, field_type);
    
    if(line.length() > 3){
        Error::set(Error::_EXTRA_TOKEN, line.get_line_count(), Error::_WARNING);
    }

}

void ParserDefinitionPaje::store_definition(const Line &line){

    string first_token;
    if(!line.item(1, first_token)){
        Error::set(Error::_EMPTY_DEF, line.get_line_count(), Error::_WARNING);
        return;
    }

    if (first_token == "EventDef"){
        enter_definition(line);
    }
    else if (first_token == "EndEventDef"){
        leave_definition(line);
        if(line.length() > 2){
            Error::set(Error::_EXTRA_TOKEN, line.get_line_count(), Error::_WARNING);
    }
    }
    else{
        add_field_to_definition(first_token,line);
    }
    
}

Definition& ParserDefinitionPaje::get_definition(unsigned int i){
    if (_definitions.count(i) != 1) {
        throw i;
    }
    return _definitions[i];
}

const map<unsigned int,Definition>* ParserDefinitionPaje::get_hash_table_of_definitions() const{
    return &_definitions;
}

void ParserDefinitionPaje::print_definitions() const {
    map<unsigned int,Definition>::const_iterator it_end = _definitions.end();

    for (map<unsigned int,Definition>::const_iterator it = _definitions.begin(); it != it_end ; ++ it){

        cout << "# " << (*it).first << endl;

        // (*it).second represents the current definition
        (*it).second.print();
    }
}
