/*
    vlad - an LDAP visualisation tool
    Copyright (C) 1999-2001 Robert Norris <rob@nauseum.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <ncurses.h>
#include <time.h>

#include "vlad.h"

static struct entry *get_next_entry(struct entry *entry) {
	struct entry *parent;

	if(!entry) return NULL;

	if(entry->subentries_head && entry->expanded)
		return entry->subentries_head;

	if(entry->next)
		return entry->next;
	
	parent = entry->parent;
	
	while(parent) {
		if(parent->next)
			return parent->next;
		parent = parent->parent;
		}

	return NULL;
	}

static struct entry *get_last_subentry(struct entry *tree) {
	struct entry *last = tree, *tail;

	while((tail = last->subentries_tail) && last->expanded)
		last = tail;

	return last;
	}

static struct entry *get_prev_entry(struct entry *entry) {
	struct entry *parent;

	if(!entry) return NULL;

	if(entry->prev)
		return get_last_subentry(entry->prev);
	
	if(!(parent = entry->parent))
		return NULL;
	
	return parent;
	}

static int MAX_Y, MAX_X;

static int redraw_tree(WINDOW *win, struct entry *top, struct entry *bottom, struct entry *sel) {
	struct entry *entry;
	int y = 0;

	wmove(win, 0, 0);
	wclrtobot(win);

	entry = top;

	while(entry && (y != MAX_Y)) {
		wmove(win, y, entry->depth);

		if(entry == sel)
			wattrset(win, A_REVERSE);
		
		if(entry->expanded)
			waddstr(win, "[-] ");
		else
			waddstr(win, "[+] ");

		waddnstr(win, entry->shortdn, MAX_X - entry->depth - 5);

		if(entry == sel) {
			wattrset(win, A_NORMAL);
			wmove(win, y, 0);
			wchgat(win, -1, A_REVERSE, COLOR_PAIR(1), NULL);
			}

		if(entry == bottom) break;

		entry = get_next_entry(entry);

		y++;
		}
	
	wrefresh(win);

	return y;
	}

static int detail_view(WINDOW *win, struct entry *entry);
static void help(int sect);

void tree_view(WINDOW *win) {
	struct entry *root, *sel, *top, *bottom, *entry;
	int y, x, i, line = 0, bottomline, err;

	getmaxyx(win, MAX_Y, MAX_X);

	if(!(root = create_root_entry_struct())) {
		endwin();
		fputs("Couldn't initialise directory tree structure.\n", stderr);
		exit(1);
		}
	
	sel = top = bottom = root;
	bottomline = 0;

	wclear(win);
	wcolor_set(win, COLOR_PAIR(1), NULL);

	wprintw(win, "[+] %s", root->shortdn);
	wmove(win, 0, 0);
	wchgat(win, -1, A_REVERSE, COLOR_PAIR(1), NULL);

	wrefresh(win);
	
	set_title("Vlad " VERSION);
	
	set_status("dn: %s", sel->shortdn);

	while(1) {
		err = 0;

		switch(wgetch(win)) {
			case 'q':
				endwin();
				exit(0);
				break;
			
			case 10:
				if(sel->expanded) {
					sel->expanded = 0;

					wmove(win, line, sel->depth + 1);
					wattrset(win, A_REVERSE);
					waddch(win, '+');
					wattrset(win, A_NORMAL);
					}

				else {

					if(!sel->subentries_timestamp) {
					// FIXME if(sel->subentries_timestamp < time(0)) {
						set_status("Expanding '%s'\n", sel->dn);
						if(get_subtree(sel, 1))
							break;
						}

					if(sel->subentries_timestamp) {
						sel->expanded = 1;

						wmove(win, line, sel->depth + 1);
						wattrset(win, A_REVERSE);
						waddch(win, '-');
						wattrset(win, A_NORMAL);

						set_status("dn: %s", sel->shortdn);
						}
					}

				if(line < (MAX_Y - 1)) {
					entry = bottom = sel;

					wmove(win, line + 1, 0);
					wclrtobot(win);
					wmove(win, line, 0);

					while((entry = get_next_entry(entry))) {
						getyx(win, y, x);
						if(y == (MAX_Y - 1)) break;

						wmove(win, y + 1, entry->depth);

						if(entry->expanded)
							waddstr(win, "[-] ");
						else
							waddstr(win, "[+] ");

						waddnstr(win, entry->shortdn, MAX_X - entry->depth - 5);

						bottom = entry;
						bottomline = y + 1;
						}
					}	

				wrefresh(win);

				break;

			case KEY_DOWN:
				if((line == (MAX_Y - 1)) && (entry = get_next_entry(bottom))) {
					top = get_next_entry(top);
					bottom = entry;

					scrollok(win, TRUE);
					wscrl(win, 1);
					scrollok(win, FALSE);

					wmove(win, line, bottom->depth);

					line--;

					if(entry->expanded)
						waddstr(win, "[-] ");
					else
						waddstr(win, "[+] ");

					waddnstr(win, bottom->shortdn, MAX_X - bottom->depth - 5);
					}

				if(sel != bottom) {
					sel = get_next_entry(sel);

					wmove(win, line, 0);
					winstr(win, buffer);
					wclrtoeol(win);
					waddstr(win, buffer);
					wmove(win, line, 0);
					wchgat(win, -1, A_NORMAL, COLOR_PAIR(0), NULL);

					line++;
					
					wmove(win, line, 0);
					winstr(win, buffer);
					wclrtoeol(win);
					waddstr(win, buffer);
					wmove(win, line, 0);
					wchgat(win, -1, A_REVERSE, COLOR_PAIR(1), NULL);

					wrefresh(win);

					set_status("dn: %s", sel->shortdn);
					}
				break;

			case KEY_UP:
				if(!line && (entry = get_prev_entry(top))) {
					top = entry;
					if(bottomline == (MAX_Y - 1))
						bottom = get_prev_entry(bottom);
					else
						bottomline++;

					scrollok(win, TRUE);
					wscrl(win, -1);
					scrollok(win, FALSE);

					wmove(win, 0, top->depth);

					line = 1;

					if(entry->expanded)
						waddstr(win, "[-] ");
					else
						waddstr(win, "[+] ");

					waddnstr(win, top->shortdn, MAX_X - top->depth - 5);
					}

				if(sel != top) {
					sel = get_prev_entry(sel);

					wmove(win, line, 0);
					winstr(win, buffer);
					wclrtoeol(win);
					waddstr(win, buffer);
					wmove(win, line, 0);
					wchgat(win, -1, A_NORMAL, COLOR_PAIR(0), NULL);

					if(!line) {
						/* scroll down */
						}
					else
						line--;
					
					wmove(win, line, 0);
					winstr(win, buffer);
					wclrtoeol(win);
					waddstr(win, buffer);
					wmove(win, line, 0);
					wchgat(win, -1, A_REVERSE, COLOR_PAIR(1), NULL);

					wrefresh(win);

					set_status("dn: %s", sel->shortdn);
					}
				break;

			case KEY_NPAGE:
				entry = get_next_entry(bottom);

				if(entry) {
					for(i = 1; i < (MAX_Y - 1); i++) {
						if(!(entry = get_next_entry(entry))) break;
						bottom = entry;
						}
					
					entry = top;

					for(; i > 0; i--) {
						if(!(entry = get_next_entry(entry))) break;
						top = entry;
						}

					sel = top;
					line = 0;

					bottomline = redraw_tree(win, top, bottom, sel);

					set_status("dn: %s", sel->shortdn);
					}
				break;

			case KEY_PPAGE:
				entry = get_prev_entry(top);

				if(entry) {
					for(i = 1; i < (MAX_Y - 1); i++) {
						if(!(entry = get_prev_entry(entry))) break;
						top = entry;
						}

					entry = bottom;

					for(; i > 0; i--) {
						if(!(entry = get_prev_entry(entry))) break;
						bottom = entry;
						}

					sel = top;
					line = 0;

					bottomline = redraw_tree(win, top, bottom, sel);

					set_status("dn: %s", sel->shortdn);
					}
				break;

			case ' ':
				err = detail_view(win, sel);

				redraw_tree(win, top, bottom, sel);

				if(!err) set_status("dn: %s", sel->shortdn);
				break;

			case '?':
				help(0);

				redraw_tree(win, top, bottom, sel);

				break;
			}
		}
	}

static int redraw_detail(WINDOW *win, struct entry *entry, int top, int bottom) {
	int line, y;
	struct attr *attr;

	wclear(win);

	for(line = top, y = 0; line < bottom; line++, y++) {
		if(!(attr = entry->attrs[line])) break;

		snprintf(buffer, MAX_X + 1, "%s: %s", attr->name, attr->value);

		wmove(win, y, 0);
		waddstr(win, buffer);
		}
	

	wrefresh(win);

	return y;
	}

static int detail_view(WINDOW *win, struct entry *entry) {
	int top, bottom, i;
	
	if(!entry->attrs_timestamp) {
	// FIXME if(entry->attrs_timestamp < time(0)) {
		set_status("Retrieving '%s'\n", entry->dn);
		if(get_attrs(entry, 1))
			return 1;
		}

	if(!entry->attrs_timestamp)
		return 0;
	
	top = 0;
	bottom = redraw_detail(win, entry, 0, MAX_Y);

	set_status("Attributes of '%s'", entry->shortdn);

	while(1) {
		switch(wgetch(win)) {
			case 'q':
				return 0;

			case KEY_DOWN:
				if(entry->attrs[bottom]) {
					scrollok(win, TRUE);
					wscrl(win, 1);
					scrollok(win, FALSE);

					wmove(win, MAX_Y - 1, 0);

					snprintf(buffer, MAX_X, "%s: %s", entry->attrs[bottom]->name, entry->attrs[bottom]->value);
					waddstr(win, buffer);

					top++;
					bottom++;
					}
				break;

			case KEY_UP:
				if(top) {
					top--;
					bottom--;

					scrollok(win, TRUE);
					wscrl(win, -1);
					scrollok(win, FALSE);

					wmove(win, 0, 0);

					snprintf(buffer, MAX_X, "%s: %s", entry->attrs[top]->name, entry->attrs[top]->value);
					waddstr(win, buffer);
					}
				break;

			case KEY_NPAGE:
				if(entry->attrs[bottom]) {
					for(i = 0; i < MAX_Y; i++, top++, bottom++)
						if(!entry->attrs[bottom]) break;

					redraw_detail(win, entry, top, bottom);
					}
				break;

			case KEY_PPAGE:
				if(top) {
					for(i = 0; i < MAX_Y; i++, top--, bottom--)
						if(!top) break;

					redraw_detail(win, entry, top, bottom);
					}
				break;

/*
			case '?':
				help(1);

				redraw_detail(win, entry, top, bottom);

				break;
*/
			}
		}
	}

static void help(int sect) {
	WINDOW *win;

	char *section[] = {
		"  Up/Dn/PgUp/PgDn: Scroll through tree\n"
		"            Enter: Expand/Collapse DN\n"
		"            Space: View DN attributes\n"
		"                q: Quit the program\n"
		"                ?: This help screen\n"
		};

	win = newwin(7, 40, (MAX_Y - 7) / 2, (MAX_X - 40) / 2);
	wbkgd(win, COLOR_PAIR(3));
	wattrset(win, A_BOLD);			

	wmove(win, 1, 0);
	waddstr(win, section[sect]);

	box(win, ACS_VLINE, ACS_HLINE);

	while(1) {
		switch(wgetch(win)) {
			case 'q':
				delwin(win);

				return;
			}
		}
	}
