/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <limits.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "vmelilo.h"

/*--------------------------------------------------------------------------*/

#define MAXLINELEN			4096

#ifndef CL_SIZE
# ifndef COMPAT_CL_SIZE
#  define CL_SIZE			4096
# else
#  define CL_SIZE			COMPAT_CL_SIZE
# endif
#endif

/*--------------------------------------------------------------------------*/

static int 			current_line_number; /* current line number				*/
static int 			section_line_number; /* line number at start of section	*/
static BOOTRECORD	boot_record;

/*--------------------------------------------------------------------------*/

enum keyword_id {
	KW_UNKNOWN,
	KW_DEFAULT,
	KW_MESSAGE,
	KW_BOOT,
	KW_PROMPT,
	KW_DELAY,
	KW_TIMEOUT,
	KW_CMDLINE,
	KW_PASSWORD,
	KW_RESTRICTED,
	KW_KERNEL,
	KW_LABEL,
	KW_ALIAS,
	KW_APPEND,
	KW_MEMSIZE,
	KW_RAMDISK,
	KW_SYMBOLS,
	KW_CALLDBG,
	KW_DEBUG,
	KW_MASTER_PASSWORD,
	KW_READ_ONLY,
	KW_READ_WRITE,
	KW_ROOT,
	KW_CONSOLE,
	KW_OPTIONAL
};

static struct keyword {
	char	*name;
	short	id;
} keywords[] = {
	{"default",			KW_DEFAULT			},
	{"message",			KW_MESSAGE			},
	{"boot",			KW_BOOT				},
	{"prompt",			KW_PROMPT			},
	{"delay",			KW_DELAY			},
	{"timeout",			KW_TIMEOUT			},
	{"cmdline",			KW_CMDLINE			},
	{"password",		KW_PASSWORD			},
	{"restricted",		KW_RESTRICTED		},
	{"image",			KW_KERNEL			},
	{"kernel",			KW_KERNEL			},
	{"label",			KW_LABEL			},
	{"alias",			KW_ALIAS			},
	{"append",			KW_APPEND			},
	{"memsize",			KW_MEMSIZE			},
	{"ramdisk",			KW_RAMDISK			},
	{"symbols",			KW_SYMBOLS			},
	{"callmonitor",		KW_CALLDBG			},
	{"calldbg",			KW_CALLDBG			},
	{"debug",			KW_DEBUG			},
	{"master-password",	KW_MASTER_PASSWORD	},
	{"read-only",		KW_READ_ONLY		},
	{"read-write",		KW_READ_WRITE		},
	{"root",			KW_ROOT				},
	{"console",			KW_CONSOLE			},
	{"optional",		KW_OPTIONAL			}
};

#define NUMKEYWORDS	(sizeof(keywords) / sizeof(keywords[0]))

/*--------------------------------------------------------------------------*/

static char *reserved_labels[] = {
	"?",
	"help",
	"su"
};

#define NUMRSVDLABELS	(sizeof(reserved_labels) / sizeof(reserved_labels[0]))

/*--------------------------------------------------------------------------*/

static
int
keyword
(	const char		*name
)
{	struct keyword	*kw = keywords;
	int				i	= NUMKEYWORDS;

	do	{
		if (strcasecmp(kw->name, name) == 0)
			return kw->id;
		kw++;
	} while (--i);

	return KW_UNKNOWN;
}

/*--------------------------------------------------------------------------*/
/* check for reserved boot record label names
 */

static
int
is_reserved_label
(	const char	*name
)
{	int			i = NUMRSVDLABELS;

	do	{
		if (strcasecmp(reserved_labels[--i], name) == 0)
			return TRUE;
	} while (i);

	return FALSE;
}

/*--------------------------------------------------------------------------*/

static
char *
strdupx
(	const char	*str
)
{	char	*p;

	if ((p = malloc(strlen(str) + 1)) == NULL)
	{
		error_nomemory();
	}

	strcpy(p, str);

	return p;
}

/*--------------------------------------------------------------------------*/

char *
skip_white
(	char	*p
)
{
	while (*p && (*p < '!'))
		p++;

	return p;
}

/*--------------------------------------------------------------------------*/

static
void
trim_white
(	char	*p
)
{	char	*w;

	while (1)
	{
		/* say no white space */
		w = NULL;

		/* skip non-white space */
		while (*p >= '!')
			p++;

		/* if end of string */
		if (*p == '\0')
		{
			break;
		}

		/* remember where start of white space is */
		w = p;

		/* skip past white space */
		do	{ ++p; } while (*p && *p < '!');

		/* if end of string */
		if (*p == '\0')
		{
			break;
		}
	}

	/* if we ended up in white space, strip it */
	if (w) *w = '\0';
}

/*--------------------------------------------------------------------------*/
/* copy string 'p2' to string 'p1' until 1st whitespace or delimiter
 */

static
char *
extract
(	char	*p1,
	char	*p2,
	int		len,				/* size of 'p1' buffer						*/
	int		delim
)
{
	/* leave room for NULL terminator */
	len--;

	/* skip leading white space */
	p2 = skip_white(p2);

	/* until we hit a delimiter */
	while ((*p2 >= '!') && (*p2 != delim))
	{
		if (len)
		{
			*p1++ = *p2++;
			len--;
		}
		else
		{
			p2++;
		}
	}

	/* NULL terminate */
	*p1 = '\0';

	/* skip white space */
	p2 = skip_white(p2);

	/* skip delimiter and following white space */
	if (delim && (*p2 == delim))
	{
		p2 = skip_white(p2 + 1);
	}

	/* return pointer to next argument */
	return p2;
}

/*--------------------------------------------------------------------------*/
/* replace all white space characters in string with hyphen.
 */

static
char *
despace
(	char	*str
)
{	char	*p = str;

	while (*p)
	{
		if (*p <= ' ')
		{
			*p = '-';
		}
		p++;
	}

	return str;
}

/*--------------------------------------------------------------------------*/

static
void
section_error
(	const char	*s
)
{
	die("%s:%d: %s\n", config_file, section_line_number, s);
}

/*--------------------------------------------------------------------------*/

static
void
line_error
(	const char	*s
)
{
	die("%s:%d: %s\n", config_file, current_line_number, s);
}

/*--------------------------------------------------------------------------*/

static
void
error_spurious_data
(	void
)
{
	line_error("spurious data at end of line\n");
}

/*--------------------------------------------------------------------------*/

static
void
clear_boot_record
(	void
)
{
	memset(&boot_record, 0, sizeof(BOOTRECORD));
}

/*--------------------------------------------------------------------------*/

static
void
redefined_option
(	const char	*name
)
{
	message("%s:%d: Redefinition of `%s'\n",
			 config_file, current_line_number, name);
}

/*--------------------------------------------------------------------------*/

static
void
reserved_label
(	const char	*name
)
{
	die("%s:%d: Boot record label or alias with reserved name `%s'\n",
			 config_file, current_line_number, name);
}

/*--------------------------------------------------------------------------*/

static
const u_long *
copy_long
(	u_long		data
)
{	u_long		*p;

	if ((p = malloc(sizeof(u_long))) == NULL)
	{
		error_nomemory();
	}

	*p = data;

	return p;
}

/*--------------------------------------------------------------------------*/

static
u_long
get_num
(	char			**str
)
{	char			c;
	int				base;
	int				digits;
	u_long			number;
	char			*p;

	p = skip_white(*str);
	trim_white(p);

	if (*p == '\0')
	{
		line_error("missing numeric constant\n");
	}

	/* determine number base from prefix */
	if (*p == '0')
	{
		if (p[1] == 'X' || p[1] == 'x')
		{
			p   += 2;
			base = 16;
		}
		else if (p[1] == 'B' || p[1] == 'b')
		{
			p   += 2;
			base = 2;
		}
		else
		{
			base = 8;
		}
	}
	else
	{
		base = 10;
	}

	number = 0;
	digits = 0;

	while (*p != '\0')
	{
		c = *p++;

		if (c < '!')
		{
			break;
		}

		/* look for 'M' and 'K' suffixes */
		if (digits && (*p < '!'))
		{
			if (c == 'M' || c == 'm')
			{
				number = number * 1024 * 1024;
				break;
			}

			if (c == 'K' || c == 'k')
			{
				number = number * 1024;
				break;
			}
		}

		if (isdigit(c))
		{
			c -= '0';
		}

		else if (islower(c))
		{
			c -= 'a' - 10;
		}

		else if (isupper(c))
		{
			c -= 'A' - 10;
		}

		else				/* unrecognised character	*/
		{
			goto fail;
		}

		if (c >= base)		/* not in number base		*/
		{
			goto fail;
		}

		digits++;
		number = number * base + c;
	}

	if (digits == 0)
	{
fail:
		line_error("invalid numeric constant\n");
	}

	*str = skip_white(p);
	return number;
}

/*--------------------------------------------------------------------------*/

static
int
get_bool
(	char	*text
)
{
	if (isdigit(text[0]))
	{
		return get_num(&text) != 0;
	}

	if ((strcasecmp(text, "true") == 0)
	||	(strcasecmp(text, "yes" ) == 0))
	{
		return TRUE;
	}

	if ((strcasecmp(text, "false") == 0)
	||	(strcasecmp(text, "no"   ) == 0))
	{
		return FALSE;
	}

	die("Boolean argument expected on line %d of %s\n",
		current_line_number, config_file);
}

/*--------------------------------------------------------------------------*/
/* Add a New Boot Record
 */

static
int
add_boot_record
(	const BOOTRECORD	*record
)
{	BOOTRECORD			**p;
	BOOTRECORD			*new;

	for (p = &config.records; *p; p = &(*p)->next)
	{
		if (equal_strings((*p)->label, record->label)
		||	equal_strings((*p)->alias, record->label)
		||	equal_strings((*p)->label, record->alias)
		||	equal_strings((*p)->alias, record->alias))
		{
			return FALSE;
		}
	}

	if ((new = malloc(sizeof(BOOTRECORD))) == NULL)
	{
		error_nomemory();
	}

	*new = *record;
	*p	 = new;

	return TRUE;
}

/*--------------------------------------------------------------------------*/

static
FILEDEF	**
find_file_def
(	const char	*path
)
{	FILEDEF		**p;

	for (p = &config.files; *p; p = &(*p)->next)
	{
		if (case_equal_strings((*p)->path, path))
		{
			break;
		}
	}

	return p;
}

/*--------------------------------------------------------------------------*/
/* Add a New File Definition
 */

static
const char *
add_file_def
(	const char	*path
)
{	FILEDEF		*file;
	FILEDEF		**link;
	char		*rpath;

/* The following section has been removed for now because
 * it breaks the --root option. realpath() reads the file system
 * to resolve symblic links but because we do not apply the root_path
 * configuration until later we can end up resolving the wrong symbolic
 * link.
 */
#if 0
	if ((rpath = malloc(PATH_MAX)) != NULL)
	{
		if (realpath(path, rpath) == NULL)
		{
			free(rpath);
			rpath = NULL;
		}
	}

	if (rpath == NULL)
#endif
	{
		rpath = strdupx(path);
	}

	link = find_file_def(rpath);

	if (*link == NULL)
	{
		if ((file = malloc(sizeof(FILEDEF))) == NULL)
		{
			error_nomemory();
		}
	
		file->path      = rpath;
		file->map       = NULL;
		file->refcount  = 1;
		file->next      = NULL;
		*link           = file;
	}
	else
	{
		(*link)->refcount++;
	}

	return rpath;
}

/*--------------------------------------------------------------------------*/
/* Dereference an existing file definition.
 */

void
remove_file_def
(	const char	*path
)
{	FILEDEF		*file;
	FILEDEF		**link;

	if (path && *path)
	{
		link = find_file_def(path);
	
		if ((file = *link) != NULL)
		{
			if (file->refcount)
			{
				file->refcount--;
			}
		}
	}
}

/*--------------------------------------------------------------------------*/

static
int
parse_options
(	char			*name,
	char			*text
)
{	unsigned long	flag;

	switch (keyword(name))
	{
		case KW_DEFAULT:
			if (config.options.boot_default)
			{
				redefined_option(name);
			}
			config.options.boot_default = strdupx(text);
			break;

		case KW_MESSAGE:
			if (config.options.boot_message)
			{
				redefined_option(name);
			}
			config.options.boot_message = add_file_def(text);
			break;

		case KW_BOOT:
			if (config.boot_device_name)
			{
				redefined_option(name);
			}
			config.boot_device_name = strdupx(text);
			break;

		case KW_DEBUG:
			flag = *text ? get_bool(text) : TRUE;
			if (config.options.boot_debug && (*config.options.boot_debug != flag))
			{
				redefined_option(name);
			}
			config.options.boot_debug = copy_long(flag);
			break;

		case KW_PROMPT:
			flag = *text ? get_bool(text) : TRUE;
			if (config.options.boot_prompt && (*config.options.boot_prompt != flag))
			{
				redefined_option(name);
			}
			config.options.boot_prompt = copy_long(flag);
			break;

		case KW_DELAY:
			if (config.options.boot_delay)
			{
				redefined_option(name);
			}

			config.options.boot_delay = copy_long(get_num(&text));
			if (*text)
			{
				error_spurious_data();
			}
			break;

		case KW_TIMEOUT:
			if (config.options.boot_timeout)
			{
				redefined_option(name);
			}

			config.options.boot_timeout = copy_long(get_num(&text));
			if (*text)
			{
				error_spurious_data();
			}
			break;

		case KW_CMDLINE:
			if (config.default_cmdline)
			{
				redefined_option(name);
			}

			if (strlen(text) >= CL_SIZE)
			{
				line_error("Boot argument is too long");
			}
			config.default_cmdline = strdupx(text);
			break;

		case KW_APPEND:
			if (config.default_append)
			{
				redefined_option(name);
			}

			if (strlen(text) >= CL_SIZE)
			{
				line_error("Boot argument is too long");
			}
			config.default_append = strdupx(text);
			break;

		case KW_PASSWORD:
			if (config.default_password)
			{
				redefined_option(name);
			}
			config.default_password = strdupx(text);
			break;

		case KW_MASTER_PASSWORD:
			if (config.options.boot_masterpswd)
			{
				redefined_option(name);
			}
			config.options.boot_masterpswd = strdupx(text);
			break;

		case KW_RESTRICTED:
			flag = *text ? get_bool(text) : TRUE;
			if (config.default_restricted && (*config.default_restricted != flag))
			{
				redefined_option(name);
			}
			config.default_restricted = copy_long(flag);
			break;

		case KW_KERNEL:
			if (config.default_kernel)
			{
				redefined_option(name);
			}
			config.default_kernel = strdupx(text);
			break;

		case KW_ROOT:
			if (config.default_root)
			{
				redefined_option(name);
			}

			if (strlen(text) >= CL_SIZE)
			{
				line_error("Root argument is too long");
			}
			config.default_root = strdupx(text);
			break;

		case KW_CONSOLE:
			if (config.default_console)
			{
				redefined_option(name);
			}

			if (strncmp(text, "/dev/", 5) == 0)
			{
				text += 5;
			}

			if (strlen(text) >= CL_SIZE)
			{
				line_error("Console argument is too long");
			}
			config.default_console = strdupx(text);
			break;

		case KW_READ_WRITE:
			flag = *text ? !get_bool(text) : FALSE;
			if (config.default_read_only && (*config.default_read_only != flag))
			{
				redefined_option(name);
			}
			config.default_read_only = copy_long(flag);
			break;

		case KW_READ_ONLY:
			flag = *text ? get_bool(text) : TRUE;
			if (config.default_read_only && (*config.default_read_only != flag))
			{
				redefined_option(name);
			}
			config.default_read_only = copy_long(flag);
			break;

		default:
			return FALSE;
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/

static
int
parse_boot
(	char			*name,
	char			*text
)
{	unsigned long	flag;

	switch (keyword(name))
	{
		case KW_OPTIONAL:
			flag = *text ? get_bool(text) : TRUE;
			if (boot_record.optional && (*boot_record.optional != flag))
			{
				redefined_option(name);
			}
			boot_record.optional = copy_long(flag);
			break;

		case KW_LABEL:
			if (boot_record.label)
			{
				redefined_option(name);
			}
			boot_record.label = strdupx(despace(text));
			if (is_reserved_label(boot_record.label))
			{
				reserved_label(boot_record.label);
			}
			break;

		case KW_ALIAS:
			if (boot_record.alias)
			{
				redefined_option(name);
			}
			boot_record.alias = strdupx(despace(text));
			if (is_reserved_label(boot_record.alias))
			{
				reserved_label(boot_record.alias);
			}
			break;

		case KW_PASSWORD:
			if (boot_record.password)
			{
				redefined_option(name);
			}
			boot_record.password = strdupx(text);
			break;

		case KW_RESTRICTED:
			flag = *text ? get_bool(text) : TRUE;
			if (boot_record.restricted && (*boot_record.restricted != flag))
			{
				redefined_option(name);
			}
			boot_record.restricted = copy_long(flag);
			break;

		case KW_CMDLINE:
			if (boot_record.cmdline)
			{
				redefined_option(name);
			}

			if (strlen(text) >= CL_SIZE)
			{
				line_error("Boot argument is too long");
			}
			boot_record.cmdline = strdupx(text);
			break;

		case KW_APPEND:
			if (boot_record.append)
			{
				redefined_option(name);
			}

			if (strlen(text) >= CL_SIZE)
			{
				line_error("Boot argument is too long");
			}
			boot_record.append = strdupx(text);
			break;

		case KW_MEMSIZE:
			if (boot_record.memsize)
			{
				redefined_option(name);
			}

			boot_record.memsize = copy_long(get_num(&text));
			if (*text)
			{
				error_spurious_data();
			}
			break;

		case KW_RAMDISK:
			if (boot_record.ramdisk)
			{
				redefined_option(name);
			}
			boot_record.ramdisk = add_file_def(text);
			break;

		case KW_SYMBOLS:
			if (boot_record.symtab)
			{
				redefined_option(name);
			}
			boot_record.symtab = add_file_def(text);
			break;

		case KW_KERNEL:
			if (boot_record.kernel)
			{
				redefined_option(name);
			}
			boot_record.kernel = add_file_def(text);
			break;

		case KW_CALLDBG:
			flag = *text ? get_bool(text) : TRUE;
			if (boot_record.callmonitor && (*boot_record.callmonitor != flag))
			{
				redefined_option(name);
			}
			boot_record.callmonitor = copy_long(flag);
			break;

		case KW_ROOT:
			if (boot_record.root)
			{
				redefined_option(name);
			}

			if (strlen(text) >= CL_SIZE)
			{
				line_error("Root argument is too long");
			}
			boot_record.root = strdupx(text);
			break;

		case KW_CONSOLE:
			if (boot_record.console)
			{
				redefined_option(name);
			}

			if (strncmp(text, "/dev/", 5) == 0)
			{
				text += 5;
			}

			if (strlen(text) >= CL_SIZE)
			{
				line_error("Console argument is too long");
			}
			boot_record.console = strdupx(text);
			break;

		case KW_READ_WRITE:
			flag = *text ? !get_bool(text) : FALSE;
			if (boot_record.read_only && (*boot_record.read_only != flag))
			{
				redefined_option(name);
			}
			boot_record.read_only = copy_long(flag);
			break;

		case KW_READ_ONLY:
			flag = *text ? get_bool(text) : TRUE;
			if (boot_record.read_only && (*boot_record.read_only != flag))
			{
				redefined_option(name);
			}
			boot_record.read_only = copy_long(flag);
			break;

		default:
			return FALSE;
	}

	return TRUE;
}

/*--------------------------------------------------------------------------*/

static
void
boot_wrapup
(	void
)
{
	if (boot_record.label == NULL)
	{
		section_error("No label specified for boot record.");
	}

	/* add file definition for default kernel */
	if (boot_record.kernel == NULL)
	{
		if (config.default_kernel)
		{
			boot_record.kernel = add_file_def(config.default_kernel);
		}
	}

	if (!add_boot_record(&boot_record))
	{
		section_error("Duplicate boot record label or alias.");
	}

	clear_boot_record();
}

/*--------------------------------------------------------------------------*/

struct parser {
	char	*section;					/* section name						*/
	void	(*wrapup)(void);			/* called at end of section			*/
	int		(*parse)(char *, char *);	/* parser function					*/
} parsers[] = {
	{"options",	NULL,		 parse_options	},
	{"boot",	boot_wrapup, parse_boot		}
};

#define NUMPARSERS (sizeof(parsers) / sizeof(parsers[0]))

/*--------------------------------------------------------------------------*/

static struct parser *section_parse = &parsers[0];

/*--------------------------------------------------------------------------*/

static
void
wrapup_section
(	void
)
{
	/* if there is a section handler */
	if (section_parse)
	{
		/* and a section wrap up handler */
		if (section_parse->wrapup)
		{
			/* call it */
			(*section_parse->wrapup)();
		}
	}
}

/*--------------------------------------------------------------------------*/

static
void
parse_line
(	char			*text
)
{	char			name[128];
	int				j;

	/* if start of a new section */
	if (*text == '[')
	{
		/* wrap up previous section */
		wrapup_section();

		/* remember where section starts */
		section_line_number = current_line_number;

		/* get section name */
		extract(name, text + 1, sizeof(name), ']');

		/* find a parser for section */
		for (j = 0; j < NUMPARSERS; j++)
		{
			if (strcasecmp(parsers[j].section, name) == 0)
			{
				section_parse = &parsers[j];
				return;
			}
		}

		die("unrecognised section `%s' in %s line %d\n",
					name, config_file, current_line_number);
	}

	/* get options name */
	text = extract(name, text, sizeof(name), '=');

	if (!(*section_parse->parse)(name, text))
	{
		die("unrecognised option `%s' in %s line %d\n",
					name, config_file, current_line_number);
	}
}

/*--------------------------------------------------------------------------*/
/* Read the Configuration File
 */

void
read_config_file
(	void
)
{	FILE			*fp;
	char			*p;
	char			*linebf;
	char			*inptbf;
	int				linelen;
	int				eof;

	if (strcmp(config_file, "-") == 0)
	{
		fp          = stdin;
		config_file = "standard input";
	}
	else
	{
		/* open input file */
		if ((fp = fopen(config_file, "r")) == NULL)
		{
			error_open(config_file);
		}
	}

	if (f_verbose)
	{
		message("Reading configuration file `%s'\n", config_file);
	}

	/* allocate line buffers */
	linebf = malloc(MAXLINELEN);
	inptbf = malloc(MAXLINELEN);
	if (linebf == NULL || inptbf == NULL)
	{
		error_nomemory();
	} 

	/* reset current configuration file line number */
	current_line_number = 0;
	section_line_number = 1;

	/* read line of input until end of file */
	eof = FALSE;
	do	{
		linelen   = 0;
		linebf[0] = '\0';
		do	{
			/* read a line of input */
			if (fgets(inptbf, MAXLINELEN, fp) == NULL)
			{
				eof = TRUE;
				break;
			}

			/* update current line number */
			current_line_number++;

			/* check for long lines */
			if (strlen(inptbf) == (MAXLINELEN - 1))
			{
				goto too_long;
			}

			/* skip leading whitespace */
			p = skip_white(inptbf);

			/* remove trailing white space */
			trim_white(p);

			/* ignore comment and blank lines */
			if (*p == '#' || *p == '\0')
			{
				continue;
			}

			/* check input length */
			if ((linelen + strlen(p)) >= (MAXLINELEN - 1))
			{
			too_long:
				die("line %d too long in file \"%s\"\n",
						current_line_number, config_file);
			}

			/* if appending to existing line */
			if (linebf[0])
			{
				/* separate with a single space */
				strcat(linebf, " ");
			}
	
			/* append new input */
			strcat(linebf, p);

			/* get new line length */
			linelen = strlen(linebf);

			/* if line doesn't end with a backslash */
			if (linelen == 0 || linebf[linelen - 1] != '\\')
			{
				/* line has been gathered */
				break;
			}

			/* remove backslash */
			linebf[linelen - 1] = '\0';

			/* remove white space before trailing backslash */
			trim_white(linebf);

			/* get trimmed line length */
			linelen = strlen(linebf);

		/* stop if end of input */
		} while (feof);

		if (linebf[0])
		{
			/* parse it */
			parse_line(linebf);
		}

	} while (!eof);

	/* wrap up final section */
	wrapup_section();

	/* check for errors */
	if (ferror(fp))
	{
		error_read(config_file);
	}

	/* finish with buffers */
	free(inptbf);
	free(linebf);

	/* finish with file */
	if (fp != stdin)
	{
		fclose(fp);
	}

	if (!config.options.boot_default)
	{
		config.options.boot_default = config.records->label;
	}
}

/*-----------------------------< end of file >------------------------------*/
