#include "global.h"

#include "net.h"
#include "netdefs.h"
#include "netImpl.h"
#include "stat.h"	/* sum_pkts_lost */
#include "channel.h"	/* Channel */
#include "rtpsess.h"

#include "world.h"	/* getCurrentWorldName */


static u_int16 nbsources;
static Session *sessions_list; /* RTP sessions */
static char rtpname[NAME_LEN];		/* rtp name */
static char rtpemail[EMAIL_LEN];	/* rtp email */
static char toolname[TOOL_LEN];		/* tool name */

#define RTP_CHECK_LIST \
  { Session *tmp = sessions_list; \
    while (tmp != NULL) { \
      if (tmp->next == tmp) fatal("RtpSession list invalid at %s %d", __FILE__, __LINE__); \
      tmp = tmp->next; \
    } \
   }

/*
 * RTP header
 */

static
u_int16 RtpCreateSeq()
{
  u_int16 seq;
  time_t t;

  srand((u_int32) time(&t));
  seq = rand();
  return seq;
}

static
u_int32 RtpCreateSsrc(int value)
{
  u_int32 ssrc;
  time_t t;

  srand((u_int32) time(&t));
  ssrc = (u_int32) random32(value);
  return ssrc;
}

void RtpCreateHeader(Session *psession, rtp_hdr_t *rtp_hdr, u_int32 ssrc)
{
  struct timeval ts;

  assert(psession);
  rtp_hdr->version = RTP_VERSION;
  rtp_hdr->p = 0;
  rtp_hdr->x = 0;
  rtp_hdr->cc = 0;
  rtp_hdr->m = 0;
  rtp_hdr->pt = PAYLOAD_TYPE;
  rtp_hdr->seq = htons(++psession->rtp_seq);
  gettimeofday(&ts, NULL);
  rtp_hdr->ts = htonl(ts.tv_sec*1000 + ts.tv_usec/1000);
  rtp_hdr->ssrc = htonl(ssrc);

#ifdef DEBUG
  trace(DBG_RTP, "RTP: %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
        hdr[0], hdr[1], hdr[2], hdr[3], hdr[4], hdr[5], hdr[6], hdr[7], hdr[8], hdr[9], hdr[10], hdr[11]);
#endif
}

void RtpInitSeq(source *s, u_int16 seq)
{
  s->base_seq = seq - 1;
  s->max_seq = seq;
  s->bad_seq = RTP_SEQ_MOD + 1;
  s->cycles = 0;
  s->received = 0;
}

int RtpUpdateSeq(source *s, u_int16 seq)
{
  u_int16 udelta = seq - s->max_seq;
  const int MAX_DROPOUT = 3000;
  const int MAX_MISORDER = 100;
  const int MIN_SEQUENTIAL = 2;

  /*
   * Source is not valid until MIN_SEQUENTIAL packets with
   * sequential sequence numbers have been received.
   */
  if (s->probation) {
    /* packet is in sequence */
    if (seq == s->max_seq + 1) {
      s->probation--;
      s->max_seq = seq;
      if (s->probation == 0) {
	RtpInitSeq(s, seq);
	s->received++;
	return 1;
      }
    } else {
      s->probation = MIN_SEQUENTIAL - 1;
      s->max_seq = seq;
    }
    return 0;
  } else if (udelta < MAX_DROPOUT) {
    /* in order, with permissible gap */
    if (seq < s->max_seq) {
      /*
       * Sequence number wrapped - count another 64K cycle.
       */
      s->cycles += RTP_SEQ_MOD;
    }
    s->max_seq = seq;
  } else if (udelta <= RTP_SEQ_MOD - MAX_MISORDER) {
    /* the sequence number made a very large jump */
    if (seq == s->bad_seq) {
      /*
       * Two sequential packets -- assume that the other side
       * restarted without telling us so just re-sync
       * (i.e., pretend this was the first packet).
       */
      RtpInitSeq(s, seq);
    }
    else {
      s->bad_seq = (seq + 1) & (RTP_SEQ_MOD-1);
      return 0;
    }
  } else {
    /* duplicate or reordered packet */
  }
  s->received++;
  return 1;
}


/*
 * Handling sources
 */

u_int16 RtpGetSourcesNumber(void)
{
  return nbsources;
}

static
source_info * RtpCreateSource(Session *psession, u_int32 ssrc)
{
  source_info *psi;

  if ((psi = (source_info *) calloc(1, sizeof(source_info))) == NULL)
    return((source_info *) NULL);	/* fatal */

  psi->ssrc = ssrc;
  psi->next = psession->sinfo;	/* P. Bellot */
  psession->nbsources++;
  nbsources++;
  trace(DBG_RTP, "RtpCreateSource: nbsources++=%d", nbsources);
  return psi;
}

/* find source by its ssrc else create source info */
source_info * RtpGetSource(u_int32 ssrc)
{
  Channel *pchan;
  source_info *psi, *psilast;
  int i = 0;
  
  pchan = getCurrentChannel(); /* maybe indeterminated */
  assert(pchan->session->sinfo);
  for (psi = psilast = pchan->session->sinfo;
       psi && (i < nbsources); psi = psi->next, i++) {
    if (psi->ssrc == ssrc)
      return psi;
    psilast = psi;
  }
  /* source not yet registered */
  if (psi == NULL) {
    /* no source found, create it */
    pchan->session->sinfo = RtpCreateSource(pchan->session, ssrc);
    trace(DBG_RTP, "RtpGetSource: create sinfo=%x", pchan->session->sinfo);
    return pchan->session->sinfo;
  }
  else {
    trace(DBG_FORCE, "RtpGetSource: last->ssrc=%x ssrc=%x", psilast->ssrc, ssrc);
    psilast->next = RtpCreateSource(pchan->session, ssrc);
    return psilast->next;
  }
}

void RtpDeleteSource(Session *psession, u_int32 ssrc)
{
  source_info *psi, *psilast;
  u_int8 found = 0;
  int i = 0;
  
  for (psilast = psi = psession->sinfo;
       !found && psi && (i < nbsources);
       psi = psi->next, i++) {
    if (psi->ssrc == ssrc) {
      found = 1;
      break;
    }
    psilast = psi;
    /* if (psi->ssrc == 0) */	/* BUG */
    /*  break; */		/* break the infinite loop */
  }
  if (found) {
    if (psilast == NULL) { /* no source found */
      psession->sinfo = NULL;
    }
    else {
#if 0
     // M.S : this is a NOOP if psi == psession->sinfo
     // Bad things happen after that
      psilast->next = psi->next;
#else
      if (psi == psession->sinfo) {
        psession->sinfo = psi->next;
      } else {
        psilast->next = psi->next;
      }
#endif
    }
    sum_pkts_lost += psi->lost;
    psession->nbsources--;
    free(psi);
  }
  nbsources--;
  trace(DBG_RTP, "nbsources--=%d", nbsources);
}

u_int16 getNumberOfMembers(void)
{
  u_int16 members = 0;
  Session *psess = sessions_list;
  source_info *psource;

  RTP_CHECK_LIST 
  for (psess = sessions_list; psess && psess->type; psess = psess->next) {
    for (psource = psess->sinfo; psource ; psource = psource->next) {
      members++;
    }
  }
  return members; 
}


/*
 * Handling SDES
 */

const char * RtcpName(void)
{
  FILE *fp;
  char *p, buf[BUFSIZ];
  static char name[NAME_LEN];

  p = getenv("HOME");
  if (p == NULL || *p == '\0')
    return ((char *) NULL);
  sprintf(buf, "%s/.RTPdefaults", p);
  if ((fp = fopen(buf, "r"))) {
    while (fgets(buf, sizeof(buf), fp)) {
      buf[strlen(buf) -1] = '\0';
      if (strncmp(buf, "*rtpName:", 9) == 0) {
        p = strchr(buf, ':');
        p += 2;
        strcpy(name, p);
        fclose(fp);
        return name;
      }
    }
    fclose(fp);
  }
  return ((char *) NULL);
}

const char * RtcpEmail(void)
{
  struct passwd *pwd;
  struct hostent *ph;
  char host[MAXHOSTNAMELEN], hostname[MAXHOSTNAMELEN];

  gethostname(hostname, MAXHOSTNAMELEN);
  ph = (struct hostent *) my_gethostbyname(hostname);
  memset(host, 0, MAXHOSTNAMELEN);
  strncpy(host, ph->h_name, MAXHOSTNAMELEN);
  my_free_netdb(ph);
  if ((pwd = getpwuid(getuid())) != NULL)
    sprintf(rtpemail, "%s@%s", pwd->pw_name, host);
  else
    sprintf(rtpemail, "unknown@%s", host);
  return rtpemail;
}

const char * AvatarName(void)
{
  const char *p;

  if ((p = RtcpName()) == NULL)
    p = RtcpEmail();
  strcpy(rtpname, p);
  return p;
}

char * ToolName(void)
{
  static char tool[TOOL_LEN];

  sprintf(toolname, "VREng-%s", VERSION);
  strcpy(tool, toolname);
  return tool;
}

char * getSdesItemBySsrc(u_int8 type, u_int32 ssrc)
{
  Session *psess = sessions_list;
  source_info *psource;
  sdes_item *sitem;

  RTP_CHECK_LIST 
  for (psess = sessions_list; psess && psess->type; psess = psess->next) {
    for (psource = psess->sinfo; psource ; psource = psource->next) {
      if (psource == psource->next) fatal("Infinite loop in getSdesItemBySsrc");
      if (psource->ssrc == ssrc) {
        for (sitem = &(psource->sdes); sitem ; sitem = sitem->next) {
          if (sitem->type == type) {
            return (char *) sitem->str;
          }
        }
      }
    }
  }
  return NULL;
}

char * getRtcpNameBySsrc(u_int32 ssrc)
{
  return getSdesItemBySsrc(RTCP_SDES_NAME, ssrc);
}

char * getRtcpEmailBySsrc(u_int32 ssrc)
{
  return getSdesItemBySsrc(RTCP_SDES_EMAIL, ssrc);
}

char * getRtcpToolBySsrc(u_int32 ssrc)
{
  return getSdesItemBySsrc(RTCP_SDES_TOOL, ssrc);
}

static
void RtpCreateSDES(Session *psession)
{
  sdes_item *scname, *sname, *semail, *stool, *sloc;

  trace(DBG_RTP, "CreateSDES psession: %x", psession);

  AvatarName();	/* fill rtpname */
  RtcpEmail();	/* fill rtpemail */
  ToolName();	/* fill toolname */

  trace(DBG_RTP, "RtpCreateSDES: rtpname=%s, rtpemail=%s", rtpname, rtpemail);

  if ((scname = (sdes_item *) calloc(1, sizeof(sdes_item))) == NULL)
    return;
  psession->sdes = scname;
  scname->type = RTCP_SDES_CNAME;
  scname->len = strlen(rtpemail);
  scname->str = (u_int8 *) rtpemail;
  
  if ((sname = (sdes_item *) calloc(1, sizeof(sdes_item))) == NULL)
    return;
  scname->next = sname;
  sname->type = RTCP_SDES_NAME;
  sname->len = strlen(rtpname);
  sname->str = (u_int8 *) rtpname;
  
  if ((semail = (sdes_item *) calloc(1, sizeof(sdes_item))) == NULL)
    return;
  sname->next = semail;
  semail->type = RTCP_SDES_EMAIL;
  semail->len = strlen(rtpemail);
  semail->str = (u_int8 *) rtpemail;
  
  if ((sloc = (sdes_item *) calloc(1, sizeof(sdes_item))) == NULL)
    return;
  semail->next = sloc;
  sloc->type = RTCP_SDES_LOC;
  if (getCurrentWorldName()) {
    sloc->len = strlen(getCurrentWorldName());
    sloc->str = (u_int8 *) getCurrentWorldName();
  }
  else {
    sloc->len = strlen("manager");
    sloc->str = (u_int8 *) "manager";
  }
  
  if ((stool = (sdes_item *) calloc(1, sizeof(sdes_item))) == NULL)
    return;
  sloc->next = stool;
  stool->type = RTCP_SDES_TOOL;
  stool->len = strlen(toolname);
  stool->str = (u_int8 *) toolname;

  stool->next = NULL;
  psession->nbsdes = 5;
}

void RtpRefreshSDES(Session *psession)
{
  sdes_item *scname, *sname, *semail, *stool, *sloc;

  trace(DBG_RTP, "SDES: psession=%x", psession);
  if (psession == NULL)
    return;
  if ((scname = psession->sdes) == NULL)
    return;
  scname->type = RTCP_SDES_CNAME;
  scname->len = strlen(rtpemail);
  scname->str = (u_int8 *) rtpemail;
  trace(DBG_RTP, "scname: %x %s", scname, scname->str);

  if ((sname = scname->next) == NULL)
    return;
  sname->type = RTCP_SDES_NAME;
  sname->len = strlen(rtpname);
  sname->str = (u_int8 *) rtpname;
  trace(DBG_RTP, "sname: %x %s", sname, sname->str);

  if ((semail = sname->next) == NULL)
    return;
  semail->type = RTCP_SDES_EMAIL;
  semail->len = strlen(rtpemail);
  semail->str = (u_int8 *) rtpemail;

  if ((sloc = semail->next) == NULL)
    return;
  sloc->type = RTCP_SDES_LOC;
  if (getCurrentWorldName()) {
    sloc->len = strlen(getCurrentWorldName());
    sloc->str = (u_int8 *) getCurrentWorldName();
  }
  else {
    sloc->len = strlen("manager");
    sloc->str = (u_int8 *) "manager";
  }
  trace(DBG_RTP, "sloc: %x %s", sloc, sloc->str);

  if ((stool = sloc->next) == NULL)
    return;
  stool->type = RTCP_SDES_TOOL;
  stool->len = strlen(toolname);
  stool->str = (u_int8 *) toolname;
  trace(DBG_RTP, "stool: %x %s", stool, stool->str);
}

static
void RtpFreeSDES(Session *psession)
{
  sdes_item *sitem, *tmp;

  if (psession == NULL)
    return;
  for (sitem = psession->sdes; sitem != NULL; ) {
    tmp = sitem;
    sitem = sitem->next;
    free(tmp);
    psession->nbsdes--;
  }
  psession->sdes = NULL;
}


/*
 * Handling Sessions
 */

void RtpClearSessionsList(void)
{
  sessions_list = NULL;
}

/* Alloc a Session */
Session * RtpAllocSession(void)
{
  Session *psession;

  RTP_CHECK_LIST 
  if ((psession = (Session *) calloc(1, sizeof(Session))) == NULL)
    return ((Session *) NULL);
  if (sessions_list == (Session *) NULL) {
    sessions_list = psession;
    psession->next = NULL;
  }
  else {
    psession->next = sessions_list;
    sessions_list = psession;
  }
  RTP_CHECK_LIST 
  trace(DBG_RTP, "RtpAllocSession: psession=%x", psession);
  return psession;
}

/* initialize variables for a session already allocated, return local ssrc */
u_int32 RtpCreateSession(Channel *pchannel,
                         u_int32 group, u_int16 rtp_port, u_int8 ttl)
{
  u_int32 ssrc;
  source_info *psi;

  trace(DBG_RTP, "RtpCreateSession");
  if ((pchannel->session = RtpAllocSession()) == NULL)
    return 0;
  pchannel->session->group = htonl(group);
  pchannel->session->rtp_port = htons(rtp_port);
  pchannel->session->rtcp_port = htons(rtp_port + 1);
  trace(DBG_RTP, "RtpCreateSession: rtp_port=%x rtcp_port=%x", pchannel->session->rtp_port, pchannel->session->rtcp_port);
  pchannel->session->ttl = ttl;

  /* seq number initialization */
  pchannel->session->rtp_seq = RtpCreateSeq();

  /* SSRC number initialization */
  ssrc = htonl(RtpCreateSsrc(pchannel->session->rtp_seq));
  pchannel->session->rtp_hdr.ssrc = ssrc;
  pchannel->session->sr.ssrc = ssrc;
  trace(DBG_RTP, "RtpCreateSession: ssrc=%x", ssrc);

  /* alloc source_info */
  psi = RtpCreateSource(pchannel->session, ssrc);
  pchannel->session->sinfo = psi;

  RtpInitSeq(&(psi->s), pchannel->session->rtp_seq);

  RtpCreateSDES(pchannel->session);

  return ssrc;
}

void RtpDeleteSession(u_int32 ssrc)
{
  Session *psession;
  source_info *psi;

  RTP_CHECK_LIST 
  for (psession = sessions_list; psession; psession = psession->next) {
    for (psi = psession->sinfo; psi; psi = psi->next) {
      if (psi->ssrc == ssrc) {
        trace(DBG_RTP, "RtpDeleteSession: ssrc=%x found", ssrc);
        RtpDeleteSource(psession, ssrc);
        return;
      }
    }
  }
  RTP_CHECK_LIST 
  trace(DBG_FORCE, "RtpDeleteSession: ssrc=%x not found", ssrc);
}

/*
 * Free a Session - strange: never called
 */
void RtpFreeSession(Session *psession)
{
  Session *pse;

  if (psession == NULL) {
    trace(DBG_FORCE, "RtpFreeSession: session already free");
    return;
  }
  RTP_CHECK_LIST 
  for (pse = sessions_list; pse ; pse = pse->next) {
    if (pse == psession) {
      sessions_list = pse->next;
      pse->next = NULL;
      if (pse == sessions_list)
        sessions_list = NULL;
      free(pse);
  RTP_CHECK_LIST 
      return;
    }
  }
  RTP_CHECK_LIST 
  sessions_list = NULL;
}

void RtpCloseSession(Session *psession, u_int32 ssrc)
{
  trace(DBG_RTP, "RtpCloseSession");
  //DAX dumpSession(psession);
  RtpFreeSDES(psession);
  RtpDeleteSource(psession, ssrc);
}

#ifdef OLD_SSRC
void updateSSRC(u_int32 ssrc)
{
    NetObject *poh;

    /* ssrc = ntohl(rtp_hdr->ssrc); */
    if ((poh = getObjectBySSRC(ssrc)) == 0) {
      warning("getObjectBySSRC: object not found\n");
    }
    else {
      if (poh->s == NULL) {
        warning("recvPacket: not yet sourced");
        poh->s = (source *) calloc(1, sizeof(source));
        /* RtpInitSeq(poh->s, session.rtp_seq); */
      }
      /* (poh->s->received)++; */
    }
}
#endif /* OLD_SSRC */

static
void dumpSource(source_info *psource)
{
  sdes_item *sitem;
  int i;

  //DAX if (psource->next && psource->next->next == psource)
  //DAX   return;	/* BUG! avoid loop */
  fprintf(stderr, "ssrc=%x\n", psource->ssrc);
  /* fprintf(stderr, "lost=%d\n", psource->lost); */
  if (&psource->sdes == NULL)
    return;	/* BUG! segfault */
  for (i = 0, sitem = &(psource->sdes); sitem ; sitem = sitem->next, i++) {
    if (sitem->type > RTCP_SDES_END && sitem->type <= RTCP_SDES_SOURCE && sitem->len >0 && sitem->len < 128 && sitem->str)
      fprintf(stderr, "  sdes[%d]=%s\n", i, sitem->str);
  }
  fflush(stderr);
}

void dumpSession(Session *psess)
{
  sdes_item *sitem;
  int i;

  fprintf(stderr, "group/port/ttl=%x/%x/%x\n",
                   psess->group, psess->rtp_port, psess->ttl);
  for (i = 0, sitem = psess->sdes; sitem ; sitem = sitem->next, i++) {
    if (sitem->type > RTCP_SDES_END && sitem->type <= RTCP_SDES_SOURCE && sitem->len > 0 && sitem->len < 128 && sitem->str)
      fprintf(stderr, "  sdes[%d]=%s\n", i, sitem->str);
  }
  fflush(stderr);
}

void dumpSources(void)
{
  Session *psess = sessions_list;
  source_info *psource;

  RTP_CHECK_LIST 
  for (psess = sessions_list; psess && psess->type ; psess = psess->next) {
    for (psource = psess->sinfo; psource ; psource = psource->next) {
      if (psource->ssrc == 0)
        return;
      dumpSource(psource);
    }
  }
}

void dumpSessions(void)
{
  Session *psess = sessions_list;

  RTP_CHECK_LIST 
  for (psess = sessions_list; psess ; psess = psess->next) {
    dumpSession(psess);
  }
}
