#include "global.h"
#include "images.h"

#ifndef VRENGD

typedef struct {
  void *th;
  ReadImageFunc read_func;
  int ch;
} ImgReader;

typedef struct _ImageRec {
  u_int16 imagic;
  u_int16 type;
  u_int16 dim;
  u_int16 xsize, ysize, zsize;
  u_int32 min, max;
  u_int32 wasteBytes;
  char name[80];
  u_int32 colorMap;
  FILE *file;
  u_int8 *tmp, *tmpR, *tmpG, *tmpB;
  u_int32 rleEnd;
  u_int32 *rowStart;
  u_int32 *rowSize;
} SgiInfo;


static
void bwtorgba(u_int8 *b, u_int8 *l, int n)
{
  while (n--) {
    l[0] = *b; l[1] = *b; l[2] = *b; l[3] = 0xff;
    l += 4; b++;
  }
}

static
void latorgba(u_int8 *b, u_int8 *a, u_int8 *l, int n)
{
  while (n--) {
    l[0] = *b; l[1] = *b; l[2] = *b; l[3] = *a;
    l += 4; b++; a++;
  }
}

static
void rgbtorgba(u_int8 *r, u_int8 *g, u_int8 *b, u_int8 *l, int n)
{
  while (n--) {
    l[0] = r[0]; l[1] = g[0]; l[2] = b[0]; l[3] = 0xff;
    l += 4; r++; g++; b++;
  }
}

static
void rgbatorgba(u_int8 *r, u_int8 *g, u_int8 *b, u_int8 *a, u_int8 *l, int n)
{
  while (n--) {
    l[0] = r[0]; l[1] = g[0]; l[2] = b[0]; l[3] = a[0];
    l += 4; r++; g++; b++; a++;
  }
}

static
void ConvertShort(u_int16 *array, long length)
{
  u_int32 b1, b2;
  u_int8 *ptr;

  ptr = (u_int8 *) array;
  while (length--) {
    b1 = *ptr++;
    b2 = *ptr++;
    *array++ = (b1 << 8) | (b2);
  }
}

static
void ConvertLong(u_int32 *array, long length)
{
  u_int32 b1, b2, b3, b4;
  u_int8 *ptr;

  ptr = (u_int8 *) array;
  while (length--) {
    b1 = *ptr++;
    b2 = *ptr++;
    b3 = *ptr++;
    b4 = *ptr++;
    *array++ = (b1 << 24) | (b2 << 16) | (b3 << 8) | (b4);
  }
}

static
void ImageGetRow(SgiInfo *img, u_int8 *buf, int y, int z)
{
  u_int8 *iPtr, *oPtr, pixel;
  int count;

  if ((img->type & 0xFF00) == 0x0100) {	/* storage = RLE (run length encoding) */
    fseek(img->file, (u_int32) img->rowStart[y+z*img->ysize], SEEK_SET);
    fread(img->tmp, 1, (u_int32)img->rowSize[y+z*img->ysize], img->file);

    iPtr = img->tmp;
    oPtr = buf;
    for (;;) {
      pixel = *iPtr++;
      count = (int)(pixel & 0x7F);
      if (!count)
	return;
      if (pixel & 0x80) {
	while (count--) *oPtr++ = *iPtr++;
      } else {
	pixel = *iPtr++;
	while (count--) *oPtr++ = pixel;
      }
    }
  } else {	/* storage = VERBATIM */
    fseek(img->file, 512+(y*img->xsize)+(z*img->xsize*img->ysize), SEEK_SET);
    fread(buf, 1, img->xsize, img->file);
  }
}

Image *
loadSGI(void *th, ReadImageFunc read_func, DisplayImageFunc display_func)
{
  ImgReader s1, *s = &s1;
  SgiInfo *img;
  Image *image;
  union {
    int word;
    char byte[4];
  } endian;
  int littlendian;
  int x, y, len;
  //OLD u_int32 *base, *lptr;
  u_int32 *base, *lptr;
  u_int8 *rbuf = NULL, *gbuf = NULL, *bbuf = NULL, *abuf = NULL;
  char filename[32];
  char buf[512];

  s->read_func = read_func;
  s->th = th;
  endian.word = 1;
  if (endian.byte[0] == 1)
    littlendian = 1;
  else
    littlendian = 0;

  if ((img = (SgiInfo *) malloc(sizeof(SgiInfo))) == NULL)
    return NULL;
  sprintf(filename, "/tmp/sgi%d", (int) getpid());
  if ((img->file = fopen(filename, "wb")) == NULL) {
    perror("open create /tmp/sgi");
    return NULL;
  }
  while ((len = read_func(th, buf, sizeof(buf))) > 0) {
    fwrite(buf, 1, len, img->file);
  }
  fclose(img->file);
  if ((img->file = fopen(filename, "rb")) == NULL) {
    perror("open read /tmp/sgi");
    return NULL;
  }
  unlink(filename);
  fread(img, 1, 12, img->file); // header

  if (littlendian)
    ConvertShort(&img->imagic, 6);
  img->tmp =  (u_int8 *) malloc(img->xsize*256);
  img->tmpR = (u_int8 *) malloc(img->xsize*256);
  img->tmpG = (u_int8 *) malloc(img->xsize*256);
  img->tmpB = (u_int8 *) malloc(img->xsize*256);
  if (!img->tmp || !img->tmpR || !img->tmpG || !img->tmpB)
    return NULL;
  if ((img->type & 0xFF00) == 0x0100) {	// storage = RLE
    x = img->ysize * img->zsize * sizeof(u_int32);
    img->rowStart = (u_int32 *) malloc(x);
    img->rowSize = (u_int32 *) malloc(x);
    if (!img->rowStart || !img->rowSize)
      return NULL;
    img->rleEnd = 512 + (2 * x);
    fseek(img->file, 512, SEEK_SET);
    fread(img->rowStart, 1, x, img->file);
    fread(img->rowSize, 1, x, img->file);
    if (littlendian) {
      ConvertLong(img->rowStart, x/(int)sizeof(u_int32));
      ConvertLong((u_int32 *)img->rowSize, x/(int)sizeof(u_int32));
    }
  }

  trace(DBG_FORCE, "sgi: magic=%x type=%x xsize=%d ysize=%d zsize=%d",
                    img->imagic, img->type, img->xsize, img->ysize, img->zsize);

  /* we alloc the data */
  if ((image = newImage(img->xsize, img->ysize, img->zsize, IMAGE_FIX)) == NULL)
    return NULL;

  trace(DBG_FORCE, "image: xsize=%d ysize=%d format=%d",
                    image->xsize, image->ysize, image->format);
  //ORIG base = (u_int32 *) image->pixmap;
  base = (u_int32 *) image->pixmap;
  if (base == NULL)
    return NULL;
  switch (img->zsize) {
  case 1:
    if ((rbuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    break;
  case 2:
    if ((rbuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    if ((abuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    break;
  case 3:
    if ((rbuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    if ((gbuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    if ((bbuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    break;
  case 4:
    if ((rbuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    if ((gbuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    if ((bbuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    if ((abuf = (u_int8 *) malloc(img->xsize*sizeof(u_int8))) == NULL)
      return NULL;
    break;
  }

  /* we read the data */
  lptr = base;
  for (y=0; y < img->ysize; y++) {
    if (img->zsize >= 4) {		// RGBA
      ImageGetRow(img, rbuf,y,0);
      ImageGetRow(img, gbuf,y,1);
      ImageGetRow(img, bbuf,y,2);
      ImageGetRow(img, abuf,y,3);
      rgbatorgba(rbuf, gbuf, bbuf, abuf, (u_int8 *)lptr, img->xsize);
      lptr += img->xsize;
    } else if (img->zsize == 3) {	// RGB
      ImageGetRow(img, rbuf,y,0);
      ImageGetRow(img, gbuf,y,1);
      ImageGetRow(img, bbuf,y,2);
      rgbtorgba(rbuf, gbuf, bbuf, (u_int8 *)lptr, img->xsize);
      lptr += img->xsize;
    } else if (img->zsize == 2) {	// LA
      ImageGetRow(img, rbuf,y,0);
      ImageGetRow(img, abuf,y,1);
      latorgba(rbuf, abuf, (u_int8 *)lptr, img->xsize);
      lptr += img->xsize;
    } else {				// BW
      ImageGetRow(img, rbuf,y,0);
      bwtorgba(rbuf, (u_int8 *)lptr, img->xsize);
      lptr += img->xsize;
    }
  }
  fclose(img->file);
  free(img->tmp);
  free(img->tmpR);
  free(img->tmpG);
  free(img->tmpB);
  free(img);
  free(rbuf);
  if (gbuf) free(gbuf);
  if (bbuf) free(bbuf);
  if (abuf) free(abuf);

  display_func(th, image);
  return image;
}

#endif /* !VRENGD */
