#ifndef MAPEDITOR_H
#define MAPEDITOR_H

#include <qcanvas.h>
#include <qlineedit.h>
#include <qcursor.h>
#include <qfiledialog.h>
#include <qevent.h>
#include <qprocess.h>
#include <qbuffer.h>

#include "mapcenterobj.h"
#include "file.h"
#include "misc.h"
#include "showtextdialog.h"

class QNetworkOperation;
class QUrlOperator;

class MapEditor : public QCanvasView , public xmlObj {
    Q_OBJECT

public:
    MapEditor(QWidget* parent=0, bool interactive=false, const char* name=0, WFlags f=0);
	~MapEditor();
    QColor color();
    QColor backgroundColor();
    MapCenterObj* getMapCenter();
	QCanvas* getCanvas();
	void adjustCanvasSize();// adjust canvas size to map and scrollview
	bool isRepositionBlocked(); // block while load or undo
	
private:
	QString getName(LinkableMapObj*);	// Get e.g. heading or filename
	void makeTmpDirs();		// create temporary directories
    QString saveToDir(const QString&,const QString &,bool, const QPoint &,LinkableMapObj*);
    void saveState(const QString &);	// save actual state to backup
    void saveState(LinkableMapObj *, const QString &);
    void saveState(const QString &, const QString &, const QString &);
    void saveState(const QString &, LinkableMapObj *, const QString &);
    void saveState(const SaveMode&, const QString &, LinkableMapObj *, const QString &, LinkableMapObj *, const QString &);
    void parseAtom(const QString &);	

    void addFloatImage(const QPixmap &img);

  private slots:
    void finishedLineEdit();
    void fetchImage(const QString &img);
    void imageDataFetched(const QByteArray &, QNetworkOperation *);
    void imageDataFinished(QNetworkOperation *);

public:
	void toggleHistoryWindow();
	bool isDefault();		// false, if map was changed once
    bool isUnsaved();		// save necessary
    bool hasChanged();		// undo possible
	void setChanged();		// called from NoteEditor via LMO
	void closeMap();
	void setFilePath (QString);	
	void setFilePath (QString,QString);	
	QString getFilePath ();	// Full path e.g. "/home/tux/map.xml"
	QString getFileName ();	// e.g. "map.xml"
	QString getMapName ();	// e.g. "map"
	QString getDestPath (); // e.g. "/home/tux/map.vym"
    ErrorCode load (QString, LoadMode );	// newmap, import/replace selection
public:
    int save(const SaveMode &);	// Save map 
	void setZipped(bool);		// save map zipped
	bool saveZipped();			// 1 if file will be saved zipped
    void print();				// print canvas
private:
    QPixmap getPixmap();
	void setHideTmpMode (HideTmpMode);	// temporary hide stuff
	HideTmpMode getHideTmpMode();		// temporary hide stuff
public:
	void setExportMode (bool);			// temporary hide stuff during export
    void exportImage (QString fn);		// export as PNG	
    void exportImage (QString fn, int);	// export in given format
    void exportOOPresentation(const QString &,const QString &);
    void exportXML(const QString&);		// export to directory
    void clear();		// clear map
    void copy();		// copy branch to clipboard
    void redo();		// redo last action
    void undo();		// undo last action
private:	
    void undoXML(const QString &, const QString &);
    void pasteNoSave();		// paste clipboard to branch
    void cutNoSave();	// cut to clipboard
public:	
    void paste();		// paste clipboard to branch and backup
    void cut();			// cut to clipboard and backup
    void move(const int &,const int&);
    void moveBranchUp();
    void moveBranchDown();
    void editHeading();					// Start editing heading 
private:
	void setHeading(const QString &);	// Just set the heading for selection
	void setURL(const QString &);		// Just set the URL for selection
	void setVymLink(const QString &);	// Set vymLink for selection
public:	
    void addNewBranch(int);			// pos allows to add above/below selection
    void addNewBranchHere();		// insert and make selection its
    void deleteSelection();
	LinkableMapObj* getSelection();	// returns selection
	void unselect();				// before changing current noteedit
	void reselect();				// after  changing current noteedit
	bool select(const QString &);	// Select 
private:	
	void select(LinkableMapObj*);	
	void selectNextBranch();		// Increment number of branch
	void selectPrevBranch();		// Decrement number of branch
public:	
    void selectUpperBranch();
    void selectLowerBranch();
    void selectLeftBranch();
    void selectRightBranch();
    void selectFirstBranch();
    void selectLastBranch();
    void setColor(QColor);
    void selectBackgroundColor();
    void setBackgroundColor(QColor);
    QColor pickColor();
    void colorItem();
    void colorBranch();
	void toggleStandardFlag(QString);
	virtual void setViewCenter();			// needed for zooming
    BranchObj* findText(QString,bool);		// Find object
    void findReset();						// Reset Find 
	void editURL();							// edit the URL
	QString getURL();						// returns URL of selection or ""
	void editHeading2URL();					// copy heading to URL
	void editBugzilla2URL();				// create URL to Bugzilla
	void editFATE2URL();					// create URL to FATE
	void editVymLink();						// edit link to another map
	void deleteVymLink();					// delete link to another map
	QString getVymLink();					// return path to map
	void toggleHideExport();				// toggle the export flag
	void removeBranchKeepChilds();			// remove but keep childs
	void removeChilds();					// remove childs
	void editMapInfo();						// dialog to enter author, ...
	void updateActions();					// update e.g. format buttons
	void updateNoteFlag();					// when TextEditor changes
	void setLinkStyle (LinkStyle);			// Set style of link
	LinkStyle getLinkStyle ();				// requested in LMO
	void setLinkColor(QColor);				// default color of links
	void setLinkColorHint();				// color of links
	void setLinkColorHint(LinkColorHint);	// color of links
	LinkColorHint getLinkColorHint();
	QColor getDefLinkColor();
	void setDefXLinkColor(QColor);
	QColor getDefXLinkColor();
	void setDefXLinkWidth (int);
	int getDefXLinkWidth();
	void toggleLinkColorHint();				// after changing linkStyles
    void selectLinkColor();
    void toggleScroll();
    void unScrollAll();
	void loadFloatImage ();
	void saveFloatImage (int);
	void setFrame(const FrameType &);
	void setIncludeImagesVer(bool);
	void setIncludeImagesHor(bool);
	void setHideLinkUnselected (bool);
	bool getHideLinkUnselected ();
private:	
    void importDir(BranchObj *,QDir);
public:	
    void importDir();
	void followXLink (int);
	void editXLink (int);
    void testFunction();				// FIXME just testing

protected:
	void ensureSelectionVisible();		
	virtual void updateViewCenter();	// needed for zooming
	virtual void contentsContextMenuEvent ( QContextMenuEvent *e );
    virtual void contentsMousePressEvent(QMouseEvent*);
    virtual void contentsMouseReleaseEvent(QMouseEvent*);
    virtual void contentsMouseDoubleClickEvent(QMouseEvent*);
    virtual void contentsMouseMoveEvent(QMouseEvent*);

    virtual void contentsDragEnterEvent(QDragEnterEvent *event);
    virtual void contentsDropEvent(QDropEvent *event);
private:
    QCanvas* mapCanvas;
    MapCenterObj* mapCenter;

	bool adjustCanvasRequested;	// collect requests until end of user event
	BranchObj *editingBO;		// entering Text into BO
    QLineEdit* lineedit;

    QColor actColor;			// actual color
	QColor defLinkColor;		// default color for links
	QColor defXLinkColor;		// default color for xlinks
	int defXLinkWidth;			// default width for xlinks
	LinkColorHint linkcolorhint;// use heading color or own color
	LinkStyle linkstyle;		// default style for links

    QCursor handOpenCursor;		// cursor while moving canvas view
	QCursor pickColorCursor;	// cursor while picking color 
	bool pickingColor;
	bool drawingLink;			// true while creating a link
	bool copyingObj;			// true while creating a link
	XLinkObj* tmpXLink;
	
    LinkableMapObj* selection;		// select a LinkableMapObj
    LinkableMapObj* selectionLast;	// last selection 
    MapObj* movingObj;				// moving a MapObj
	MapObj* linkingObj_src;			// part of a link
    QPoint movingObj_orgPos;		// org. pos of mouse before move
    QPoint movingObj_start;			// rel. pos of mouse to absPos 
    QPoint movingCont_start;		// inital pos of moving Content or
    QPoint movingVec;				// how far has Content moved
	QPoint movingCenter;			// used when zooming
    QPrinter* printer;				// Printing

    bool mapDefault;				// Flag if map is untouched
    bool mapChanged;				// Flag if undo is possible
	bool mapUnsaved;				// Flag if map should be saved

	bool printFrame;			// Print frame around map
	bool printFooter;			// Print footer below map

	bool zipped;				// should map be zipped
static	int mapNum;				// unique number for Editor
	QString fileName;			// short name of file (for tab)
	QString filePath;			// path to file which will be saved
	QString fileDir;			// dir where file is saved
	QString destPath;			// path to .vym file (needed for vymlinks)
	QString mapName;			// fileName without ".vym"
	QString lastImageDir;		// save dir for adding images

	bool isInteractive;			// non interactive don't need tmpdirs
	QString tmpMapDir;			// tmp directory with data for undo/redo
	int undosTotal;				// total number of undos 
	int undoNum;				// current number of bakMapDir to be used 
	int undosAvail;				// how many actions can currently be undone
	bool blockReposition;		// block while load or undo
	bool blockSaveState;		// block while load or undo

	BranchObj* itFind;			// next object in find process
	bool EOFind;				// true, if search failed

	QPoint exportOffset;		// set before export, used in save
	HideTmpMode hidemode;			// true while exporting to hide some stuff
	
    void resizeEvent( QResizeEvent * );

  QUrlOperator *urlOperator;
  QDataStream *imageData;
  QBuffer *imageBuffer;

	ShowTextDialog *historyWindow;

};
#endif

