#
# WAJIG - Debian Package Management Front End
#
# Manage the files keeping track of changes
#
# Copyright (c) Graham.Williams@csiro.au
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

#------------------------------------------------------------------------
#
# Standard python modules
#
#------------------------------------------------------------------------
import os
import string
import tempfile
import socket

#------------------------------------------------------------------------
#
# APT module
#
#------------------------------------------------------------------------
import apt_pkg

#------------------------------------------------------------------------
#
# Wajig modules
#
#------------------------------------------------------------------------
import perform

#------------------------------------------------------------------------
#
# LOCATIONS
#
# Wajig can be run on several machines sharing the same home directories
# (often through NFS) so we need to have host specific status files. 
# Also make sure the directories exist.
#------------------------------------------------------------------------
init_dir = os.path.expanduser("~/.wajig/")
if not os.path.exists(init_dir): os.mkdir(init_dir)
init_dir = init_dir + socket.gethostname()
if not os.path.exists(init_dir): os.mkdir(init_dir)

#
# Temporarily, remove old files from .wajig
# After a few versions remove this code.
#
tmp_dir = os.path.expanduser("~/.wajig")
if os.path.exists(tmp_dir + "/Available"): 
    os.rename(tmp_dir + "/Available", init_dir + "/Available")
if os.path.exists(tmp_dir + "/Available.prv"): 
    os.rename(tmp_dir + "/Available.prv", init_dir + "/Available.prv")
if os.path.exists(tmp_dir + "/Installed"): 
    os.rename(tmp_dir + "/Installed", init_dir + "/Installed")

available_file = init_dir + "/Available"
previous_file  = init_dir + "/Available.prv"
installed_file = init_dir + "/Installed"

#
# Set the temporary directory to the init_dir.  
# Large files are not generally written there so should be okay.
#
tempfile.tempdir = init_dir

def get_installed_filename():
    """Obtain the name of the file containing list of installed packages.

    Arguments:

    Returns:
    string	The filename"""

    return installed_file

def get_available_filename():
    """Obtain the name of the file containing list of available packages.

    Arguments:

    Returns:
    string	The filename"""

    return available_file

def get_previous_filename():
    """Obtain the name of the file containing list of previous packages.

    Arguments:

    Returns:
    string	The filename"""

    return previous_file

#------------------------------------------------------------------------
#
# UPDATE AVAILABLE
#
#------------------------------------------------------------------------
def update_available(noreport=0):
    """Generate current list of available packages, backing up the old list.

    Arguments:
    noreport	If set then do not report on the number of packages.

    Returns:"""

    if not os.path.exists(available_file):
	f = open(available_file,"w")
	f.close()

    temporary_file = tempfile.mktemp()

    os.rename(available_file, temporary_file)

    command = "apt-cache dumpavail " +\
	      "| egrep '^(Package|Version):' " +\
	      "| tr '\n' ' '" +\
	      "| perl -p -e 's|Package: |\n|g; s|Version: ||g'" +\
 	      "| sort | tail +2 > " + available_file
    perform.execute(command, noquiet=1, root=1)

    os.rename(temporary_file, previous_file)

    available_packages = len(open(available_file).readlines())
    previous_packages = len(open(previous_file).readlines())
    diff = available_packages - previous_packages
    if noreport == 0:
        if diff < 0:
            direction = str(0-diff) + " down on"
        elif diff == 0:
            direction = "the same as"
        else:
            direction = str(diff) + " up on"
        #print "There are " + str(available_packages) + " packages now " +\
	#      "available which is " + direction + " the previous count."
        print "This is " + direction + " the previous count."

#------------------------------------------------------------------------
#
# UPDATE INSTALLED
#
#------------------------------------------------------------------------
def update_installed():
    """Generate current list of installed packages and their status.

    Arguments:

    Returns:"""

    if not os.path.exists(installed_file):
	f = open(installed_file,"w")
	f.close()

    command = "cat /var/lib/dpkg/status | " +\
	      "egrep '^(Package|Status|Version):' | " +\
              "awk '/^Package: / {pkg=$2} " +\
	      "     /^Status: / {s1=$2;s2=$3;s3=$4}" +\
	      "     /^Version: / {print pkg,$2,s1,s2,s3}' | " +\
	      "grep 'ok installed' | " +\
	      "awk '{print $1,$2}' | sort > " + installed_file

    perform.execute(command, noquiet=1)

def count_upgrades():
    f = tempfile.mktemp()
    command = "join " + previous_file + " " + available_file + " |" +\
                  "awk '$2 != $3 {print}' |" +\
                  "join - " + installed_file + " |" +\
                  "awk '$4 != $3 {print}' | wc -l | awk '{print $1}' > " + f
    perform.execute(command, noquiet=1)
    fo = open(f)
    updated = string.split(fo.read())[0];
    print "There are " + str(updated) + " new upgrades"
    if os.path.exists(f): os.remove(f)

def reset_files():    
    if os.path.exists(available_file): os.remove(available_file)
    if os.path.exists(previous_file): os.remove(previous_file)
    if os.path.exists(installed_file): os.remove(installed_file)
    update_available(noreport=1)
    update_available(noreport=1)
    update_installed()


#------------------------------------------------------------------------
#
# ENSURE INITIALISED 
#
#------------------------------------------------------------------------
def ensure_initialised(): 
    """Make sure the init_dir and files exist and if not, create them.

    Arguments:

    Returns:"""

    if not os.path.exists(available_file): reset_files()
    if not os.path.exists(installed_file): update_installed()

    

#------------------------------------------------------------------------
# 
# Dictionaries of available, installed, and previously available
#
#------------------------------------------------------------------------
available_list = {}
previous_list = {}
installed_list = {}

def load_dictionaries():
    """Read the wajig packages files to create dictionaries of available/installed pkgs

    Arguments:

    Returns:"""

    ensure_initialised()
    
    afile = open(available_file,"r").readlines()
    for i in range(0,len(afile)):
        available_list[string.split(afile[i])[0]] = string.split(afile[i])[1]

    pfile = open(previous_file,"r").readlines()
    for i in range(0,len(pfile)):
        previous_list[string.split(pfile[i])[0]] = string.split(pfile[i])[1]

    ifile = open(installed_file,"r").readlines()
    for i in range(0,len(ifile)):
        installed_list[string.split(ifile[i])[0]] = string.split(ifile[i])[1]

def get_available_list():
    """Obtain the dictionary of available packages.

    Arguments:

    Returns:
    dictionary	Available packages"""

    return available_list

def get_previous_list():
    """Obtain the dictionary of previously available packages.

    Arguments:

    Returns:
    dictionary	Previously available packages"""

    return previous_list

def get_installed_list():
    """Obtain the dictionary of installed packages.

    Arguments:

    Returns:
    dictionary	Installed packages"""

    return installed_list

def get_available_version(pkg):
    """Obtain the package's available version number.

    Arguments:

    Returns:
    string	available version"""

    return available_list[pkg]

def get_previous_version(pkg):
    """Obtain the package's previously available version number.

    Arguments:

    Returns:
    string	Previous version"""

    return previous_list[pkg]

def get_installed_version(pkg):
    """Obtain the package's installed version number.

    Arguments:

    Returns:
    string	Installed version"""

    return installed_list[pkg]

def get_new_available():
    """Obtain the packages available now but not previously.

    Arguments:

    Returns:
    list	New packages"""

    new_list = []
    for pkg in available_list.keys():
        if not previous_list.has_key(pkg):
            new_list.append(pkg)
    return new_list

def get_new_upgrades():
    """Obtain the packages upgraded since previously.

    Arguments:

    Returns:
    list	Newly upgraded packages"""

    upgraded_list = []
    apt_pkg.InitSystem(); # Not sure why!
    for pkg in installed_list.keys():
        if available_list.has_key(pkg) \
           and previous_list.has_key(pkg) \
           and apt_pkg.VersionCompare(available_list[pkg],
                                      previous_list[pkg]) > 0:
            upgraded_list.append(pkg)
    return upgraded_list

def get_to_upgrade():
    """Obtain the packages with newer versions available.

    Arguments:

    Returns:
    list	Upgradeable packages"""

    upgraded_list = []
    apt_pkg.InitSystem(); # Not sure why!
    for pkg in installed_list.keys():
        if available_list.has_key(pkg) \
           and apt_pkg.VersionCompare(available_list[pkg],
                                      installed_list[pkg]) > 0:
            upgraded_list.append(pkg)
    return upgraded_list
