#!/usr/bin/perl -w
# ldap-users.pl - Administrate users in ldap
#
# $Id: ldap-users.pl,v 1.101 2004/09/24 15:16:03 andreas Exp $

use Data::Dumper;
use Net::LDAP;
use Net::LDAP::Entry;
use Net::LDAP::Util qw( ldap_error_name ldap_error_text);
use Text::Unaccent;
use HTML::FromText;
use Time::HiRes qw(usleep);
use locale;
use POSIX;
use strict;

END {ldap_close()}

# setlocale(LC_COLLATE, "no_NO.UTF-8");    

my %g;    # this is "g" for global. but only global to this file,
          # not the whole process.

# Connect to LDAP server.
sub ldap_connect {
    my ( $server, $rootdn, $basedn, $minid, $maxid ) = @_;

    my (@err, @dbg);

    $g{minid}         = $minid;
    $g{maxid}         = $maxid;
    $g{server}        = $server;
    $g{rootdn}        = $rootdn;
    $g{basedn}        = $basedn;
    $g{ldap}          = undef;
    $g{err}           = \@err;
    $g{dbg}           = \@dbg;
    $g{allreadyusers} = undef;
    $g{modified}      = undef;

    # Take a look at bug 404 if this doesn't work
    $g{ldap} = Net::LDAP->new($server,version => '3') 
	|| die "Unable to connect to LDAP server on host \"$server\": $! ";
    my $result = $g{ldap}->start_tls( verify => 'none');
#    $g{ldap} = Net::LDAP->new( $server, version => '3',debug=>15 )
#      || die "Configuration error or LDAP server not running on $server: $! ";
#    my $result = $g{ldap}->start_tls( verify => 'require',
#				      cafile => '/var/lib/pyca/Root/cacert.pem');
    die "LDAP Error: "
      . ldap_error_name( $result->code ) . " "
      . ldap_error_text( $result->code ) . " "
      if $result->code();
}

sub ldap_check_capabilities {
    my ($needed_ref) = @_;

    my $base   = $g{basedn};
    my $filter = "cn=capabilities";
    my $result = $g{ldap}->search( base => $base, filter => $filter );
    my $entry  = $result->pop_entry;
    if ( $result->code or !$entry ) {
        error_msg( "No capabilities found:"
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    foreach my $cap ( $entry->get_value("capability") ) {
        ( $cap, my $have_version ) = split ( / /, $cap );
        if ( $needed_ref->{$cap} and $needed_ref->{$cap} > $have_version ) {
            return undef;
        }
        else {
            delete $needed_ref->{$cap} if  $needed_ref->{$cap};
        }
    }

    if ( keys %$needed_ref ) {

        # we depend on capabilities the ldap directory does not meet.
	return undef;
    }
    return "can run";
}

# Make a unique username from common name.
sub make_uid {
    my ( $firstname, $lastname, $newnames_ref ) = @_;
    $g{allreadyusers} = $newnames_ref;
    my $username = make_unique_username( $firstname, $lastname );
    return $username;
}

sub change_slicing {
    my ( $filter_var, $slice_point, $uppercase_ref ) = @_;

    my @filter_var_list;
    my $n = $slice_point;    # this is done this way because
                             # the s/// replaces the k-1 occurance,
                             # so $n needs to lag one behind.
    $slice_point++;
    my $star_cnt = $filter_var =~ tr/*/*/;
    if ( $slice_point > $star_cnt ) {
        $slice_point = undef;
        return ( [], $slice_point );
    }

    for my $char ( 'a' .. 'z', '0' .. '9', @$uppercase_ref ) {
        my $filter_var_new = $filter_var;

        $filter_var_new =~ s/(^[^*]*(?:\*.*?){$n})\*/$1$char*/x;
        push @filter_var_list, $filter_var_new;
    }

    return ( \@filter_var_list, $slice_point );
}

sub check_slice_empty {
    my ($slice_ref) = @_;
    return 1 if $#$slice_ref == -1;
    return 0;
}

sub sort_result {
    my ( $sort_array_ref, $sort_key ) = @_;

    my @result =
      sort { $a->get_value($sort_key) cmp $b->get_value($sort_key); }
      @{$sort_array_ref};
    return @result;
}

sub get_slice {
    my ( $base, $filter_fix, $filter_var, $uppercase_ref, $depth ) = @_;
    my ( @slice, $filters );

    $depth++;

    # final filter:"(&(objectClass=lisGroup)(cn="*a*b*"))"
    # filter_fix: "&(objectClass=lisGroup)"
    # variable filter: cn="*a*b*"
    my $filter = "($filter_fix($filter_var))";
    my $mesg = $g{ldap}->search( base => $base, filter => $filter );
    if ( 4 == $mesg->code ) {

        # We got a LDAP_SIZELIMIT_EXEEDED error.
        #split up slice 
        my $slice_point = 0;
        do {
            ( my $filter_var_list_ref, $slice_point ) =
              change_slicing( $filter_var, $slice_point, $uppercase_ref );
            for my $filter_var_new ( @{$filter_var_list_ref} ) {
                if ( 9 < $depth ) { last }
                ;    # savety against infinit recursion
                 # we should not go deeper then 9 since logins have just 8 chars
                 #get sub-slices
                my ( $slice_ref, $filters_applied ) = get_slice(
                    $base,          $filter_fix, $filter_var_new,
                    $uppercase_ref, $depth
                );

                # merge new slices 
                push ( @slice, @{$slice_ref} );
                $filters .= $filters_applied if $filters_applied;
            }
        } while ( check_slice_empty( \@slice ) and defined $slice_point );
        if ( 1 == $depth ) { # this happens after we are done with our recursion
                #here we get all other entries,
                #which for some reason were not covered in the other searches.
            $filter = "($filter_fix$filters($filter_var))\n";
            $mesg = $g{ldap}->search( base => $base, filter => $filter );
            my @tmp = $mesg->all_entries();
            debug_msg( "key $filter  returned " . ( $#tmp + 1 ) . " users\n" );
            push ( @slice, $mesg->all_entries() );
        }
    }
    elsif ( $mesg->code ) {    # some other error!
        error_msg( "ldap problem: $base $filter : "
            . ldap_error_name( $mesg->code ) . " "
            . ldap_error_text( $mesg->code ) . "\n" )
          if $mesg->code;
        return ( [], "" );
    }
    else {    # We got the whole slice in one piece.
        @slice   = $mesg->all_entries();
        $filters = "(!$filter_var)";

        my @tmp = $mesg->all_entries();
        debug_msg( "key $filter  returned " . ( $#tmp + 1 ) . " users\n" )
          unless ( -1 == $#tmp );
    }
    if ( 1 == $depth ) {
        $filter_var =~ m/^(.*)=/;
        my $sort_key = $1;
        @slice = sort_result( \@slice, $sort_key );
    }
    return ( \@slice, $filters );
}

# Get all users in a given domain $basedn.
sub ldap_get_users {

    my $base = "ou=People," . $g{basedn};

    # since we dont have unique cn for users, it is unsave to search 
    # the cn with a limited SIZELIMIT therefore we must limit us to the 
    # uid, which is unique. it is much more efficent, too.
    my $filter_fix = "&(objectClass=posixAccount)";
    my $filter_var = "uid=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
}

# Search for users in a given domain $basedn.
sub ldap_search_users {
    my ( $searchin, $searchfor ) = @_;

    my $base       = "ou=People," . $g{basedn};
    my $filter_fix = "&(objectClass=posixAccount)";
    my $filter_var = "$searchin=$searchfor*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
}

# Search in a given domain $basedn.
sub ldap_search {
    my ( $searchin, $searchfor ) = @_;
    my ( $ou, $oc, $attribut );
    unless ($searchin) {
        goto error_exit;
    }
    elsif ( $searchin eq "class_cn" ) {
        $oc       = "lisGroup";
        $attribut = "cn";
        $ou       = "ou=Group,";
    }
    elsif ( $searchin eq "user_login" ) {
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_cn" ) {
        $oc       = "posixAccount";
        $attribut = "cn";             #XXX cn ist nicht unique!!!
        $ou       = "ou=People,";
    }
    else {
        error_exit:
        return ( "illegal search type\n", undef );
    }
    unless ( $searchfor and $searchfor =~ /\*/ ) {
        $searchfor .= "*";    #XXX quick fix, make this an exact search instead.
    }
    my $base       = $ou . $g{basedn};
    my $filter_fix =
      "&(objectClass=$oc)(!(|(groupType=private)(cn=admin)(cn=nextID)))";
    my $filter_var = "$attribut=$searchfor";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
        return ( "OK", @{$slice_ref} );
    }
    else {
        return ( "EMPTY", undef );
    }
}

sub ldap_disable_user_login {
    my ( $user_login, $rootpw ) = @_;
    my $result = $g{ldap}->bind( $g{rootdn}, 'password' => $rootpw );
    if ( $result->code ) {
        error_msg( "Could not bind to ldap server! "
            . ldap_error_name( $result->code ) . ": "
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    my $base   = "ou=People," . $g{basedn};
    my $filter = "&(objectClass=posixAccount)(uid=$user_login)";

    $result = $g{ldap}->search( base => $base, filter => $filter );
    my $entry = $result->pop_entry;
    if ( $result->code or !$entry ) {
        debug_msg( "No such user \"$user_login\" found:"
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    if ( $config{sambasync} ){
	system( "smbpasswd -d $user_login  >/dev/null 2>&1" );
    }

    my $inactive_flag = $entry->get_value('shadowFlag');
    if ($inactive_flag) {
        debug_msg( "User $user_login is allready disabled by flag:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        $result =
          $g{ldap}->modify( $entry,
            changes => [ add => [ 'shadowFlag' => 1 ] ] );
        if ( $result->code ) {
            debug_msg( "Error deactivating user's ldap-flag! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $passwd = $entry->get_value('userPassword');
    if ( $passwd =~ /^DISABLED!/ ) {
        debug_msg( "User $user_login is allready disabled by password:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        $result =
          $g{ldap}->modify( $entry,
            changes => [ replace => [ 'userPassword' => "DISABLED!$passwd" ] ]
        );
        if ( $result->code ) {
            debug_msg( "Error deactivating user's password! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $shell = $entry->get_value('loginShell');
    if ( $shell =~ /^DISABLED!/ ) {
        debug_msg( "User $user_login is allready disabled by shell-distrortion:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        $result =
          $g{ldap}->modify( $entry,
            changes => [ replace => [ 'loginShell' => "DISABLED!$shell" ] ] );
        if ( $result->code ) {
            debug_msg( "Error deactivating user's shell! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $homedir = $entry->get_value('homeDirectory');
    if ( ( ( stat($homedir) )[2] & 07777 ) == 00000 ) {
        debug_msg(
"User $user_login is allready disabled by homedirectory permissoions:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        my $ret = chmod 0000, $homedir;
        if ( $ret != 1 ) {
            debug_msg("Error deactivating user's homedirectory! $homedir \n");
        }
    }
    $g{modified} = 1;
    return 1;
}

sub ldap_enable_user_login {
    my ( $user_login, $rootpw ) = @_;
    my $result = $g{ldap}->bind( $g{rootdn}, 'password' => $rootpw );
    if ( $result->code ) {
        error_msg( "Could not bind to ldap server! "
            . ldap_error_name( $result->code ) . ": "
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    my $base   = "ou=People," . $g{basedn};
    my $filter = "(&(objectClass=posixAccount)(uid=$user_login))";

    $result = $g{ldap}->search( base => $base, filter => $filter );

    my $entry = $result->pop_entry;
    if ( $result->code or !$entry ) {
        debug_msg( "No such user \"$user_login\" found:"
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    if ( $config{sambasync} ){
	system( "smbpasswd -e $user_login  >/dev/null 2>&1" );
    }

    my $inactive_flag = $entry->get_value('shadowFlag');
    unless ($inactive_flag) {
        debug_msg( "User $user_login is allready ldap-enabled:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        $result =
          $g{ldap}->modify( $entry,
            changes => [ delete => [ 'shadowFlag' => 1 ] ] );
        if ( $result->code ) {
            debug_msg( "Error enabling user! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $homedir = $entry->get_value('homeDirectory');
    if ( ( ( stat($homedir) )[2] & 07777 ) == 00755 ) {
        debug_msg( "User $user_login is allready enabled for the homedir:"
            . ldap_error_text( $result->code ) . "\n" );
    }
    else {
        my $ret = chmod 0755, $homedir;
        if ( $ret != 1 ) {
            debug_msg("Error enabling user's home directory $homedir!\n");
        }
    }

    my $shell = $entry->get_value('loginShell');
    unless ( $shell =~ /^DISABLED!(.*)$/ ) {
        debug_msg("User $user_login is allready enabled on shell level.");
    }
    else {
        $shell  = $1;
        $result =
          $g{ldap}->modify( $entry,
            changes => [ replace => [ 'loginShell' => $shell ] ] );
        if ( $result->code ) {
            debug_msg( "Error enabling user's shell $shell! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }

    my $passwd = $entry->get_value('userPassword');
    unless ( $passwd =~ /^DISABLED!(.*)$/ ) {
        debug_msg("User $user_login is allready enabled on shell level.");
    }
    else {
        $passwd = $1;
        $result =
          $g{ldap}->modify( $entry,
            changes => [ replace => [ 'userPassword' => $passwd ] ] );
        if ( $result->code ) {
            debug_msg( "Error enabling user's shell $shell! "
                . ldap_error_name( $result->code ) . "\n" );
        }
    }
    $g{modified} = 1;
    return 1;
}

sub ldap_search_user_disabled_logins {
    my ( $searchin, $searchfor ) = @_;
    my ( $ou, $oc, $attribut );

    my $filter_fix =
      "&(shadowFlag=1)(!(|(groupType=private)(cn=admin)(cn=nextID)))";
    
    unless ($searchin) {
        goto error_exit;
    }
    elsif ( $searchin eq "class_cn" ) {
	my @uids; 
	my($msg, @list) = ldap_search($searchin, $searchfor);
	return  ( "EMPTY", undef ) if $msg ne "OK";
	for my $class (@list){
	    push @uids, $class->get_value("memberUID");
	}
	my $filter_append;
	for my $uid ( unique( @uids ) ){
	    $filter_append .= "(uid=$uid)";      
	}
	$filter_fix .= "(|$filter_append)" if $filter_append;
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_login" ) {
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_cn" ) {
        $oc       = "posixAccount";
        $attribut = "cn";       #XXX cn ist nicht unique!!!
        $ou       = "ou=People,";
    }
    else {
        error_exit:
        return ( "illegal search type\n", undef );
    }
    unless ( $searchfor and $searchfor =~ /\*/ ) {
        $searchfor .= "*";    #XXX quick fix, make this an exact search instead.
    }
    my $base       = $ou . $g{basedn};
    my $filter_var = "$attribut=$searchfor";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
        return ( "OK", @{$slice_ref} );
    }
    else {
        return ( "EMPTY", undef );
    }
}

sub ldap_get_group_login_status{
    my ( $cn ) = @_;

    my $base       = "ou=People," . $g{basedn};
    my $filter_var = "uid=*";

    my $group = ldap_get_group($cn);
    my @uids = $group->get_value("memberUID");
    
    my $filter_append;
    for my $uid ( unique( @uids ) ){
	$filter_append .= "(uid=$uid)";      
    }

    my ($login_enabled_flag, $login_disabled_flag) = (undef, undef);
    if ($filter_append) {
	my $filter_fix = "&(!(shadowFlag=1))(|$filter_append)"; 
	my ( $slice_ref, $filters_applied ) =
	    get_slice( $base, $filter_fix, $filter_var );
	if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
	    $login_enabled_flag = "1";
	}

	$filter_fix = "&(shadowFlag=1)(|$filter_append)"; 
	( $slice_ref, $filters_applied ) =
	    get_slice( $base, $filter_fix, $filter_var );
	if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
	    $login_disabled_flag = "1";
	}
    }
    return ($login_enabled_flag, $login_disabled_flag);
}

sub ldap_search_group_disabled_logins
{    # this wont work like this, need to loop over all users in group
    my ($searchfor) = @_;
    my $base       = "ou=People," . $g{basedn};
    my $filter_fix = "&(userPassword=\!*)(gidNumber=$searchfor)";
    my $filter_var = "uid=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
}

sub ldap_search_user_enabled_logins {
    my ( $searchin, $searchfor ) = @_;
    my ( $oc, $ou, $attribut );

    my $filter_fix =
      "&(!(shadowFlag=1))(!(|(groupType=private)(cn=admin)(cn=nextID)))";

    unless ($searchin) {
        goto error_exit;
    }
    elsif ( $searchin eq "class_cn" ) {
	my @uids; 
	my($msg, @list) = ldap_search($searchin, $searchfor);
	return  ( "EMPTY", undef ) if $msg ne "OK";
	for my $class (@list){
	    push @uids, $class->get_value("memberUID");
	}
	my $filter_append;
	for my $uid ( unique( @uids ) ){
	    $filter_append .= "(uid=$uid)";      
	}
	$filter_fix .= "(|$filter_append)" if $filter_append;
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_login" ) {
        $oc       = "posixAccount";
        $attribut = "uid";
        $ou       = "ou=People,";
    }
    elsif ( $searchin eq "user_cn" ) {
        $oc       = "posixAccount";
        $attribut = "cn";       #XXX cn ist nicht unique!!!
        $ou       = "ou=People,";
    }
    else {
        error_exit:
        return ( "illegal search type\n", undef );
    }
    unless ( $searchfor and $searchfor =~ /\*/ ) {
        $searchfor .= "*";    #XXX quick fix, make this an exact search instead.
    }
    my $base       = $ou . $g{basedn};
    my $filter_var = "$attribut=$searchfor";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    if ( $slice_ref->[0] and $slice_ref->[0]->get_value('cn') ) {
        return ( "OK", @{$slice_ref} );
    }
    else {
        return ( "EMPTY", undef );
    }
}

sub ldap_search_group_enabled_logins
{    # this wont work like this, need to loop over all users in group
    my ($searchfor) = @_;
    my $base       = "ou=People," . $g{basedn};
    my $filter_fix = "&(!userPassword=!*)(gidNumber=$searchfor)";
    my $filter_var = "uid=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
}

# Get all groups in a domain $basedn.
sub ldap_get_groups {
    my ($group_type) = @_;

    my $base       = "ou=Group," . $g{basedn};
    my $filter_fix = "&(objectClass=lisGroup)";
    if ($group_type) {
        $filter_fix .= "(groupType=$group_type)";
    }
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

# Get all groups a user $uid is member of, given a domain $basedn.
sub ldap_get_membergroups {
    my ($uid) = @_;

    my $base       = "ou=Group," . $g{basedn};
    my $filter_fix = "&(objectClass=lisGroup)(|(memberUid=$uid)(member=uid=$uid,ou=People,".$g{basedn}."))";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

# this is a more general function for retrieving a 
# list of groups a user is a member of with more 
# finegrained search criteria.
sub ldap_get_member_grouplist {
    my ( $uid, $grouptype ) = @_;

    my $base       = "ou=Group," . $g{basedn};
    my $filter_fix =
      "&(objectClass=lisGroup)(memberUid=$uid)(groupType=$grouptype)";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

sub ldap_get_not_member_grouplist {
    my ( $uid, $grouptype ) = @_;

    my $base       = "ou=Group," . $g{basedn};
    my $filter_fix =
      "&(objectClass=lisGroup)(groupType=$grouptype)(!(memberUid=$uid))";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

# takes gidNumber, returns list of complete user entries
sub ldap_get_member_userlist {
    my ($gidNumber) = @_;

    my @list;
    my $base   = "ou=Group," . $g{basedn};
    my $filter = "(&(objectClass=lisGroup)(gidNumber=$gidNumber))";
    my $mesg   = $g{ldap}->search( base => $base, filter => $filter );
    my $entry  = $mesg->pop_entry;
    unless ($entry) { return undef }
    my @uids = $entry->get_value("memberUid");

    for my $uid (@uids) {
        my $user = ldap_get_user($uid);
        push @list, $user;
    }
    return @list;
}

# Exemple ageGroup structure:
# dn: cn=year2009,ou=Group,dc=skole
# cn: exampleAgeGroup
# memberUid: bernd     this is the user's uid
# memberUid: kalle       ... teachers are in several age groups
# ageGroup: mathe2     this is the group's cn
# ageGroup: physics2     ... these classes are suitable for students that age

# A Generic AgeGroup "GeAGroup" would contain both all students 
# and all classes applicable for all students, regardless their age.

# between schoolyears pupils and teachers are removed 
# from their classes and moved to their new classes:
# 1) the ageGroup attributes (reflecting the groups/classes) 
#    are moved down to the following ageGroup
# 2) the users are removed from those moved groups 
#    and added to similar groups, which names index is increased by one,
#    if those exist.
# (this is done in step 1) those groups are then added to the ageGroup group
#
# one could search for missing groups in a step early in the 
# process and offer to create those groups/abort.

# get all users who are not yet members in a group
sub ldap_get_not_member_userlist {
    my ($gidNumber) = @_;

    # we need to get all users who are part of the 
    # ageGroups but not of the gid. 
    # first get all users in $gidNumber 
    my @list;
    my $base   = "ou=Group," . $g{basedn};
    my $filter = "(&(objectClass=lisGroup)(gidNumber=$gidNumber))";
    my $mesg   = $g{ldap}->search( base => $base, filter => $filter );
    my $entry  = $mesg->pop_entry;
    unless ($entry) { return undef }
    my @black_list = $entry->get_value("memberUid");
    my $cn         = $entry->get_value("cn");

=head1
    # find the age groups this group is part of.
    $base = "ou=Group," . $g{basedn};
    my $filter_fix = "&(groupType=age_group)(ageGroup=$cn)";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    if ( -1 == $#$slice_ref ) {

        # this group is not part of an ageGroup, 
        # use the worst case (generic ageGroup) instead.
        my $filter = "(&(objectClass=lisGroup)(cn=genagegp))";
        my $mesg   = $g{ldap}->search( base => $base, filter => $filter );
        my $entry  = $mesg->pop_entry;
        $slice_ref = [$entry];
    }

    # from those age groups use all users
    # put them in a hash, for faster access and to "uniq" them
    my %all_uids;
    for my $group (@$slice_ref) {
        for my $uid ( $group->get_value("memberUid") ) {
            $all_uids{$uid} = 1;
        }
    }

    # remove the users in the blacklist
    for my $uid (@black_list) {
        delete $all_uids{$uid};    # if ( $all_uids{$uid} );
    }

    my $filter_fix_append;

    # we now have the uids of the wanted users ready, 
    # but we need to get their complete entries
    # therefor we get all user entries   
    for my $uid ( keys %all_uids ) {
        $filter_fix_append .= "(uid=$uid)";
    }
    $base = "ou=People," . $g{basedn};
    if ($filter_fix_append) {
        $filter_fix = "&(objectClass=posixAccount)(|$filter_fix_append)";
    }
    else {
        return undef;

        #$filter_fix = "&(objectClass=posixAccount)";
    }
    ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
    return @{$slice_ref};
=cut

    #make blacklist a hash, for faster access
    my (%blacklist, $blacklist_filter);
    for my $uid (@black_list) {
        $blacklist{$uid} = 1;    
	$blacklist_filter .= "(uid=$uid)";
    }
    
    # Get all users.
    $base = "ou=People," . $g{basedn};
    my $filter_fix = "&(objectClass=posixAccount)";
    unless ( -1 == $#black_list ) {
	$filter_fix .= "(!(|$blacklist_filter))";
    }
    my $filter_var = "uid=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );
  
    return @$slice_ref;

#    my @ret_list;
#    # remove the users in the blacklist
#    for my $user (@$slice_ref) {
#        my $uid = $user->get_value("uid");  
#        unless ( $blacklist{$uid} ) {
#            push @ret_list, $user;
#        }
#    }
#    
#    return @ret_list;
}

# Get all groups a user $uid is not member of, given a domain $basedn.
sub ldap_get_not_membergroups {
    my ($uid) = @_;
    my @notGroups;

    my $grpBase    = "ou=Group,"  . $g{basedn};
    my $actBase    = "ou=People," . $g{basedn};
    my $filter_fix = "&(objectClass=lisGroup)(!(memberUid=$uid))";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $grpBase, $filter_fix, $filter_var );

    foreach my $group ( @{$slice_ref} ) {
        my $gidNumber = $group->get_value('gidNumber');
        my $filter    = "(&(objectClass=posixAccount)(gidNumber=$gidNumber))";
        my $privGroup = $g{ldap}->search(
            base   => $actBase,
            filter => $filter
        );
        my $didfindgroup = $privGroup->count();
        unless ($didfindgroup) {
            push ( @notGroups, $group );
        }
    }
    return @notGroups;
}

# In a domain $basedn, get data about a user $uid. 
# If user is not found, undef is returned.
sub ldap_get_user {
    my ($uid) = @_;

    my $base   = "ou=People," . $g{basedn};
    my $filter = "uid=$uid";

    my $mesg = $g{ldap}->search( base => $base, filter => $filter );
    error_msg( "ldap_get_user: " . ldap_error_text( $mesg->code ) . "\n" )
      if $mesg->code;

    return $mesg->pop_entry;
}

# In a domain $basedn, get data about a user $uid. 
# If user is not found, undef is returned.

sub ldap_get_old_user {
    my ($uid) = @_;

    my $base   = "ou=Attic," . $g{basedn};
    my $filter = "uid=$uid";

    my $mesg = $g{ldap}->search( base => $base, filter => $filter );
    error_msg( "ldap_get_user: " . ldap_error_text( $mesg->code ) . "\n" )
      if $mesg->code;

    return $mesg->pop_entry;
}

# Returns if the group $gidNumber exists in domain $basedn.
sub ldap_group_exists {
    my ($gid) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "cn=$gid"
    );
    debug_msg( "ldap_group_exists: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;
    return $mesg->count();
}

# Returns if a given (group or user)-id exists in domain $basedn.
sub ldap_id_exists {
    my ($id) = @_;
    my $base = "ou=Group," . $g{basedn};

    my $mesg = $g{ldap}->search( base => $base, filter => "gidNumber=$id" );
    debug_msg( "ldap_id_exists: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;
    unless ( $mesg->count() ) {    # usually we should not need to do this,
            # because UID and GID are likely the same for users.
        $mesg = $g{ldap}->search( base => $base, filter => "uidNumber=$id" );
        debug_msg( "ldap_id_exists: " . ldap_error_name( $mesg->code ) . "\n" )
          if $mesg->code;
    }
    return $mesg->count();
}

# Returns the name of a group $gid in the domain $basedn,
# or the string "(not in ldap)" if the group is not found.
sub ldap_get_groupname {
    my ($gidNumber) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "gidNumber=$gidNumber",
        attrs  => "cn",
    );
    debug_msg( "ldap_get_groupname: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;

    return "(not in ldap)" unless ( $mesg->count() );

    my $entry = $mesg->pop_entry;
    return $entry->get_value('cn');
}

# Returns the group $gid in the domain $basedn,
# or undef if the group is not found.
sub ldap_get_group {
    my ($cn) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "cn=$cn"
    );
    debug_msg( "ldap_get_groupname: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;

    return $mesg->pop_entry;
}

# Returns the description of the $gid in the domain $basedn,
# or the string "(not in ldap)" if the group is not found.
sub ldap_get_groupdescription {
    my ($gidNumber) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "gidNumber=$gidNumber",
        attrs  => "cn",
    );
    debug_msg(
        "ldap_get_groupdescription: " . ldap_error_name( $mesg->code ) . "\n" )
      if $mesg->code;

    return "(not in ldap)" unless ( $mesg->count() );

    my $entry = $mesg->pop_entry;
    return $entry->get_value('description');
}

sub ldap_find_current_id {

    my @list  = ldap_get_groups();
    my $maxid = $g{minid};
    while (@list) {
        my $entry = pop @list;
        my $id    = $entry->get_value('gidNumber');
        $maxid = $id if ( $id > $maxid );
    }

    @list = ldap_get_users();
    while (@list) {
        my $entry = pop @list;
        my $id    = $entry->get_value('gidNumber');
        $maxid = $id if ( $id > $maxid );
        $id = $entry->get_value('uidNumber');
        $maxid = $id if ( $id > $maxid );
    }

    return $maxid;
}

sub ldap_create_entry_nextid {

    my $entry = Net::LDAP::Entry->new();
    $entry->dn( "ou=Variables," . $g{basedn} );
    $entry->add(
        objectclass => [ 'organizationalUnit', 'top' ],
        ou          => "Variables"
    );
    $entry->update( $g{ldap} );
    $entry = Net::LDAP::Entry->new();
    $entry->dn( "cn=nextID,ou=Variables," . $g{basedn} );
    my $current_id = ldap_find_current_id();
    $entry->add(
        objectclass => [ 'posixGroup', 'top' ],
        cn          => "nextID",
        gidNumber   => $current_id,
    );
    ldap_update( $entry );

}

# looks up cn=nextID,ouVariables,$basedn to get the new ID it should use
# function fetched from http://www.ccie.org.uk/resources/newuid.html 
# function somewhat rewritten to suit our needs
sub ldap_get_newid {
    my ($rootpw) = @_;
    my $loop = 2;
    my ( $result, $object, $newid );
    my $minid = $g{minid};
    my $maxid = $g{maxid};
    my $base  = "ou=variables," . $g{basedn};

    $result = $g{ldap}->bind( $g{rootdn}, 'password' => $rootpw );
    if ( $result->code ) {
        error_msg( "Could not bind to ldap server! "
            . ldap_error_name( $result->code ) . ": "
            . ldap_error_text( $result->code ) . "\n" );
        return undef;
    }

    while ($loop) {
        $result = $g{ldap}->search(
            base   => $base,
            filter => "cn=nextID"
        );
        $object = $result->pop_entry;
        if ( $result->code or !$object ) {

            # It the Search Fails it's most likely that this is an old
            # installation, with a missing dn=nextID, ou=Variables
            debug_msg( "Need to generate nextID!:"
                . ldap_error_text( $result->code ) . "\n" );
            ldap_create_entry_nextid();

            # Okay , try searching again
            $result = $g{ldap}->search(
                base   => $base,
                filter => "cn=nextID"
            );
            $object = $result->pop_entry;
        }

        if ($object) {
            $newid = $object->get_value('gidnumber');
            debug_msg("Got a valid nextID: $newid !\n");
        }
        else {
            debug_msg( "Got no nextID! Quit trying?!?"
                . ldap_error_text( $result->code ) . "\n" );
            undef $newid;
            last;
        }
        debug_msg("Increasing nextID!\n");
        $result = $g{ldap}->modify(
            $object,
            changes => [
                delete => [ 'gidNumber' => $newid ],
                add    => [ 'gidNumber' => $newid + 1 ]
            ]
        );
        if ( $result->code
            || getpwuid($newid)
            || getgrgid($newid)
            || ldap_id_exists($newid)
            || ( $newid > $maxid )
            || ( $newid < $minid ) )
        {
            debug_msg( "Something suspicous happend while increasing nextID! "
                . ldap_error_name( $result->code ) . "\n" );
            if ( ( $newid gt $maxid ) || ( $newid lt $minid ) ) {
                unless ($loop) {
                    error_msg("nextID out of bounds. But quit tying.\n");
                    undef $newid;
                    last;
                }
                debug_msg(
"now attempt to set nextID, since it was out of bounds: $minid, $maxid\n"
                );
                $result = $g{ldap}->modify(
                    $object,
                    changes => [
                        delete => [ 'gidNumber' => $newid + 1 ],
                        add    => [ 'gidNumber' => $minid ]
                    ]
                );
                if ( $result->code ) {
                    debug_msg(
                        "Something suspicous happend while setting nextID! "
                        . ldap_error_name( $result->code ) . "\n" );
                }
                $loop--;
            }
            debug_msg("Race Condition? Need to try again!\n");
            undef $newid;
            usleep( int( rand(500) ) );
        }
        else {
            debug_msg("I think i succeeded getting a nextID!\n");
            last;
        }
    }
    debug_msg("Returning now $newid!\n");
    $g{modified} = 1;
    return ($newid);
}

# Add a user to LDAP.
sub ldap_add_user {
    my (
        $cn,        $uid,       $userpw,
        $uidNumber, $gidNumber, $rootpw,
        $homedir,   $maildir,   $userpw_crypt
      )
      = @_;

    # if there is no given crypt-hash generate it
    unless ( $userpw_crypt ) {
        $userpw_crypt = gen_crypt($userpw);
    }

    debug_msg("Trying to add this users: uidNumber:$uidNumber  uid:$uid!\n");
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );    #XXX no error check!
    my $entry     = Net::LDAP::Entry->new();
    my $new_entry = "uid=$uid,ou=People," . $g{basedn};
    $entry->dn($new_entry);
    $entry->add(
        objectclass   => [ 'posixAccount', 'top', 'shadowAccount', 'imapUser' ],
        cn            => $cn,
        uid           => $uid,
        uidNumber     => $uidNumber,
        gidNumber     => $gidNumber,
        homeDirectory => $homedir,
        mailMessageStore => $maildir,
        userPassword     => "{crypt}" . $userpw_crypt,
        loginShell       => "/bin/bash",
    );

    return ldap_update($entry);
}

# Add a machine to LDAP.
sub ldap_add_machine {
    my (
        $cn,        $uid,  , $uidNumber,
        $gidNumber, $rootpw, 
	)
	= @_;
    
    debug_msg("Trying to add this machine: uidNumber:$uidNumber  uid:$uid!\n");
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    my $entry     = Net::LDAP::Entry->new();
    my $new_entry = "uid=$uid,ou=Machines,ou=People," . $g{basedn};
    $entry->dn($new_entry);
    $entry->add(
		objectclass   => [ 'posixAccount', 'top', 'account'],
		cn            => $cn,
		uid           => $uid,
		uidNumber     => $uidNumber,
		gidNumber     => $gidNumber,
		homeDirectory => "/dev/null",
		loginShell       => "/bin/false",
		);
    
    return ldap_update($entry);;
}

# Change an attribute $attr to the value $value for the user $uid.
sub ldap_mod_user {
    my ( $uid, $rootpw, $attr, $value ) = @_;

    $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    $g{ldap}->modify( "uid=$uid,ou=People," . $g{basedn},
        replace => { $attr => "$value" } );
    $g{modified} = 1;
}

# Change many attributes/value pairs in the 
#   hash-ref $changes_ref for the user $uid.
sub ldap_modify_user {
    my ( $uid, $changes_ref, $rootpw ) = @_;

    my %additions;    # this is for new fields in the ldap-entry
    my $user = ldap_get_user($uid);

    for my $key ( keys %$changes_ref ) {
        unless ( $user->get_value($key) ) {
	    next if ( $key eq "userPassword" );
            $additions{$key} = $changes_ref->{$key};
            delete $changes_ref->{$key};
        }
    }
    my $auth_msg = $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    return "auth_failed" if $auth_msg->code();
    

    my $mesg = $g{ldap}->modify(
        "uid=$uid,ou=People," . $g{basedn}, replace => $changes_ref,
        add => \%additions
    );
    $g{modified} = 1;
    error_msg( "ldap_modify_user Error: "
        . ldap_error_name( $mesg->code ) . " "
        . ldap_error_text( $mesg->code ) . " " )
      if ( $mesg->code );
    return undef;
}

# Change attributes/value pairs in the 
#   hash-ref $changes_ref for the group $gidNumber.
sub ldap_modify_group {
    my ( $cn, $changes_ref, $rootpw ) = @_;

    my %additions;    # this is for new fields in the ldap-entry
    my $group = ldap_get_group($cn);
    for my $key ( keys %$changes_ref ) {
        unless ( $group->get_value($key) ) {
            $additions{$key} = $changes_ref->{$key};
            delete $changes_ref->{$key};
        }
    }
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );

    if ( keys %additions ) {
        my $mesg =
          $g{ldap}->modify( "cn=$cn,ou=Group," . $g{basedn},
            add => \%additions );
        error_msg( "ldap_modify_user Error: "
            . ldap_error_name( $mesg->code ) . " "
            . ldap_error_text( $mesg->code ) . " " )
          if ( $mesg->code );
    }
    if ( keys %$changes_ref ) {
        my $mesg =
          $g{ldap}->modify( "cn=$cn,ou=Group," . $g{basedn},
            replace => $changes_ref );
        error_msg( "ldap_modify_user Error: "
            . ldap_error_name( $mesg->code ) . " "
            . ldap_error_text( $mesg->code ) . " " )
          if ( $mesg->code );
    }
    $g{modified} = 1;
}

# Make a new group $gid, with number $gidNumber, to LDAP.
sub ldap_add_group {
    my ( $cn, $gidNumber, $rootpw, $type, 
#         $age_groups_ref, 
         $description ) = @_;

    if( getgrnam( $cn ) ) {
	error_msg("Group $cn exists allready. ");
	return undef;
    } 

    unless ($type)        { $type        = "dontcare" }
    unless ($description) { $description = "dontcare" }
    
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    my $entry = Net::LDAP::Entry->new();
    $entry->dn( "cn=$cn,ou=Group," . $g{basedn} );
    
    if ( $type eq "authority_group" ) {
        $entry->add(
            objectclass => [ 'lisAclGroup', 'top', 'lisGroup' ],
            cn          => $cn,
            gidNumber   => $gidNumber,
            groupType   => $type,
            description => $description,
            member => "cn=admin,ou=People," . $g{basedn},
        );
    }
    else {
        $entry->add(
            objectclass => [ 'posixGroup', 'top', 'lisGroup' ],
            cn          => $cn,
            gidNumber   => $gidNumber,
            groupType   => $type,
            description => $description,
        );
    }
#    for my $age_group ( @{$age_groups_ref} ) {
#        ldap_add_group_to_age_group( $cn, $age_group, $rootpw );
#    }
    return ldap_update($entry);
}

sub ldap_add_group_to_age_group {
    my ( $cn, $age_group, $rootpw ) = @_;

    my $mesg = $g{ldap}->search(
        base   => $g{basedn},
        filter => "(cn=$age_group)"
    );

    my $group_entry = $mesg->pop_entry;
    return "agegroupdoesntexist" unless ($group_entry);

    $g{ldap}->bind( $g{rootdn}, password => $rootpw );

    my $group_type = $group_entry->get_value("grouptype");
    if ( $group_type eq "age_group" ) {
        $g{ldap}->modify(
            "cn=$age_group,ou=Group," . $g{basedn},
            add => { 'ageGroup' => "$cn" }
        );
	$g{modified} = 1;
        return "groupaddedtoagegroup";
    }
    else {
        return "targetgroupisnoagegroup";
    }
}

sub ldap_update_age_groups {

    # group cn wants to be member in all the 
    # ageGroups given in $age_groups_ref
    my ( $cn, $age_groups_ref, $rootpw ) = @_;

    my %age_groups_target;
    for (@$age_groups_ref) {
        $age_groups_target{$_} = 1;
    }

    for my $entry ( ldap_find_age_groups($cn) ) {
        my $cn_actual = $entry->get_value("cn");

        # eliminate the groups which are correct allready.
        if ( $age_groups_target{$cn_actual} ) {
            delete $age_groups_target{$cn_actual};
        }
        else {

            # delete the once that are not wanted anymore
            ldap_del_group_from_age_group( $cn, $cn_actual, $rootpw );
        }
    }

    # now we have the target groups left, 
    # which need to be made member in the age groups
    for my $cn_age_group ( keys %age_groups_target ) {
        ldap_add_group_to_age_group( $cn, $cn_age_group, $rootpw );
    }
}

sub ldap_find_age_groups {
    my ($cn) = @_;

    my $base       = $g{basedn};
    my $filter_fix = "&(objectClass=lisGroup)(ageGroup=$cn)";
    my $filter_var = "cn=*";
    my ( $slice_ref, $filters_applied ) =
      get_slice( $base, $filter_fix, $filter_var );

    return @{$slice_ref};
}

sub ldap_del_group_from_age_group {
    my ( $cn, $age_group, $rootpw ) = @_;

    my $mesg = $g{ldap}->search(
        base   => $g{basedn},
        filter => "(&(cn=$age_group)(ageGroup=$cn))"
    );

    my $group_entry = $mesg->pop_entry;
    return "no such agegroup with that membergroup exists"
      unless ($group_entry);

    $g{ldap}->bind( $g{rootdn}, password => $rootpw );

    my $group_type = $group_entry->get_value("grouptype");
    if ( $group_type eq "age_group" ) {
        $g{ldap}->modify(
            "cn=$age_group,ou=Group," . $g{basedn},
            delete => { 'ageGroup' => "$cn" }
        );
	$g{modified} = 1;
        return "groupdeletedfromagegroup";
    }
    else {
        return "targetgroupisnoagegroup";
    }
}

# Add a user $uid to group $gid.
#strings returned correspond to strings in the lang/en file!
sub ldap_add_user_to_group {
    my ( $uid, $gid, $rootpw ) = @_;

    my $mesg = $g{ldap}->search(
        base   => $g{basedn},
        filter => "(&(cn=$gid)(memberUid=$uid))"
    );
    return "useralreadymember" if ( $mesg->pop_entry );
    $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "(cn=$gid)"
    );
    my $group_entry = $mesg->pop_entry;
    return "groupdoesntexist" unless ($group_entry);
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    my $group_type = $group_entry->get_value("grouptype");

    if ( $group_type ne "authority_group" ) {
        $g{ldap}->modify(
            "cn=$gid,ou=Group," . $g{basedn},
            add => { 'memberUid' => "$uid" }
        );
    }
    else {
        $g{ldap}->modify(
            "cn=$gid,ou=Group," . $g{basedn},
            add => {
                'memberUid' => "$uid",
                'member'    => "uid=$uid,ou=People," . $g{basedn},
            }
        );
    }
    $g{modified} = 1;
    return "addedgroupuser";
}

# Delete a user $uid from group $gid. 
sub ldap_del_user_from_group {
    my ( $uid, $gid, $rootpw ) = @_;

    my $mesg = $g{ldap}->search(
        base   => "ou=Group," . $g{basedn},
        filter => "(&(cn=$gid)(|(memberUid=$uid)(member=uid=$uid,ou=People,".$g{basedn}.")))"
    );
    my $group_entry = $mesg->pop_entry;
    return "(not a member)" unless ($group_entry);
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    my $group_type = $group_entry->get_value("grouptype");
    if ( $group_type ne "authority_group" ) {
        $mesg = $g{ldap}->modify(
	    "cn=$gid,ou=Group," . $g{basedn},
	    delete => { 'memberUid' => "$uid" }
        );
    }
    else {
        $mesg = $g{ldap}->modify(
            "cn=$gid,ou=Group," . $g{basedn},
            delete => {
                'memberUid' => "$uid",
                'member'    => "uid=$uid,ou=People," . $g{basedn},
            }
        );
    }
    $g{modified} = 1;
    return "(User removed from group)";
}

# Delete a user $uid.
sub ldap_delete_user {
    my ( $uid, $rootpw ) = @_;
    
    $g{modified} = 1;
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    my $result = $g{ldap}->delete( "uid=$uid,ou=People,".$g{basedn} );
    return $result->code;    
}

# Move a user $uid to the attic.
sub ldap_remove_user {
    my ( $uid, $rootpw ) = @_;
    
    $g{modified} = 1;
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    my $result = $g{ldap}->moddn( "uid=$uid,ou=People,".$g{basedn},
                                  newrdn => "uid=$uid",
                                  deleteoldrdn => 1,
                                  newsuperior => "ou=Attic,".$g{basedn},
                                  );
    return $result->code;    
}

# Delete a group $gid.
sub ldap_delete_group {
    my ( $cn, $rootpw ) = @_;
    $g{modified} = 1;
    $g{ldap}->bind( $g{rootdn}, password => $rootpw );
    my $msg = $g{ldap}->delete( "cn=$cn,ou=Group," . $g{basedn} );
    return $msg->code;
}

# Print error message to browser if ldap_error.
sub print_ldap_error {
    my ( $errormsg, $result ) = @_;
    my $output = $errormsg;
    $output .= text('error') . ": \n";
    $output .= ldap_error_name( $result->code ) . ": "
      . ldap_error_text( $result->code );
    return $output;
}

# Close the connetction to the LDAP server.
sub ldap_close {
    $g{ldap}->unbind() if $g{ldap};
    if ($g{modified}) {
	system ("nscd -i passwd >/dev/null 2>&1; nscd -i group >/dev/null 2>&1") 
	    if( -e "/var/run/nscd.pid" );
    }
}

sub ldap_update {
    my( $entry ) = @_;
    my $ret = $entry->update( $g{ldap} ) if $entry;
    $g{modified} = 1;

    return $ret;
}

# Encrypt a password.
sub gen_crypt {
    my ($plain) = @_;
    my $salt;

    # Get a truly random salt.
    open FILE, "</dev/urandom";
    sysread FILE, $salt, 2;
    close FILE;

    # If crypt needs salt to be a set of usable characters, uncomment this.
    #   srand($salt);
    #   $salt = chr(int(rand(97)) + 25).chr(int(rand(97)) + 25);
    # Encrypt.
    return crypt( $plain, $salt );

}

sub make_unique_username {
    my ( $firstname, $lastname, $prefered_login ) = @_;
    my @names = user_suggest_unames( $firstname, $lastname, $prefered_login );
    unless ( @names > 0 ) {
        error_msg("Found no suitable unames from '$firstname $lastname'.\n");
        return undef;
    }
    return $names[0];
}

sub make_cn {
    my ( $first_name, $last_name ) = @_;

    my $cn = "$first_name";
    $cn .= " $last_name" if ($last_name);
    return $cn;
}

sub conv_name {
    my ( $s, $alt ) = @_;


    $s =~ s/[]/ae/g;    # Different mapping of  and ,
    $s =~ s/[]/aa/g;    # used when creating usernames.
    $s =~ s/[]/ae/g;
    $s =~ s/[]/oe/g;
    $s =~ s/[]/ue/g;


    # Vask bort ymse sre tegn
    $s =~ tr
        ({[}]|\\)
        (AOAaooaAAAAACEEEEIIIINOOOOOUUUUYaaaaaceeeeiiiinooooouuuuyaAaAooO);
    $s =~ s//Dh/g;
    $s =~ s//dh/g;
    $s =~ s//Th/g;
    $s =~ s//th/g;
    $s =~ s//ss/g;
    $s =~ s/[\200-\377]/x/g;
    $s =~ tr/A-Z/a-z/;
    $s =~ s/[^a-zA-Z0-9 ]//g;
    return $s;
}

sub user_suggest_unames {
    my ( $fname, $lname, $prefered_login ) = @_;
    my $initial;
    my $firstinit;
    my @potuname = ();
    my $goal     = 1;    # minimum number of alternatives to find.

    # to make this work with cn as input.
    $lname = ( split ( / /, $fname ) )[-1] unless $lname;

    push ( @potuname, $prefered_login ) 
	if ( $prefered_login and validate_new_uname($prefered_login) );
    my $complete_name = &conv_name( "$fname $lname", 1 );

    # Remember just the first initials.

    if ( $complete_name =~ /^(.*)[ -]+(\S+)\s+(\S+)$/ ) {

        # at least three names
        $firstinit = $1;
        $firstinit =~ s/([- ])(\S)[^- ]*/$1$2/g;
        $firstinit =~ s/^(\S).*?($|[- ])/$1/;
        $firstinit =~ s/[- ]//g;
    }

    # Remove hyphens.  People called "Geir-Ove Johnsen Hansen" generally
    # prefer "geirove" to just "geir".

    $complete_name =~ s/-//g;

    $complete_name =~ /^(\S+)?(.*\s+(\S)\S*)?\s+(\S+)?$/;
    $fname = substr( $1, 0, 8 );
    $initial = $3;
    $lname   = substr( $4, 0, 8 );

    # print "DEBUG: split into '$fname' '$initial' '$lname'\n";

    $lname = $fname unless $lname;    # Sane behaviour if only one name.

    # First try the obvious, first and last name.  Should be
    # restricted to interactively created accounts (== usually
    # non-student).

    if (0) {
        push ( @potuname, $fname ) if validate_new_uname($fname);
        push ( @potuname, $lname ) if validate_new_uname($lname);
    }

    # For people with many names, we prefer to use all initials:
    # Example:  Geir-Ove Johnsen Hansen
    #           ffff fff i       llllll
    # Here, firstinit is "GO" and initial is "J".
    #
    # gohansen gojhanse gohanse gojhans ... gojh goh
    # ssllllll ssilllll sslllll ssillll     ssil ssl
    #
    # ("ss" means firstinit, "i" means initial, "l" means last name)

    my ( $i, $j, $try );

    if ( $firstinit && length($firstinit) > 1 ) {
        $i = length($firstinit);
        my $llen = length($lname);
        $llen = 8 - $i if ( $llen > 8 - $i );
        for ( $j = $llen ; $j > 0 ; $j-- ) {
            $try = $firstinit . substr( $lname, 0, $j );
            push ( @potuname, $try ) if validate_new_uname($try);
            if ( $j > 1 && $initial ) {
                $try = $firstinit . $initial . substr( $lname, 0, $j - 1 );
                push ( @potuname, $try ) if validate_new_uname($try);
            }
            last if @potuname >= $goal;
        }
    }

    # Now try different substrings from first and last name.
    #
    # geiroveh,
    # fffffffl
    # geirovh geirovha geirovjh,
    # ffffffl ffffffll ffffffil
    # geiroh geirojh geiroha geirojha geirohan,
    # fffffl fffffil fffffll fffffill ffffflll
    # geirh geirjh geirha geirjha geirhan geirjhan geirhans
    # ffffl ffffil ffffll ffffill fffflll ffffilll ffffllll
    # ...
    # gjh gh gjha gha gjhan ghan ... gjhansen ghansen
    # fil fl fill fll filll flll     fillllll fllllll

    my $flen = length($fname);

    $flen = 7 if $flen > 7;
    for ( $i = $flen ; $i > 0 ; $i-- ) {
        my $llim = length($lname);
        $llim = 8 - $i if ( $llim > 8 - $i );
        for ( $j = 1 ; $j <= $llim ; $j++ ) {
            if ($initial) {

                # Is there room for an initial?
                if ( $j == $llim && $i + $llim < 8 ) {
                    $try =
                      substr( $fname, 0, $i ) . $initial
                      . substr( $lname, 0, $j );
                    push ( @potuname, $try ) if validate_new_uname($try);
                }

                # Is there room for an initial if we chop a letter off
                # last name?
                if ( $j > 1 ) {
                    $try =
                      substr( $fname, 0, $i ) . $initial
                      . substr( $lname, 0, $j - 1 );
                    push ( @potuname, $try ) if validate_new_uname($try);
                }
            }
            $try = substr( $fname, 0, $i ) . substr( $lname, 0, $j );
            push ( @potuname, $try ) if validate_new_uname($try);
        }
        last if @potuname >= $goal;
    }

    # Absolutely last ditch effort:  geirov1, geirov2 etc.

    $i = 1;
    $flen = 6 if $flen > 6;
    while ( @potuname < $goal && $i < 100 ) {
        $try = substr( $fname, 0, $flen ) . $i++;
        push ( @potuname, $try ) if validate_new_uname($try);
    }

    return @potuname;
}

sub validate_new_uname {
    my ($uname) = @_;

    # We only allow numbers and lowercase letters
    if ( $uname =~ m/[^a-z0-9]/ ) {
        error_msg(  text( "login_illegal_chars", $uname ) );
        return undef;
    }
    elsif ( length($uname) <= 1 ) {
        error_msg(  text( "login_too_short" ) );
        return undef;

        #    } elsif (gethostbyname($uname)) {
        #       debug_msg("Username `$uname' in use as hostname");
        #       return undef;
    }
    elsif ( ldap_get_user($uname) ) {
        error_msg( text( "login_allready_used", $uname ) );
        return undef;
    }
    elsif ( getgrnam($uname) ) {
        error_msg( text( "login_is_file_group", $uname ) );
        return undef;
    }
    elsif ( ldap_get_group($uname) ) {
        error_msg( text( "login_is_ldap_group", $uname ) );
        return undef;
    }
    
    elsif ( ldap_get_old_user($uname) ) {
        error_msg( text ( "login_was_used", $uname ) );
        return undef;
    }
    elsif ( getpwnam($uname) || grep { $_ eq $uname; }

        qw(root bin daemon adm lp sync shutdown halt mail nobody
        nobodyv ftp uucp uucpa auth cron core tcb ris wnn sys
        nuucp hpdb opc_op predmail smbnull sysadm cmwlogin diag
        auditor dbadmin sgiweb rfindd ezsetup demos outofbox
        guest 4dgifts noaccess rpcuser teachers admins students 
	jradmins)
      )
    {
        error_msg("'$uname' is a system user.\n");
        return undef;
    }
    elsif ( getpwnam($uname) || grep { $_ eq $uname; }

        qw(shit fuck dimwit whore fuckwit)
      )    # to be continued...
    {
        error_msg("'$uname' is a dirty word.\n");
        return undef;
    }
    return 1;
}

sub error_msg {
    my ($mesg) = @_;
    push @{ $g{err} }, $mesg;
}

sub ldap_err_output {
    my $ret;

    while ( @{ $g{err} } ) {
        $ret .= shift @{ $g{err} };
    }
    return $ret;
}

sub debug_msg {
    my ($mesg) = @_;
    push @{ $g{dbg} }, $mesg;
}

sub ldap_dbg_output {
    my $ret;

    while ( @{ $g{dbg} } ) {
        $ret .= shift @{ $g{dbg} };
    }
    return $ret;
}

# this function does not at all belong in the ldap module. 
# needs to go elsewhere later.
sub read_adduser_config {
    my $config_file = "/etc/adduser.conf";
    read_file( $config_file, \%config );    # reuse a function from web-lib.pl
         #now the config data is in the config hash. (c:

}

sub read_miniserv_config {
    my $config_file = "/etc/webmin/miniserv.conf";
    read_file( $config_file, \%config );    # reuse a function from web-lib.pl
}

1;

__END__
