/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "pushbutton.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sigc++/object_slot.h>

// How much to pad the edges by if we don't
// have a background
static const unsigned no_back_padding = 6;

namespace wftk {

PushButton::PushButton() :
  upPic_(0),
  downPic_(0),
  light_(128, 128, 128),
  dark_(64, 64, 64)
{ 
  getResourceBackground("pushbutton");

  setPadding(no_back_padding, no_back_padding, no_back_padding, no_back_padding);
  setUpPic("pushbutton_up_surface");
  setDownPic("pushbutton_down_surface");

  pressed.connect(SigC::slot(*this, &PushButton::handlePressed));
  released.connect(SigC::slot(*this, &PushButton::handleReleased));
  handleReleased(); // set the background

  Color::Resource *shadow_light_color = Color::registry.get("shadow_light_color");
  if(shadow_light_color)
    light_ = shadow_light_color->res();
  Color::Resource *shadow_dark_color = Color::registry.get("shadow_dark_color");
  if(shadow_dark_color)
    dark_ = shadow_dark_color->res();
}


PushButton::PushButton(const std::string& text, const Font &font) :
  Button(text, font),
  upPic_(0),
  downPic_(0),
  light_(128, 128, 128),
  dark_(64, 64, 64)
{ 
  getResourceBackground("pushbutton");

  setPadding(no_back_padding, no_back_padding, no_back_padding, no_back_padding);
  setUpPic("pushbutton_up_surface");
  setDownPic("pushbutton_down_surface");

  pressed.connect(SigC::slot(*this, &PushButton::handlePressed));
  released.connect(SigC::slot(*this, &PushButton::handleReleased));
  handleReleased(); // set the background

  Color::Resource *shadow_light_color = Color::registry.get("shadow_light_color");
  if(shadow_light_color)
    light_ = shadow_light_color->res();
  Color::Resource *shadow_dark_color = Color::registry.get("shadow_dark_color");
  if(shadow_dark_color)
    dark_ = shadow_dark_color->res();
}

PushButton::~PushButton()
{
  if(upPic_)
    upPic_->free();
  if(downPic_)
    downPic_->free();
}

void PushButton::draw(Surface& surf, const Point& offset, const Region& r)
{
  Button::draw(surf, offset, r);

  if(noBackground()) {

    Region border1 = Rect(0, height() - 2, width(), 2);
    border1 |= Rect(0, 0, 2, height());
    border1.offset(offset);
    border1 &= r;

    Region border2 = Rect(0, 0, width(), 2);
    border2 |= Rect(width() - 2, 0, 2, height());
    border2 -= Point(0, 1);
    border2 -= Point(width() - 2, height() - 1);
    border2.offset(offset);
    border2 &= r;

    if(isPressed()) {
      surf.fill(border1, light_);
      surf.fill(border2, dark_);
    }
    else {
      surf.fill(border1, dark_);
      surf.fill(border2, light_);
    }
  }
  
  if(hasFocus())
    {
      Region border = Rect(2, 2, width() - 4, 1);
      border |= Rect(height() - 3, 2, width() - 4, 1);
      border |= Rect(2, 2, 1, height() - 4);
      border |= Rect(2, width() - 3, 1, height() - 4);
      border.offset(offset);
      border &= r;
      surf.fill(border, light_);
    }
}

void PushButton::setUpPic(Surface::Resource* bm)
{
  bool had_back = anyBackground();

  if(bm && bm->res()->empty())
    bm = 0;

  if(bm == upPic_)
    return;

  if(upPic_)
    upPic_->free();

  upPic_ = bm;

  if(upPic_)
    upPic_->bind();

  if(had_back != anyBackground()) {
    unsigned padding = anyBackground() ? 0 : no_back_padding;
    setPadding(padding, padding, padding, padding);
  }

  if(!isPressed())
    setBackground(upPic_, false);
}

void PushButton::setUpPic(const Surface& surf)
{
  Surface* s = new Surface(surf);
  if(s->width() != width() || s->height() != height())
    s->scale(width(), height());

  Surface::Resource* r = new Surface::Resource(s);
  setUpPic(r);
  r->free();
}

void PushButton::setUpPic(const std::string& name)
{
  Surface::Resource *surface = Surface::registry.get(name);
  if(surface)
    setUpPic(surface);
}

void PushButton::setDownPic(Surface::Resource* bm)
{
  bool had_back = anyBackground();

  if(bm && bm->res()->empty())
    bm = 0;

  if(bm == downPic_)
    return;

  if(downPic_)
    downPic_->free();
  downPic_ = bm;
  if(downPic_)
    downPic_->bind();

  if(isPressed())
    setBackground(downPic_, false);

  if(had_back != anyBackground()) {
    unsigned padding = anyBackground() ? 0 : no_back_padding;
    setPadding(padding, padding, padding, padding);
  }
}

void PushButton::setDownPic(const Surface& surf)
{
  Surface* s = new Surface(surf);
  if(s->width() != width() || s->height() != height())
    s->scale(width(), height());

  Surface::Resource* r = new Surface::Resource(s);
  setDownPic(r);
  r->free();
}

void PushButton::setDownPic(const std::string& name)
{
  Surface::Resource *surface = Surface::registry.get(name);
  if(surface)
    setDownPic(surface);
}

} // namespace wftk
