/* copyop.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2005 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: copyop.cc,v 1.69 2005/10/19 18:06:50 ralf Exp $ */

#include "listermode.h"
#include "normalmode.h"
#include "worker.h"
#include "copyop.h"

const char *CopyOp::name="CopyOp";

CopyOp::CopyOp() : FunctionProto()
{
  follow_symlinks=false;
  move=false;
  do_rename=false;
  same_dir=false;
  request_dest=false;
  request_flags=false;
  overwrite=COPYOP_OVERWRITE_NORMAL;
  fastcopy=COPYOP_NORMALCOPY;
  preserve_attr=true;
  hasConfigure = true;
}

CopyOp::~CopyOp()
{
}

CopyOp*
CopyOp::duplicate() const
{
  CopyOp *ta=new CopyOp();
  ta->follow_symlinks=follow_symlinks;
  ta->move=move;
  ta->do_rename=do_rename;
  ta->same_dir=same_dir;
  ta->request_dest=request_dest;
  ta->request_flags=request_flags;
  ta->overwrite=overwrite;
  ta->fastcopy=fastcopy;
  ta->preserve_attr=preserve_attr;
  return ta;
}

bool
CopyOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *
CopyOp::getName()
{
  return name;
}

int
CopyOp::run( WPUContext *wpu, ActionMessage *msg )
{
  ListerMode *lm1;
  if(msg->mode!=msg->AM_MODE_DNDACTION) {
    Lister *l1 = msg->getWorker()->getActiveLister();
    if(l1!=NULL) {
      startlister=l1;
      endlister = msg->getWorker()->getOtherLister(startlister);
      lm1=startlister->getActiveMode();
      if(lm1!=NULL)
        if(lm1->isType("NormalMode")==true)
          normalmodecopy( msg );
    }
  } else {
    normalmodecopy( msg );
  }
  return 0;
}

bool
CopyOp::save(Datei *fh)
{
  if ( fh == NULL ) return false;
  WConfig::configPutPairBool( fh, "followsymlinks", follow_symlinks );
  WConfig::configPutPairBool( fh, "move", move );
  WConfig::configPutPairBool( fh, "rename", do_rename );
  WConfig::configPutPairBool( fh, "samedir", same_dir );
  WConfig::configPutPairBool( fh, "requestdest", request_dest );
  WConfig::configPutPairBool( fh, "requestflags", request_flags );
  switch(overwrite) {
    case COPYOP_OVERWRITE_ALWAYS:
      WConfig::configPutPair( fh, "overwrite", "always" );
      break;
    case COPYOP_OVERWRITE_NEVER:
      WConfig::configPutPair( fh, "overwrite", "never" );
      break;
    default:
      WConfig::configPutPair( fh, "overwrite", "normal" );
      break;
  }
  switch(fastcopy) {
    case COPYOP_FASTCOPY:
      WConfig::configPutPair( fh, "copymode", "fast" );
      break;
    case COPYOP_FASTESTCOPY:
      WConfig::configPutPair( fh, "copymode", "fastest" );
      break;
    default:
      WConfig::configPutPair( fh, "copymode", "normal" );
      break;
  }
  WConfig::configPutPairBool( fh, "preserveattr", preserve_attr );
  return true;
}

const char *
CopyOp::getDescription()
{
  return catalog.getLocaleCom(8);
}

int
CopyOp::normalmodecopy( ActionMessage *am )
{
  struct NM_copyorder copyorder;
  NormalMode *nm1=NULL,
             *nm2=NULL;
  ListerMode *lm1=NULL;
  NM_specialsourceExt *specialsource=NULL;
  bool do_request,cont=true;
  char *tstr,*tstr2,*dest=NULL;
  
  if(am->mode==am->AM_MODE_DNDACTION) {
    lm1 = am->dndmsg->getSourceMode();
    if(lm1==NULL) return 1;
    if(lm1->isType("NormalMode")!=true) return 1;
    nm1=(NormalMode*)lm1;
    lm1 = am->dndmsg->getDestMode();
    if(lm1!=NULL)
      if(lm1->isType("NormalMode")==true)
        nm2=(NormalMode*)lm1;
  } else {
    if(startlister==NULL) return 1;
    lm1=startlister->getActiveMode();
    if(lm1==NULL) return 1;
    if(lm1->isType("NormalMode")!=true) return 1;
    nm1=(NormalMode*)lm1;
    if(endlister!=NULL) {
      lm1=endlister->getActiveMode();
      if(lm1!=NULL)
        if(lm1->isType("NormalMode")==true)
          nm2=(NormalMode*)lm1;
    }
  }
  
  if(request_flags==true) {
    if(doconfigure(1)!=0) cont=false;
  } else {
    // set values in t* variables
    tfollow_symlinks=follow_symlinks;
    tmove=move;
    trename=do_rename;
    tsame_dir=same_dir;
    trequest_dest=request_dest;
    toverwrite=overwrite;
    tfastcopy=fastcopy;
    tpreserve_attr=preserve_attr;
  }
  
  if(cont==true) {
    memset( &copyorder, 0, sizeof( copyorder ) );
    copyorder.ignoreLosedAttr = false;
    // detect what to copy
    if(am->mode==am->AM_MODE_ONLYACTIVE)
      copyorder.source=copyorder.NM_ONLYACTIVE;
    else if(am->mode==am->AM_MODE_DNDACTION) {
      // insert DND-element into list
      copyorder.source=copyorder.NM_SPECIAL;
      copyorder.sources=new std::list<NM_specialsourceExt*>;
      specialsource = new NM_specialsourceExt( am->dndmsg->getFE() );
      specialsource->row=-1;
      copyorder.sources->push_back(specialsource);
    } else if(am->mode==am->AM_MODE_SPECIAL) {
      copyorder.source=copyorder.NM_SPECIAL;
      copyorder.sources=new std::list<NM_specialsourceExt*>;
      specialsource = new NM_specialsourceExt( am->getFE() );
      specialsource->row=-1;
      copyorder.sources->push_back(specialsource);
    } else
      copyorder.source=copyorder.NM_ALLENTRIES;

    /* now source defined
       next find destination
       
       Priority:
        1.Flag for requesting destination
        2.Flag for same dir
        3.When DNDAction destination of DND
        4.nonactive lister
       
       when no destination then also requesting
    */

    do_request=false;
    if(trequest_dest==true)
      do_request=true;
    else if(tsame_dir==true) {
      tstr = nm1->getCurrentDir();
      if(tstr!=NULL)
        dest=dupstring(tstr);
    } else if(am->mode==am->AM_MODE_DNDACTION) {
      const char *ddstr = am->dndmsg->getDestDir();
      if(ddstr!=NULL) dest=dupstring(ddstr);
      else dest=NULL;
    } else {
      if(nm2!=NULL) {
        tstr = nm2->getCurrentDir();
        if(tstr!=NULL)
          dest=dupstring(tstr);
        else
          do_request=true;
      } else
        do_request=true;
    }

    if(do_request==true) {
      const char *defdir;
      if(dest!=NULL) _freesafe(dest);
      
      if ( nm2 != NULL ) {
        defdir = nm2->getCurrentDir();
        if ( defdir != NULL ) {
          if ( strlen( defdir ) < 1 )
            defdir = nm1->getCurrentDir();
        } else {
          defdir = nm1->getCurrentDir();
        }
      } else {
        defdir = nm1->getCurrentDir();
      }
      
      if ( requestdest( defdir, &dest ) != 0 ) {
        _freesafe(dest);
        dest=NULL;
      }
    }
    /* if dest==null nothing to do
       otherwise: */
    if(dest!=NULL) {
      /* if dest starts with no slash then take it as local dir
         so add currentDir */
      if(dest[0]!='/') {
        tstr=catstring(nm1->getCurrentDir(),"/");
        tstr2=catstring(tstr,dest);
        _freesafe(tstr);
        _freesafe(dest);
        dest=tstr2;
      }
      /* now also destination determined
         next describing what to do */
      copyorder.move=tmove;
      copyorder.follow_symlinks=tfollow_symlinks;
      copyorder.preserve_attr=tpreserve_attr;
      copyorder.do_rename=trename;
      switch(toverwrite) {
        case COPYOP_OVERWRITE_ALWAYS:
          copyorder.overwrite=copyorder.NM_OVERWRITE_ALWAYS;
          break;
        case COPYOP_OVERWRITE_NEVER:
          copyorder.overwrite=copyorder.NM_OVERWRITE_NEVER;
          break;
        default:
          copyorder.overwrite=copyorder.NM_OVERWRITE_NORMAL;
      }
      switch(tfastcopy) {
        case COPYOP_FASTCOPY:
          copyorder.copymode=copyorder.NM_COPYMODE_NODATABASE;
          break;
        default:
          copyorder.copymode=copyorder.NM_COPYMODE_NORMAL;
          break;
      }
      copyorder.destdir=dest;
      // now start copy process
      copyorder.cowin=new CopyOpWin( am->getWorker()->getAGUIX(),
                                     copyorder.move );
      nm1->copy(&copyorder);
      delete copyorder.cowin;
    }
    if(dest!=NULL) _freesafe(dest);
    if(copyorder.source==copyorder.NM_SPECIAL) {
      if ( specialsource != NULL ) delete specialsource;
      delete copyorder.sources;
    }
  }
  return 0;
}

int
CopyOp::doconfigure(int mode)
{
  AGUIX *aguix = Worker::getAGUIX();
  AWindow *win;
  ChooseButton *fscb,*mcb,*rcb,*sdcb,*rfcb=NULL,*rdcb,*pacb;
//  CycleButton *ocyb,*fccyb;
  CycleButton *fccyb;
  AGMessage *msg;
  int endmode=-1;
  Requester *req;
  char *tstr;
  int i;
  const int cincw = AContainer::ACONT_MINH +
                    AContainer::ACONT_MINW +
                    AContainer::ACONT_MAXH;
  const int cincwnr = cincw +
                      AContainer::ACONT_NORESIZE;
  const int cfix = AContainer::ACONT_MINH +
                   AContainer::ACONT_MINW +
                   AContainer::ACONT_MAXH +
                   AContainer::ACONT_MAXW;
  
  req=new Requester(aguix);

  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(catalog.getLocaleCom(8))+1);
  sprintf(tstr,catalog.getLocale(293),catalog.getLocaleCom(8));
  win = new AWindow( aguix, 10, 10, 10, 10, 0, tstr );
  win->create();
  _freesafe(tstr);

  AContainer *ac1 = win->setContainer( new AContainer( win, 1, 10 ), true );
  ac1->setMinSpace( 5 );
  ac1->setMaxSpace( 5 );

  ac1->add( new Text( aguix, 0, 0, catalog.getLocaleCom( 8 ), 1 ), 0, 0, cincwnr );

  fscb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( follow_symlinks == true ) ? 1 : 0,
						    catalog.getLocale( 298 ), LABEL_RIGHT, 1, 0 ), 0, 1, cincwnr );

  mcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( move == true ) ? 1 : 0,
						   catalog.getLocale( 299 ), LABEL_RIGHT, 1, 0 ), 0, 2, cincwnr );

  rcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( do_rename == true ) ? 1 : 0,
						   catalog.getLocale( 300 ), LABEL_RIGHT, 1, 0 ), 0, 3, cincwnr );

  sdcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( same_dir == true ) ? 1 : 0,
						    catalog.getLocale( 301 ), LABEL_RIGHT, 1, 0 ), 0, 4, cincwnr );

  rdcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( request_dest == true ) ? 1 : 0,
						    catalog.getLocale( 302 ), LABEL_RIGHT, 1, 0 ), 0, 5, cincwnr );

  pacb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( preserve_attr == true ) ? 1 : 0,
						    catalog.getLocale( 150 ), LABEL_RIGHT, 1, 0 ), 0, 6, cincwnr );

/*  ttext=(Text*)win->add(new Text(aguix,x,y,catalog.getLocale(303),1));
  x+=ttext->getWidth()+5;
  ocyb=(CycleButton*)win->add(new CycleButton(aguix,x,y,100,1,0,0));
  ocyb->addOption(catalog.getLocale(304));
  ocyb->addOption(catalog.getLocale(305));
  ocyb->addOption(catalog.getLocale(306));
  ocyb->resize(ocyb->getMaxSize(),ocyb->getHeight());
  switch(overwrite) {
    case COPYOP_OVERWRITE_ALWAYS:
      ocyb->setOption(1);
      break;
    case COPYOP_OVERWRITE_NEVER:
      ocyb->setOption(2);
      break;
    default:
      ocyb->setOption(0);
      break;
  }
  
  y+=ocyb->getHeight()+5;
  tw=x+ocyb->getWidth()+5;
  if(tw>w) w=tw;
  x=5;*/

  AContainer *ac1_1 = ac1->add( new AContainer( win, 2, 1 ), 0, 7 );
  ac1_1->setMinSpace( 5 );
  ac1_1->setMaxSpace( 5 );
  ac1_1->setBorderWidth( 0 );

  ac1_1->add( new Text( aguix, 0, 0, catalog.getLocale( 307 ), 1 ), 0, 0, cfix );
  fccyb = (CycleButton*)ac1_1->add( new CycleButton( aguix, 0, 0, 100, 1, 0, 0 ), 1, 0, cincw );
  fccyb->addOption(catalog.getLocale(308));
  fccyb->addOption(catalog.getLocale(309));
//  fccyb->addOption("schnellster Modus");
  fccyb->resize(fccyb->getMaxSize(),fccyb->getHeight());
  switch(fastcopy) {
    case COPYOP_FASTCOPY:
      fccyb->setOption(1);
      break;
/*    case COPYOP_FASTESTCOPY:
      fccyb->setOption(2);
      break;*/
    default:
      fccyb->setOption(0);
      break;
  }
  ac1_1->readLimits();
  
  if(mode==0) {
    rfcb = (ChooseButton*)ac1->add( new ChooseButton( aguix, 0, 0, 20, 20, ( request_flags == true ) ? 1 : 0,
						      catalog.getLocale( 294 ), LABEL_RIGHT, 1, 0 ), 0, 8, cincwnr );
  }

  AContainer *ac1_2 = ac1->add( new AContainer( win, 2, 1 ), 0, 9 );
  ac1_2->setMinSpace( 5 );
  ac1_2->setMaxSpace( -1 );
  ac1_2->setBorderWidth( 0 );
  Button *okb =(Button*)ac1_2->add( new Button( aguix,
                                                0,
                                                0,
                                                catalog.getLocale( 11 ),
                                                1,
                                                0,
                                                0 ), 0, 0, cfix );
  Button *cb = (Button*)ac1_2->add( new Button( aguix,
						0,
						0,
						catalog.getLocale( 8 ),
						1,
						0,
						0 ), 1, 0, cfix );
  
  okb->takeFocus();
  win->setDoTabCycling( true );
  win->contMaximize( true );
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) {
            endmode = 0;
          } else if(msg->button.button==cb) endmode=1;
          break;
        case AG_KEYPRESSED:
          if(win->isParent(msg->key.window,false)==true) {
            switch(msg->key.key) {
              case XK_1:
                fscb->setState((fscb->getState()==1)?0:1);
                break;
              case XK_2:
                mcb->setState((mcb->getState()==1)?0:1);
                break;
              case XK_3:
                rcb->setState((rcb->getState()==1)?0:1);
                break;
              case XK_4:
                sdcb->setState((sdcb->getState()==1)?0:1);
                break;
              case XK_5:
                rdcb->setState((rdcb->getState()==1)?0:1);
                break;
              case XK_6:
/*                i=ocyb->getSelectedOption()+1;
                if(i>=3) i=0;
                ocyb->setOption(i);
                break;
              case XK_7:*/
                i=fccyb->getSelectedOption()+1;
                if(i>=2) i=0;
                fccyb->setOption(i);
                break;
              case XK_Return:
                if ( cb->getHasFocus() == false ) {
                  endmode = 0;
                }
                break;
              case XK_Escape:
                endmode=1;
                break;
            }
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if(endmode==0) {
    // ok
    if(mode==1) {
      // store in t-variables
      tfollow_symlinks=(fscb->getState()==1)?true:false;
      tmove=(mcb->getState()==1)?true:false;
      trename=(rcb->getState()==1)?true:false;
      tsame_dir=(sdcb->getState()==1)?true:false;
      trequest_dest=(rdcb->getState()==1)?true:false;
      tpreserve_attr=(pacb->getState()==1)?true:false;
      toverwrite=COPYOP_OVERWRITE_NORMAL;
/*      switch(ocyb->getSelectedOption()) {
        case 1:
          toverwrite=COPYOP_OVERWRITE_ALWAYS;
          break;
        case 2:
          toverwrite=COPYOP_OVERWRITE_NEVER;
          break;
        default:
          toverwrite=COPYOP_OVERWRITE_NORMAL;
          break;
      }*/
      switch(fccyb->getSelectedOption()) {
        case 1:
          tfastcopy=COPYOP_FASTCOPY;
          break;
/*        case 2:
          tfastcopy=COPYOP_FASTESTCOPY;
          break;*/
        default:
          tfastcopy=COPYOP_NORMALCOPY;
          break;
      }
    } else {
      // store in normal variables
      follow_symlinks=(fscb->getState()==1)?true:false;
      move=(mcb->getState()==1)?true:false;
      do_rename=(rcb->getState()==1)?true:false;
      same_dir=(sdcb->getState()==1)?true:false;
      request_dest=(rdcb->getState()==1)?true:false;
      request_flags=(rfcb->getState()==1)?true:false;
      preserve_attr=(pacb->getState()==1)?true:false;
      overwrite=COPYOP_OVERWRITE_NORMAL;
/*      switch(ocyb->getSelectedOption()) {
        case 1:
          overwrite=COPYOP_OVERWRITE_ALWAYS;
          break;
        case 2:
          overwrite=COPYOP_OVERWRITE_NEVER;
          break;
        default:
          overwrite=COPYOP_OVERWRITE_NORMAL;
          break;
      }*/
      switch(fccyb->getSelectedOption()) {
        case 1:
          fastcopy=COPYOP_FASTCOPY;
          break;
/*        case 2:
          fastcopy=COPYOP_FASTESTCOPY;
          break;*/
        default:
          fastcopy=COPYOP_NORMALCOPY;
          break;
      }
    }
  }
  
  delete win;
  delete req;

  return endmode;
}

int
CopyOp::configure()
{
  return doconfigure(0);
}

int
CopyOp::requestdest( const char *defaultstr, char **dest )
{
  Requester *req=new Requester( Worker::getAGUIX());
  char *buttonstr;
  const char *textstr;
  int erg;
  
  textstr=catalog.getLocale(212);
  buttonstr=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+
                              strlen(catalog.getLocale(8))+1);
  sprintf(buttonstr,"%s|%s",catalog.getLocale(11),
                            catalog.getLocale(8));
  erg = req->string_request( catalog.getLocale( 123 ),
                             textstr,
                             ( defaultstr != NULL ) ? defaultstr : "",
                             buttonstr,
                             dest,
                             Requester::REQUEST_SELECTALL );
  _freesafe(buttonstr);
  delete req;
  return erg;
}

void CopyOp::setFollowSymlinks(bool nv)
{
  follow_symlinks=nv;
}

void CopyOp::setMove(bool nv)
{
  move=nv;
}

void CopyOp::setRename(bool nv)
{
  do_rename=nv;
}

void CopyOp::setSameDir(bool nv)
{
  same_dir=nv;
}

void CopyOp::setRequestDest(bool nv)
{
  request_dest=nv;
}

void CopyOp::setRequestFlags(bool nv)
{
  request_flags=nv;
}

void CopyOp::setOverwrite(overwrite_t nv)
{
  overwrite=nv;
}

void CopyOp::setPreserveAttr(bool nv)
{
  preserve_attr=nv;
}

void CopyOp::setFastcopy( fastcopy_t nv )
{
  fastcopy = nv;
}

/*******************
 * Class CopyOpWin *
 *******************/

CopyOpWin::CopyOpWin( AGUIX*taguix, bool _move )
{
  this->aguix=taguix;
  source=dupstring("");
  dest=dupstring("");
  copied_bytes_curfile=0;
  bytes_curfile=0;
  copied_bytes=0;
  bytes=0;
  copied_files=0;
  files=0;
  copied_dirs=0;
  dirs=0;
  slim=false;
  
  window=NULL;
  melw = 0;
  filenamespace[0] = 0;
  filenamespace[1] = 0;
  ismessage[0] = false;
  ismessage[1] = false;

  update_sourcetext = false;
  update_desttext = false;
  update_filetext = false;
  update_globtext = false;
  update_files2gotext = false;
  update_dirs2gotext = false;
  
  mintime = 1000000;
  lastbytes = 0;
  lastrate = 0.0;
  
  glob_percent = 0;
  
  move = _move;
  
  lencalc = new AFontWidth( aguix, NULL );
  memset( &stoptime, 0, sizeof( stoptime ) );
  timer_stopped = false;
}

CopyOpWin::~CopyOpWin()
{
  if(source!=NULL) _freesafe(source);
  if(dest!=NULL) _freesafe(dest);
  if(window!=NULL) close();
  delete lencalc;
}

int
CopyOpWin::open(bool tslim)
{
  int w,h;
  int tx,ty;
  Text *ttext;

  close();
  this->slim=tslim;
  w=400;
  h=10;
  window = new AWindow( aguix,
			10, 10,
			w, h,
			0, ( move == true ) ? catalog.getLocale( 299 ) : catalog.getLocaleCom( 8 ) );
  window->create();
  tx=5;
  ty=5;
  sourcetext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  ismessage[0] = true;
  ty+=sourcetext->getHeight()+5;
  desttext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  ismessage[1] = true;
  ty+=desttext->getHeight()+5;
  fsb=(SolidButton*)window->add(new SolidButton(aguix,tx+1,ty+1,w-2*(tx+1),"",2,3,0));
  sbw=fsb->getWidth();
  bb1=(BevelBox*)window->add(new BevelBox(aguix,tx,ty,w-2*tx,fsb->getHeight()+2,1));
  fsb->toFront();
  ty+=bb1->getHeight()+5;
  tx=10;
  filetext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  ty+=filetext->getHeight()+5;

  ttext = (Text*)window->add( new Text( aguix, tx, ty, catalog.getLocale( 90 ), 1 ) );
  tx += ttext->getWidth() + 5;
  fileavgtext = (Text*)window->add( new Text( aguix, tx, ty, "", 1 ) );
  ty += fileavgtext->getHeight() + 5;

  tx=5;
  bb2=(BevelBox*)window->add(new BevelBox(aguix,tx,ty,w-2*tx,2,0));
  ty+=bb2->getHeight()+5;

  ttext=(Text*)window->add(new Text(aguix,tx,ty,catalog.getLocale((slim==true)?284:118),1));
  tx+=ttext->getWidth()+5;
  files2gotext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  tx=5;
  ty+=ttext->getHeight()+5;
  ttext=(Text*)window->add(new Text(aguix,tx,ty,catalog.getLocale((slim==true)?285:119),1));
  tx+=ttext->getWidth()+5;
  dirs2gotext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  tx=5;
  ty+=ttext->getHeight()+5;
  if(slim==false) {
    gsb=(SolidButton*)window->add(new SolidButton(aguix,tx+1,ty+1,w-2*(tx+1),"",2,3,0));
    bb3=(BevelBox*)window->add(new BevelBox(aguix,tx,ty,w-2*tx,gsb->getHeight()+2,1));
    gsb->toFront();
    ty+=bb3->getHeight()+5;
  }
  tx=10;
  ttext=(Text*)window->add(new Text(aguix,tx,ty,catalog.getLocale(120),1));
  tx+=ttext->getWidth()+5;
  globtext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
  tx=5;
  ty+=globtext->getHeight()+5;
  if(slim==false) {
    ttext=(Text*)window->add(new Text(aguix,tx,ty,catalog.getLocale(121),1));
    tx+=ttext->getWidth()+5;
    timetext=(Text*)window->add(new Text(aguix,tx,ty,"",1));
    ty+=timetext->getHeight()+5;
  }
  tx=5;
  bb4=(BevelBox*)window->add(new BevelBox(aguix,tx,ty,w-2*tx,2,0));
  ty+=bb4->getHeight()+5;
  cb = (Button*)window->add( new Button( aguix, tx, ty, catalog.getLocale( 8 ), 1, 0, 0 ) );
  cb->move(w/2-cb->getWidth()/2,cb->getY());
  ty+=cb->getHeight()+5;
  h=ty;
  window->setDoTabCycling( true );
  window->resize(w,h);
  window->setMinSize(w,h);
  //  window->setMaxSize(w,h);
  window->centerScreen();
  window->show();

  filenamespace[0] = w - window->getBorderWidth() - sourcetext->getX();
  filenamespace[0] -= aguix->getTextWidth( catalog.getLocale( ( move == true ) ? 594 : 116 ) ) - 2 * aguix->getTextWidth( " " );
  filenamespace[1] = w - window->getBorderWidth() - desttext->getX();
  filenamespace[1] -= aguix->getTextWidth( catalog.getLocale( 117 ) ) - 2 * aguix->getTextWidth( " " );

  return 0;
}

void
CopyOpWin::close()
{
  if(window!=NULL) {
    delete window;
  }
  window=NULL;
}

void
CopyOpWin::starttimer()
{
  gettimeofday( &starttime, NULL );
}

void
CopyOpWin::set_files_to_copy(long nfiles)
{
  if(nfiles>0) files=nfiles;
  else files=0;
  update_files2gotext=true;
}

void
CopyOpWin::set_dirs_to_copy(long ndirs)
{
  if(ndirs>0) dirs=ndirs;
  else dirs=0;
  update_dirs2gotext=true;
}

void
CopyOpWin::set_bytes_to_copy(loff_t nbytes)
{
  if(nbytes>0) bytes=nbytes;
  else bytes=0;
  update_globtext=true;
}

void
CopyOpWin::set_bytes_to_copy_curfile( loff_t nbytes )
{
  if ( nbytes > 0 ) bytes_curfile = nbytes;
  else bytes_curfile = 0;
  update_filetext = true;
}

void
CopyOpWin::dir_finished()
{
  copied_dirs++;
  update_dirs2gotext=true;
}

void
CopyOpWin::file_finished()
{
  // inc counter
  copied_files++;
  update_files2gotext=true;
}

void
CopyOpWin::add_curbytes_copied( loff_t nbytes )
{
  if ( nbytes > 0 ) {
    copied_bytes_curfile += nbytes;
    copied_bytes += nbytes;
  }
  update_filetext = true;
  update_globtext = true;
}

void
CopyOpWin::newfile(char *name,char *ndest)
{
  if(source!=NULL) _freesafe(source);
  if(name!=NULL)
    source=dupstring(name);
  else
    source=dupstring("");
  if(dest!=NULL) _freesafe(dest);
  if(ndest!=NULL)
    dest=dupstring(ndest);
  else
    dest=dupstring("");
  gettimeofday( &filestarttime, NULL );
  update_sourcetext=true;
  update_desttext=true;
  ismessage[0] = false;
  ismessage[1] = false;
  copied_bytes_curfile = 0;
  update_filetext=true;

  lastbytes = 0;
  lasttime = filestarttime;
  lastrate = 0.0;
}

int
CopyOpWin::redraw()
{ /* this functions will redraw the window AND process events for this window
   * returns values:
   *  0 for regular exit
   *  1 when clicked Cancel-Button or closed window
   *  other not implemented */
  int returnvalue=0;
  long rtime;
  double drtime;
  struct timeval curtime;
  double difftime2, lastdiff2;
  char tstr[1024],
       *tstr2;
  double rate;
  int tw;
  bool doresize = false;
  int newglob_percent;
  std::string copied_bytes_curfile_str,
              bytes_curfile_str,
              copied_bytes_str,
              bytes_str;
  
  char *tstr3;

  timerclear( &curtime );
  if ( ( update_sourcetext == true ) && ( ismessage[0] == false ) ) {
    if ( source != NULL ) {
      tstr3 = Datei::shrinkFilename( source, filenamespace[0], *lencalc );
      if ( tstr3 == NULL)
        tstr3 = dupstring( source );
      tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( ( move == true ) ? 594 : 116 ) ) + strlen( tstr3 ) + 1 );
      sprintf( tstr2, catalog.getLocale( ( move == true ) ? 594 : 116 ), tstr3 );
      _freesafe( tstr3 );
    } else {
      tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( ( move == true ) ? 594 : 116 ) ) + 1 );
      sprintf( tstr2, catalog.getLocale( ( move == true ) ? 594 : 116 ), "" );
    }
    sourcetext->setText(tstr2);
    _freesafe(tstr2);
    /* no need for explicit redraw */
    update_sourcetext=false;
  }
  if ( ( update_desttext == true ) && ( ismessage[1] == false ) ) {
    if ( dest != NULL ) {
      tstr3 = Datei::shrinkFilename( dest, filenamespace[1], *lencalc );
      if ( tstr3 == NULL)
        tstr3 = dupstring( dest );
      tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( 117 ) ) + strlen( tstr3 ) + 1 );
      sprintf( tstr2, catalog.getLocale( 117 ), tstr3 );
      _freesafe(tstr3);
    } else {
      tstr2 = (char*)_allocsafe( strlen( catalog.getLocale( 117 ) ) + 1 );
      sprintf(tstr2,catalog.getLocale(117),"");
    }
    desttext->setText(tstr2);
    _freesafe(tstr2);
    // no need for explicit redraw
    update_desttext=false;
  }
  if(update_filetext==true) {
    if ( ! timerisset( &curtime ) ) gettimeofday( &curtime, NULL );
    difftime2 = diffgtod( &curtime, &filestarttime );
    lastdiff2 = diffgtod( &curtime, &lasttime );

    if ( lastdiff2 >= mintime ) {
      lastrate = (double)( copied_bytes_curfile - lastbytes ) * 1000000.0 / lastdiff2;
      lastbytes = copied_bytes_curfile;
      lasttime = curtime;
    }
    
    MakeLong2NiceStr( copied_bytes_curfile, copied_bytes_curfile_str );

    MakeLong2NiceStr( bytes_curfile, bytes_curfile_str );
    snprintf( tstr, sizeof( tstr ), "%s / %s @ %.2f KB/s",
	      copied_bytes_curfile_str.c_str(),
	      bytes_curfile_str.c_str(),
	      lastrate / 1024.0 );
    tstr[ sizeof( tstr ) - 1 ] = '\0';
    filetext->setText(tstr);
    tw = filetext->getX() + filetext->getWidth() + window->getBorderWidth();
    if ( tw > melw ) {
      melw = tw;
      doresize = true;
    }
    
    if(difftime2>0) {
      rate = (double)copied_bytes_curfile * 1000000.0 / difftime2;
    } else rate=0.0;
    snprintf( tstr, sizeof( tstr ), "%.2f KB/s", rate / 1024.0 );
    tstr[ sizeof( tstr ) - 1 ] = '\0';
    fileavgtext->setText( tstr );

    if(bytes_curfile>0)
      rate = (double)copied_bytes_curfile/(double)bytes_curfile;
    else
      rate=0.0;
    if ( rate < 0.01 )
      rate = 0.0;
    snprintf( tstr, sizeof( tstr ), "%.2f %%", rate * 100.0 );
    tstr[ sizeof( tstr ) - 1 ] = '\0';
    fsb->setText(tstr);
    fsb->resize( w_max( (int)( sbw * rate ), 2 ), fsb->getHeight() );
    update_filetext=false;
  }
  if(update_globtext==true) {
    if ( ! timerisset( &curtime ) ) gettimeofday( &curtime, NULL );
    difftime2 = diffgtod( &curtime, &starttime );
    if(difftime2>0.0) {
      rate = (double)copied_bytes * 1000000.0 / difftime2;
    } else rate=0.0;

    MakeLong2NiceStr( copied_bytes, copied_bytes_str );
    if(slim==true) {
      snprintf( tstr, sizeof( tstr ), "%s @ %.2f KB/s", copied_bytes_str.c_str(), rate / 1024.0 );
      tstr[ sizeof( tstr ) - 1 ] = '\0';
    } else {
      MakeLong2NiceStr( bytes, bytes_str );
      snprintf( tstr, sizeof( tstr ), "%s / %s @ %.2f KB/s", copied_bytes_str.c_str(), bytes_str.c_str(), rate / 1024.0 );
      tstr[ sizeof( tstr ) - 1 ] = '\0';
    }
    
    globtext->setText(tstr);
    tw = globtext->getX() + globtext->getWidth() + window->getBorderWidth();
    if ( tw > melw ) {
      melw = tw;
      doresize = true;
    }
    
    if(slim==false) {
      if(bytes>0)
        rate = (double)copied_bytes/(double)bytes;
      else
        rate=0.0;
      if ( rate < 0.01 )
        rate = 0.0;
      snprintf( tstr, sizeof( tstr ), "%.2f %%", rate * 100.0 );
      tstr[ sizeof( tstr ) - 1 ] = '\0';
      gsb->setText(tstr);
      gsb->resize( w_max( (int)( sbw * rate ), 2 ), gsb->getHeight() );
      
      newglob_percent = (int)( rate * 100.0 );
      if ( newglob_percent != glob_percent ) {
        glob_percent = newglob_percent;
        tstr3 = (char*)_allocsafe( strlen( ( move == true ) ? catalog.getLocale( 299 ) : catalog.getLocaleCom( 8 ) ) +
                                   strlen( " ( %)" ) + 
                                   A_BYTESFORNUMBER( glob_percent ) +
                                   1 );
        if ( ( glob_percent > 0 ) && ( glob_percent <= 100 ) ) {
          sprintf( tstr3, "%s (%d %%)",
                   ( move == true ) ? catalog.getLocale( 299 ) : catalog.getLocaleCom( 8 ),
                   glob_percent );
        } else {
          sprintf( tstr3, "%s",
                   ( move == true ) ? catalog.getLocale( 299 ) : catalog.getLocaleCom( 8 ) );
        }
        window->setTitle( tstr3 );
        _freesafe( tstr3 );
      }
      
      if(rate>0) {
        drtime = difftime2 / rate;
        drtime -= difftime2;
        rtime = (long)( drtime / 1000000.0 );
      } else
        rtime = 700000;  // no rate available => no real time calculatable
      rtime++; // add a second because the last second will shown as 0:00
               // and at least I don't like this
      if ( rtime > 604800 ) // more then 7 days
        sprintf( tstr, "--:--" );
      else 
        sprintf( tstr, "%ld:%02ld", rtime / 60, rtime % 60 );
      timetext->setText(tstr);
      tw = timetext->getX() + timetext->getWidth() + window->getBorderWidth();
      if ( tw > melw ) {
        melw = tw;
        doresize = true;
      }
    }
    update_globtext=false;
  }
  if(update_files2gotext==true) {
    if(slim==true) sprintf(tstr,"%ld",copied_files);
    else sprintf(tstr,"%ld",files-copied_files);
    files2gotext->setText(tstr);
    tw = files2gotext->getX() + files2gotext->getWidth() + window->getBorderWidth();
    if ( tw > melw ) {
      melw = tw;
      doresize = true;
    }
    
    update_files2gotext=false;
  }
  if(update_dirs2gotext==true) {
    if(slim==true) sprintf(tstr,"%ld",copied_dirs);
    else sprintf(tstr,"%ld",dirs-copied_dirs);
    dirs2gotext->setText(tstr);
    tw = dirs2gotext->getX() + dirs2gotext->getWidth() + window->getBorderWidth();
    if ( tw > melw ) {
      melw = tw;
      doresize = true;
    }
    
    update_dirs2gotext=false;
  }
  if ( doresize == true ) {
    // only resize if window is smaller
    if ( window->getWidth() < melw ) {
      window->resize( melw, window->getHeight() );
    }
    // anyway the min size has changed so set it again
    window->setMinSize( melw, bb4->getY() + bb4->getHeight() + 5 +cb->getHeight() + window->getBorderWidth() );
  }

  //window->redraw();
  aguix->Flush();
  /* update complete
     now check for X-messages */
  AGMessage *msg;
  do {
    msg=aguix->GetMessage( window );
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==window->getWindow()) returnvalue=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==cb) returnvalue=1;
          break;
        case AG_SIZECHANGED:
          if ( msg->size.window == window->getWindow() ) {
            bb1->resize( msg->size.neww - 2 * window->getBorderWidth(), bb1->getHeight() );
            sbw = bb1->getWidth() - 2;
            bb2->resize( msg->size.neww - 2 * window->getBorderWidth(), bb2->getHeight() );
            if ( slim == false ) {
              bb3->resize( msg->size.neww - 2 * window->getBorderWidth(), bb3->getHeight() );
            }
            bb4->resize( msg->size.neww - 2 * window->getBorderWidth(), bb4->getHeight() );
            cb->move( msg->size.neww / 2 - cb->getWidth() / 2,
                      msg->size.newh - window->getBorderWidth() - cb->getHeight() );
            
            filenamespace[0] = msg->size.neww - window->getBorderWidth() - sourcetext->getX();
            filenamespace[0] -= aguix->getTextWidth( catalog.getLocale( ( move == true ) ? 594 : 116 ) ) - 2 * aguix->getTextWidth( " " );
            filenamespace[1] = msg->size.neww - window->getBorderWidth() - desttext->getX();
            filenamespace[1] -= aguix->getTextWidth( catalog.getLocale( 117 ) ) - 2 * aguix->getTextWidth( " " );
            update_sourcetext = true;
            update_desttext = true;
  
            // now gsb and fsb are not scaled for new width
            // but next call of redraw they will so this shouldn't hurt anybody
            // but a solution would be to first react for messages and then redraw
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  } while(msg!=NULL);
  
  if ( returnvalue == 1 ) {
    std::string str1;
    int erg;
    
    // user canceled -> request to be sure
    str1 = catalog.getLocale( 11 );
    str1 += "|";
    str1 += catalog.getLocale( 544 );
    erg = window->request( catalog.getLocale( 123 ),
                           catalog.getLocale( 543 ),
                           str1.c_str() );
    if ( erg != 0 ) returnvalue = 0;
  }
  
  return returnvalue;
}

void
CopyOpWin::setmessage( const char *msg, int line )
{
  switch(line) {
    case 0:
      if(msg!=NULL) sourcetext->setText(msg);
      else sourcetext->setText("");
      ismessage[0] = true;
      break;
    case 1:
      if(msg!=NULL) desttext->setText(msg);
      else desttext->setText("");
      ismessage[1] = true;
      break;
  }
}

void
CopyOpWin::forceredraw()
{
  window->redraw();
}

void
CopyOpWin::dec_file_counter(unsigned long f)
{
  files-=f;
  update_files2gotext=true;
}

void
CopyOpWin::dec_dir_counter(unsigned long d)
{
  dirs-=d;
  update_dirs2gotext=true;
}

void
CopyOpWin::dec_byte_counter(loff_t b)
{
  bytes-=b;
  update_globtext=true;
}

void
CopyOpWin::stoptimer()
{
  gettimeofday( &stoptime, NULL );
  timer_stopped = true;
}

void
CopyOpWin::conttimer()
{
  struct timeval curtime;
  long s, us;

  if ( timer_stopped == false ) return;
  
  gettimeofday( &curtime, NULL );
  
  s = curtime.tv_sec - stoptime.tv_sec;
  us = curtime.tv_usec - stoptime.tv_usec;
  if ( us < 0 ) {
    s--;
    us += 1000000;
  }

  starttime.tv_sec += s;
  starttime.tv_usec += us;
  if ( starttime.tv_usec > 1000000 ) {
    starttime.tv_sec++;
    starttime.tv_usec -= 1000000;
  }
  filestarttime.tv_sec += s;
  filestarttime.tv_usec += us;
  if ( filestarttime.tv_usec > 1000000 ) {
    filestarttime.tv_sec++;
    filestarttime.tv_usec -= 1000000;
  }
}

int CopyOpWin::request( const char *title,
                        const char *text,
                        const char *buttons,
                        Requester::request_flags_t flags )
{
  if ( window != NULL ) {
    return window->request( title, text, buttons, flags );
  } else {
    return Worker::getRequester()->request( title, text, buttons, flags );
  }
}

int CopyOpWin::string_request( const char *title,
                               const char *lines,
                               const char *default_str,
                               const char *buttons,
                               char **return_str,
                               Requester::request_flags_t flags )
{
  if ( window != NULL ) {
    return window->string_request( title, lines, default_str, buttons, return_str, flags );
  } else {
    return Worker::getRequester()->string_request( title, lines, default_str, buttons, return_str, flags );
  }
}

void CopyOpWin::file_skipped( loff_t byteSum, bool subCopied )
{
  if ( files > 0 ) files--;
  if ( byteSum > 0 ) bytes -= byteSum;
  if ( subCopied == true ) copied_bytes -= copied_bytes_curfile;
  update_globtext = true;
  update_files2gotext = true;
}

