/* mutex.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2002 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: mutex.cc,v 1.2 2002/03/08 12:42:10 ralf Exp $ */

#include "mutex.h"

MutEx::MutEx()
{
#ifdef WANT_THREADS
  pthread_mutex_init(&mutex,NULL);
  pthread_mutex_init(&dmutex,NULL);
  pthread_cond_init(&condvar,NULL);
  owner=0;
  owncount=0;
#endif
}

MutEx::~MutEx()
{
#ifdef WANT_THREADS
  pthread_mutex_destroy(&mutex);
  pthread_mutex_destroy(&dmutex);
  pthread_cond_destroy(&condvar);
#endif
}

void MutEx::lock()
{
#ifdef WANT_THREADS
  // first lock the data mutex
  pthread_mutex_lock(&dmutex);
  // nw check if we are the owner
  if((owncount>0)&&(pthread_self()==owner)) {
    // then only increase counter
    owncount++;
  } else {
    // lock the real lock
    while(owncount>0) pthread_cond_wait(&condvar,&dmutex);
    pthread_mutex_lock(&mutex);
    // raise counter and store owner
    owncount++;
    owner=pthread_self();
  }
  // now unlock the data mutex
  pthread_mutex_unlock(&dmutex);
#endif
}

void MutEx::unlock()
{
#ifdef WANT_THREADS
  // first lock the data mutex
  pthread_mutex_lock(&dmutex);
  // nw check if we are the owner
  if((owncount>0)&&(pthread_self()==owner)) {
    // decrease counter
    owncount--;
    if(owncount==0) {
      // last unlock so really unlock
      owner=0;
      pthread_mutex_unlock(&mutex);
      pthread_cond_broadcast(&condvar);
    }
  }
  // now unlock the data mutex
  pthread_mutex_unlock(&dmutex);
#endif
}

bool MutEx::trylock()
{
#ifdef WANT_THREADS
  bool returnvalue;
  // first lock the data mutex
  pthread_mutex_lock(&dmutex);
  // nw check if we are the owner
  if((owncount>0)&&(pthread_self()==owner)) {
    // then only increase counter
    owncount++;
    returnvalue=true;
  } else if(owncount==0) {
    // lock the real lock
    pthread_mutex_lock(&mutex);
    // raise counter and store owner
    owncount++;
    owner=pthread_self();
    returnvalue=true;
  } else {
    // locked but not my
    returnvalue=false;
  }
  // now unlock the data mutex
  pthread_mutex_unlock(&dmutex);
  return returnvalue;
#else
  return true;
#endif
}

