/*
 *  WSoundPrefs - WMSound Server Preferences Program
 *
 * Copyright (c) 1998, 1999 Pascal Hofstee
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/utsname.h>
#include <pwd.h>
#include <WINGs.h>
#include <WUtil.h>

#include "WSoundPrefs.h"
#include "SystemInfo.h"

#ifdef __FreeBSD__
 #include <sys/sysctl.h>
 #include <sys/time.h>
#endif

char *getCPUclass();

SystemInfo*
getSystemInfo(WMScreen *scr)
{
	SystemInfo		*sysinfo;
	struct passwd	*user;
	struct utsname	uts;
	Bool			haveUserInfo;
	Bool			haveUname;
	
	sysinfo = wmalloc(sizeof(SystemInfo));
	
	if ( (user = getpwuid(getuid())) == NULL)
		haveUserInfo = False;
	else
		haveUserInfo = True;
	
	if (uname(&uts) < 0)
		haveUname = False;
	else
		haveUname = True;
	

	/* WINGs Version */
	sysinfo->WINGsVersion = WINGS_H_VERSION;
	/* Window Server Version */
	sysinfo->VendorRelease = VendorRelease(WMScreenDisplay(scr));
	/* System Release */
	if (haveUname) {
		sysinfo->SystemName = wstrdup(uts.sysname);
		sysinfo->SystemRelease = wstrdup(uts.release);
	#ifdef __FreeBSD__
		sysinfo->Processor = getCPUclass();
	#else
		sysinfo->Processor = wstrdup(uts.machine);
	#endif
		sysinfo->Host = wstrdup(uts.nodename);
	}
	else {
		sysinfo->SystemName = wstrdup("System Release");
		sysinfo->SystemRelease = wstrdup("Unknown");
		sysinfo->Processor = wstrdup("Unknown");
		sysinfo->Host = wstrdup("Unknown");
	}
	
	if (haveUserInfo) {
		sysinfo->User = wstrdup(user->pw_name);
	}
	else {
		sysinfo->User = wstrdup("Unknown");
	}
	
	return sysinfo;
}

void
destroySystemInfo(SystemInfo *sysinfo)
{
	if (sysinfo) {
		if (sysinfo->SystemName)
			free(sysinfo->SystemName);
		if (sysinfo->SystemRelease)
			free(sysinfo->SystemRelease);
		if (sysinfo->Processor)
			free(sysinfo->Processor);
		if (sysinfo->User)
			free(sysinfo->User);
		if (sysinfo->Host)
			free(sysinfo->Host);

		free(sysinfo);
	}
	else {
		wsyserror(_("Trying to free non-existent SystemInfo struct"));
	}
}


#ifdef __FreeBSD__
char*
getCPUclass()
{
	int		mib[2];
	size_t	size;
	char	*CPUclass;
	
	mib[0] = CTL_HW;
	mib[1] = HW_MODEL;
	
	sysctl(mib, 2, NULL, &size, NULL, 0);
	CPUclass = wmalloc(size);
	sysctl(mib, 2, CPUclass, &size, NULL, 0);
	
	return CPUclass;
}
#endif


char*
getHostname()
{
   struct utsname	uts;

   if (uname(&uts) == 0)
	  return wstrdup(uts.nodename);
   else
	  return wstrdup("Unknown");
}
