/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <cmath>
using namespace std;

#include "SFFT.h"
#include "Inline.h"

#include <wx/object.h>		// Not sure why !
#include <wx/debug.h>

const double STABCOEFF = 0.9999;

CSFFT::CSFFT(int len, int first, int last) :
m_fftLen(len),
m_first(first),
m_last(last),
m_ptr(0)
{
	wxASSERT(m_fftLen > 0);
	wxASSERT(m_first > 0);
	wxASSERT(m_last > first);

	m_cos     = new double[m_fftLen];
	m_sin     = new double[m_fftLen];
	m_history = new complex<double>[m_fftLen];
	m_bins    = new complex<double>[m_fftLen];

	for (int i = 0; i < m_fftLen; i++) {
		double val = double(i) * 2.0 * M_PI / double(m_fftLen);

		m_cos[i] = ::cos(val) * STABCOEFF;
		m_sin[i] = ::sin(val) * STABCOEFF;

		m_history[i] = complex<double>(0.0, 0.0);
		m_bins[i]    = complex<double>(0.0, 0.0);
	}

	m_corr = ::pow(STABCOEFF, double(m_fftLen));
}

CSFFT::~CSFFT()
{
	delete[] m_cos;
	delete[] m_sin;
	delete[] m_history;
	delete[] m_bins;
}

/*
 * Sliding FFT
 */
void CSFFT::process(double in, double* bins)
{
	wxASSERT(bins != NULL);

	/* restore the sample fftlen samples back */
	complex<double> old = m_history[m_ptr] * m_corr;

	/* save the new sample */
	m_history[m_ptr] = complex<double>(in, 0.0);

	/* advance the history pointer */
	m_ptr = (m_ptr + 1) % m_fftLen;

	/* calculate the wanted bins */
	for (int i = m_first; i < m_last; i++) {
		complex<double> z = m_bins[i];

		z -= old;
		z += in;

		double real = z.real() * m_cos[i] - z.imag() * m_sin[i];
		double imag = z.real() * m_sin[i] + z.imag() * m_cos[i];

		m_bins[i] = complex<double>(real, imag);

		bins[i] = ::CABS(m_bins[i]);
	}
}

void CSFFT::process(double in, complex<double>* bins)
{
	wxASSERT(bins != NULL);

	/* restore the sample fftlen samples back */
	complex<double> old = m_history[m_ptr] * m_corr;

	/* save the new sample */
	m_history[m_ptr] = complex<double>(in, 0.0);

	/* advance the history pointer */
	m_ptr = (m_ptr + 1) % m_fftLen;

	/* calculate the wanted bins */
	for (int i = m_first; i < m_last; i++) {
		complex<double> z = m_bins[i];

		z -= old;
		z += in;

		double real = z.real() * m_cos[i] - z.imag() * m_sin[i];
		double imag = z.real() * m_sin[i] + z.imag() * m_cos[i];

		bins[i] = m_bins[i] = complex<double>(real, imag);
	}
}
