/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "FSK441BurstData.h"
#include "FSK441Lookups.h"
#include "FSK441Defs.h"

#include "common/Correlation.h"

#include <cmath>
using namespace std;

const int MAX_SAMPLES_COUNT = 2 * FSK441_SAMPLE_RATE / 5;

CFSK441BurstData::CFSK441BurstData() :
m_data(),
m_dataLen(0),
m_strength(),
m_startTime(0),
m_endTime(0),
m_inBurst(false),
m_DF(0),
m_message(wxEmptyString)
{
	m_data[0] = new double[MAX_SAMPLES_COUNT];
	m_data[1] = new double[MAX_SAMPLES_COUNT];
	m_data[2] = new double[MAX_SAMPLES_COUNT];
	m_data[3] = new double[MAX_SAMPLES_COUNT];
}

CFSK441BurstData::~CFSK441BurstData()
{
	delete[] m_data[0];
	delete[] m_data[1];
	delete[] m_data[2];
	delete[] m_data[3];
}

void CFSK441BurstData::clear()
{
	m_dataLen = 0;

	m_strength.clear();

	m_startTime = 0;
	m_endTime   = 0;

	m_inBurst = false;

	m_DF = 0;

	m_message = wxEmptyString;
}

void CFSK441BurstData::setStartBurst()
{
	m_dataLen = 0;

	m_strength.clear();

	m_startTime = 0;
	m_endTime   = 0;

	m_inBurst = true;

	m_DF = 0;

	m_message = wxEmptyString;
}

void CFSK441BurstData::setDF(int DF)
{
	m_DF = DF;
}

void CFSK441BurstData::setData(int tim, double tone0, double tone1, double tone2, double tone3)
{
	if (!m_inBurst)
		return;

	if (m_startTime == 0)
		m_startTime = tim;

	m_endTime = tim;

	m_strength.addValue(tone0);
	m_strength.addValue(tone1);
	m_strength.addValue(tone2);
	m_strength.addValue(tone3);

	if (m_dataLen >= MAX_SAMPLES_COUNT)
		return;

	m_data[0][m_dataLen] = tone0;
	m_data[1][m_dataLen] = tone1;
	m_data[2][m_dataLen] = tone2;
	m_data[3][m_dataLen] = tone3;

	m_dataLen++;
}

bool CFSK441BurstData::getInBurst() const
{
	return m_inBurst;
}

int CFSK441BurstData::getStartTime() const
{
	return m_startTime;
}

int CFSK441BurstData::getEndTime() const
{
	return m_endTime;
}

int CFSK441BurstData::getLength() const
{
	return m_endTime - m_startTime;
}

int CFSK441BurstData::getDF() const
{
	return m_DF;
}

bool CFSK441BurstData::processBurst()
{
	if (!m_inBurst)
		return false;

	double max = -999.0;
	int offset = -1;

	for (int i = 0; i < (m_dataLen - 10); i++) {
		int j = i + 5;
		int k = i + 10;

		CCorrelation corr;

		corr.addProduct(m_data[0][i], 3.0);
		corr.addProduct(m_data[1][i], -1.0);
		corr.addProduct(m_data[2][i], -1.0);
		corr.addProduct(m_data[3][i], -1.0);

		corr.addProduct(m_data[0][j], -1.0);
		corr.addProduct(m_data[1][j], -1.0);
		corr.addProduct(m_data[2][j], -1.0);
		corr.addProduct(m_data[3][j], 3.0);

		corr.addProduct(m_data[0][k], -1.0);
		corr.addProduct(m_data[1][k], -1.0);
		corr.addProduct(m_data[2][k], -1.0);
		corr.addProduct(m_data[3][k], 3.0);

		double val;
		corr.getRawValue(val);

		if (val > max) {
			max    = val;
			offset = i;
		}
	}

	if (max <= 0.0)
		return false;

	offset %= 15;

	CFSK441Lookups lookups;

	for (int i = offset; i < (m_dataLen - 10); i += 15) {
		int j = i + 5;
		int k = i + 10;

		int t[3];
		t[0] = 0;
		t[1] = 0;
		t[2] = 0;

		double val[3];
		val[0] = m_data[0][i];
		val[1] = m_data[0][j];
		val[2] = m_data[0][k];

		for (int n = 1; n < 4; n++) {
			if (m_data[n][i] > val[0]) {
				val[0] = m_data[n][i];
				t[0]   = n;
			}
			if (m_data[n][j] > val[1]) {
				val[1] = m_data[n][j];
				t[1]   = n;
			}
			if (m_data[n][k] > val[2]) {
				val[2] = m_data[n][k];
				t[2]   = n;
			}
		}

		wxString text;
		bool     ret;

		ret = lookups.lookupMultiTones(t[0], t[1], t[2], text);

		if (ret)
			m_message.Append(text);
		else
			m_message.Append(wxT("*"));
	}

	return m_message.Find(wxT(' ')) != -1;
}

wxString CFSK441BurstData::getMessage() const
{
	return m_message;
}

int CFSK441BurstData::getStrength(double noise) const
{
	wxASSERT(noise != 0.0);

	double strength = 99.0;

	if (m_strength.getCount() > 0)
		strength = 10.0 * ::log10(m_strength.getMaximum() / noise);

	return int(strength + 0.5);
}
