/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
  FmErrors.c
  
  (c) Simon Marlow 1990-1993
  (c) Albert Graef 1994

  modified 2005,2006 by Bernhard R. Link (see Changelog)

  Error handling routines
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <xfmconfig.h>

#include <errno.h>
#include <stdlib.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw3d/Form.h>
#include <X11/Xaw3d/Label.h>
#include <X11/Xaw3d/Box.h>

#include "global.h"

#include "Am.h"
#include "Fm.h"

#define LABEL_WIDTH 300

/*-----------------------------------------------------------------------------
  STAIC DATA
-----------------------------------------------------------------------------*/

typedef struct {
  Widget shell, label1, label2;
} ErrorPopupRec;

static ErrorPopupRec errors;

static Boolean error_flag = True;

/*-----------------------------------------------------------------------------
  Widget Argument Lists
-----------------------------------------------------------------------------*/

static Arg shell_args[] = {
  { XtNtitle, (XtArgVal) "Error" },
};

static Arg *form_args = NULL;

static Arg label_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNlabel, (XtArgVal) NULL },
  { XtNwidth, (XtArgVal) LABEL_WIDTH },
  { XtNfont, (XtArgVal) NULL },
  { XtNresize, False },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainTop },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainRight }
};

static Arg bitmap_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNbitmap, (XtArgVal) NULL },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainTop },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainLeft }
};

static Arg button_box_args[] = {
  { XtNfromHoriz, (XtArgVal) NULL },
  { XtNfromVert, (XtArgVal) NULL },
  { XtNtop, XtChainTop },
  { XtNbottom, XtChainTop },
  { XtNleft, XtChainLeft },
  { XtNright, XtChainLeft }
};

/*-----------------------------------------------------------------------------
  PRIVATE FUNCTIONS
-----------------------------------------------------------------------------*/

static void errorOkCb(UNUSED(Widget w), UNUSED(void *client_data),
                      UNUSED(void *call_data))
{
  XtPopdown(errors.shell);
  error_flag = True;
}

/*-----------------------------------------------------------------------------
  Button Data
-----------------------------------------------------------------------------*/

static const ButtonRec error_buttons[] = {
  { "cancel", "Cancel", errorOkCb }
};

/*-----------------------------------------------------------------------------
  PUBLIC FUNCTIONS
-----------------------------------------------------------------------------*/

void createErrorPopup(void)
{
  Widget form, bitmap, button_box;

  /* create shell */
  errors.shell = XtCreatePopupShell("error", transientShellWidgetClass,
				   aw.shell, shell_args, XtNumber(shell_args));

  /* create outer form */
  form = XtCreateManagedWidget("form", formWidgetClass, errors.shell,
				      form_args, XtNumber(form_args) );

  /* create  bitmap */
  bitmap_args[2].value = (XtArgVal) bm[EXCL_BM];
  bitmap = XtCreateManagedWidget("bitmap", labelWidgetClass, form, bitmap_args,
					XtNumber(bitmap_args) );

  /* create label 1 */
  label_args[0].value = (XtArgVal) bitmap;
  label_args[4].value = (XtArgVal) resources.label_font;
  errors.label1 = XtCreateManagedWidget("label1",labelWidgetClass, form,
				       label_args, XtNumber(label_args) );

  /* create label 2 */
  label_args[1].value = (XtArgVal) errors.label1;
  errors.label2 = XtCreateManagedWidget("label2",labelWidgetClass, form,
				       label_args, XtNumber(label_args) );

  /* create button box */
  button_box_args[1].value = (XtArgVal) bitmap;
  button_box = XtCreateManagedWidget("button box", boxWidgetClass, form,
			     button_box_args, XtNumber(button_box_args) );
  createButtons(error_buttons, XtNumber(error_buttons), button_box,
		NULL);

#ifdef ENHANCE_POP_ACCEL
  XtInstallAllAccelerators(form,form);
  XtInstallAllAccelerators(errors.shell,form);
#endif

  XtRealizeWidget(errors.shell);
  setWMProps(errors.shell);
}

/*---------------------------------------------------------------------------*/

static void untilOk(void)
{
  XEvent e;

  error_flag = False;

  do {
    XtAppNextEvent(app_context, &e);
    XtDispatchEvent(&e);
  } while (!error_flag);
}

/*---------------------------------------------------------------------------*/

void sysError(const char *string1)
{
  if (!error_flag) /* recursive errors are possible - we just ignore them */
    return;
  XtVaSetValues(errors.label1, XtNlabel, string1, NULL);
  XtVaSetValues(errors.label2, XtNlabel, strerror(errno), NULL);
  popupByCursor(errors.shell, XtGrabExclusive);
  untilOk();
}

/*---------------------------------------------------------------------------*/

void error(const char *string1, const char *string2)
{
  if (!error_flag)
    return;
  XtVaSetValues(errors.label1, XtNlabel, string1, NULL);
  XtVaSetValues(errors.label2, XtNlabel, string2, NULL);
  popupByCursor(errors.shell, XtGrabExclusive);
  untilOk();
}

/*---------------------------------------------------------------------------*/

void abortXfm(const char *message)
{
  fprintf(stderr,"%s: %s -- abort\n", progname, message);
  exit(1);
}

