/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#ifndef __XMMS_KDEDB_H__
#define __XMMS_KDEDB_H__

#include <string>

#if defined __GNUC__ && (__GNUC__ > 2)
  #include <ext/hash_set>
  using namespace __gnu_cxx;
#else
  #include <hash_set>
#endif

#include <kconfig.h>

#include <qdatetime.h>
#include <qhbuttongroup.h>
#include <qevent.h>
#include <qprogressbar.h>
#include <qfiledialog.h>
#include <qwidget.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qpoint.h>
#include <qpixmap.h>
#include <qsize.h>
#include <qradiobutton.h>
#include <qthread.h>
#include <qvbuttongroup.h>
#include <qsplitter.h>
#include <qcombobox.h>
#include <qwaitcondition.h>
#include <qmutex.h>
#include <qptrqueue.h>

#include "sqlite/sqlite.h"

#include "playerinterface.h"

using namespace std;

class StatusLabel;
class ProgressLabel;

class InsertThread;
class SearchThread;
class SearchEvent;


class XmmsKdeDB : public QObject {

  Q_OBJECT

public:
  XmmsKdeDB(KConfig *conf, QPixmap *icon);

  ~XmmsKdeDB();

  bool connectDB();

  void disconnectDB();

  bool isConnectedDB();

  bool isEnabled() {
    return enable;
  }

  void disableDB() {
    enable = false;
    disconnectDB();
  }

  void enableDB() {
    enable = true;
  }

  QWidget* getConfigurationWidget(QWidget *parent);
  void writeConfig();

  bool getSync() {

    return sync;
  }

  sqlite *db;
  sqlite *querydb;

signals:

  void statusChanged(QString status);

public slots:

  void stopInsertThread();
  void updateDatabase();

private slots:

  void addPathToList();
  void removePathFromList();

  void configurationChanged();

private:

  void readConfig();

  bool enable;
  bool connected;

  QString name;

  KConfig *config;

  QCheckBox *enableBox;
  QListBox *pathListBox;
  QStringList pathList;
  QFrame *statusFrame;
  QPixmap *icon;
  InsertThread *insertThread;
  bool sync;
  QDateTime updated;
};


class XmmsKdeDBQuery : public QSplitter {

  Q_OBJECT

public:

  XmmsKdeDBQuery(XmmsKdeDB *datab, PlayerInterface *p,
		 QPixmap *icon, KConfig *conf);

  ~XmmsKdeDBQuery();
 void customEvent(QCustomEvent *e);

  void setPlayer(PlayerInterface *p);
  void popup();

  QWidget* getConfigurationWidget(QWidget *parent);
  void writeConfig();

  int getPopup() {

    return pop;
  }

signals:

public slots:

private slots:

  void firstBoxChanged();
  void secondBoxChanged();

  void secondDClicked(QListBoxItem *item);

  void firstComboChanged(int index);
  void secondComboChanged(int index);

  void activateSearch();
  void firstFilterChanged(const QString &text);
  void secondFilterChanged(const QString &text);
  void resultFilterChanged(const QString &text);

  void play(int index);
  void setPlayList();
  void addPlayList();

  void popupChanged(int index);
  void clickChanged(int index);

private:

  void readConfig();

  QString getResultQuery();
  QString getSecondQuery();
  QString getFirstQuery();

  QString prepareList(QStringList list);

  XmmsKdeDB *db;
  PlayerInterface *player;

  QListBox *resultBox;
  QListBox *firstBox;
  QListBox *secondBox;

  QComboBox *firstCombo;
  QComboBox *secondCombo;

  QLineEdit *firstFilterEdit;
  QLineEdit *secondFilterEdit;
  QLineEdit *resultFilterEdit;

  KConfig *config;
  QVButtonGroup *queryGroup;
  QVButtonGroup *clickGroup;  

  QPoint framePos;
  QSize frameSize;
  int pop, click, numActiveSearch;
  enum EFilterMode {ARTIST=0, ALBUM, YEAR, GENRE};

  EFilterMode firstMode;
  EFilterMode secondMode;

  QStringList firstList;
  QStringList secondList;

  QString allText;
  QStringList modeTexts;
  QStringList searchTexts;

  SearchThread *searchThread;
};

class QueryItem : public QListBoxText {


public:
  QueryItem(QString text, QString file);
  ~QueryItem();

  QString getFile();


private:

  QString filename;

};


//! A class to run a thread which executes queries.
/*!
  SearchThread is meant to be running as long as the application is running.
	It provides a method "addSearch" for adding queries that it should execute.
	Results are returned by posting an SearchEvent to the XmmsKdeDBQuery object given in the constuctor.
*/
class SearchThread : public QThread {

 public:
   //! create a new SearchThread.
   /*!
   \param r		receiver of posted SearchEvents.
   \param d		database object needed for queries.
   */
   SearchThread(XmmsKdeDBQuery *r, XmmsKdeDB *d) : db(d), receiver(r) {
     shouldExit = false;
   }

   //! The main method of the thread.
   /*!
   This method will run in a loop processing searches until it is killed.
   */
   void run();

   //! Terminate the thread in a nice manner.
   void kill() {
      shouldExit = true;
      waitCondition.wakeAll();
   }

   //! Add a search to be processed by the thread.
   /*!
   SearchEvent is placed in a mutex-controlled queue and the thread is (optionally) woken to process it.
   \param e				SearchEvent to be added.
   \param activate	True if thread should be interrupted from its beauty sleep.
   */
   void addSearch(SearchEvent *e, bool activate = true) {
      mutex.lock();
      queue.enqueue(e);
      mutex.unlock();
      if(activate)
	waitCondition.wakeAll();
   }

   
   //! return the next SearchEvent from the queue.
   SearchEvent *getSearch() {
     mutex.lock();
     SearchEvent *temp = queue.dequeue();
     mutex.unlock();
     return temp;
   }

   //! Carry out the requested search.
   void doSearch(SearchEvent *search);

   //! Queue of SearchEvents to be processed.
   QPtrQueue<SearchEvent> queue;
   
   //! keeps threads from messing with the queue at the same time.
   QMutex mutex;

   //! set by method kill to inform thread that it should exit.
   bool shouldExit;

   //! database object.
   XmmsKdeDB *db;

   //! receiver of the resulting SearchEvent.
   XmmsKdeDBQuery *receiver;

   //! waitcondition to allow the thread to sleep when it has noting to do.
   QWaitCondition waitCondition;
};



//! A class to insert items into a database in a new thread.
/*!
  InsertThread is used to update the contents of the database.
  A new thread is created to allow the GUI to be updated as this operation
  goes on.
  InsertThread reads the contents of the database and compares them to the
  given pathnames. When there are differences between the entries in the
  database and the filesystem, the tags of the files are scanned and inserted
  into the database.
*/
class InsertThread : public QThread {

 public:
  //! create a new InsertThread.
  /*!
    \param database       the sqlite database.
    \param directories    the pathnames to process.
    \param statusLabel    the status label in the GUI to update accordingly.
    \param fileLabel      the label in the GUI that shows the current filename.
    \param progressLabel  the progressbar in the GUI.
  */
  InsertThread(sqlite *database,
	       QStringList directories,
	       StatusLabel *statusLabel,
	       StatusLabel *fileLabel,
	       ProgressLabel *progressLabel,
	       QDateTime updated);

  //! the main method of the thread.
  /*!
    The database is updated to contain the same entries as the processed
    paths.
  */
  virtual void run();

 private:

  //! update the database
  /*!
    First, the database entries are read into a hashtable. Then, the filenames
    in the given paths are read into another hashtable.
    Then, the files that are no longer in the filesystem but in the database
    are deleted from the database.
    At last, the files that are in the filesystem but not in the database
    are added.
    \param dir the pathnames to process.
  */
  void updateDatabase(QStringList dir);

  //! inserts the tag of a file into the database.
  /*!
    \param file    the file to add to the database.
  */
  void insertIntoDatabase(QString file);

  //! delete a file entry from the database.
  /*!
    \param file  the file to delete.
  */
  void deleteFromDatabase(QString file);

  //! build a vector with the filenames of all .ogg and .mp3 files in the path.
  /*!
    \param path    the path to process.
    \param vector  the vector to build from the path.
  */
  void addPathToVector(QString path, vector<QString *> *vector);

  //! the sqlite database object.
  sqlite *db;

  //! the directories to add to the database.
  QStringList dir;

  //! the label which shows the status of this InsertThread.
  StatusLabel *statusLabel;

  //! the label which shows the file that is currently processed.
  StatusLabel *fileLabel;

  //! the progessbar.
  ProgressLabel *progressLabel;

  //! the last time the database was updated
  QDateTime updated;
};

enum	SearchType {TO_FIRSTBOX=60045, TO_FIRSTBOX_FOCUS, TO_SECONDBOX, TO_RESULTBOX };

class SearchEvent: public QCustomEvent {

 public:
  SearchEvent(SearchType t): QCustomEvent(t) {}
  SearchEvent(SearchType t, QString q, QString f=QString::null) : QCustomEvent(t), query(q), focusText(f) {}
  QString query;
  QString focusText;
  QPtrQueue<QListBoxItem> items;

};

class StatusEvent: public QCustomEvent {

 public:
  StatusEvent(QString m): QCustomEvent(60042), s(m) {}

  QString message() const {
    return s;
  }
 private:
  QString s;
};

class ProgressStepEvent: public QCustomEvent {

 public:
  ProgressStepEvent(int p): QCustomEvent(60043) {
    prog = p;
  }

  int progress() const {
    return prog;
  }
 private:
  int prog;
};

class ProgressTotalEvent: public QCustomEvent {

 public:
  ProgressTotalEvent(int p): QCustomEvent(60044) {
    prog = p;
  }

  int progress() const {
    return prog;
  }
 private:
  int prog;
};

class StatusLabel: public QLabel {

 public:
  StatusLabel(char *c, QWidget *parent, int len): QLabel(c, parent) {
    length = len;
  }

  void customEvent(QCustomEvent *e) {
    if (e->type() == 60042) {
      StatusEvent *s = (StatusEvent *) e;
      QString st = s->message();
      st.truncate(length);
      setText(st);
    }
  }

 private:
  int length;

};

class ProgressLabel: public QProgressBar {

 public:
  ProgressLabel(QWidget *parent): QProgressBar(0, parent) {

    //setTotalSteps(0);
  }

  void customEvent(QCustomEvent *e) {
    if (e->type() == 60043) {
      // progress step
      ProgressStepEvent *s = (ProgressStepEvent *) e;
      setProgress(s->progress());

    } else if (e->type() == 60044) {
      // progress total
      ProgressTotalEvent *s = (ProgressTotalEvent *) e;
      setTotalSteps(s->progress());
    }
  }

 private:
  int length;

};




#endif



