/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <ctype.h>

#include "singit_macros.h"
#include "singit_macros_private.h"

#include "singit_song.h"
#include "singit_song_private.h"

#include "singit_sha.h"

#include "singit_tools.h"

inline gint get_timetag_type(gchar* tag)
{
	if ((tag[0] == '[') && (isdigit(tag[1])) && (isdigit(tag[2])) &&
		(tag[3] == ':') && (isdigit(tag[4])) && (isdigit(tag[5]))) {
		if (tag[6] == ']') { return 1; }
		if (tag[6] == ':') {
			if ((isdigit(tag[7])) && (isdigit(tag[8])) && (isdigit(tag[9]))
				&& (tag[10] == ']')) { return 2; }
		}
	}
	return 0;
}

gint l_song_get_timetag_type(gchar* tag)
{
	guint length = strlen(tag);
	if (length < 7) { return 0; }
	if ((tag[6] == ':') && (length < 11)) { return 0; }
	return get_timetag_type(tag);
}

inline gboolean extrakt_timetag_information(gchar* tag, guint* time)
{
	gboolean result = FALSE;
	switch (get_timetag_type(tag)) {
		case 1:
			tag[3] = '\0';
			tag[6] = '\0';
			*time = (atoi(&tag[1]) * 60 + atoi(&tag[4])) * 1000;
			result = TRUE;
			break;
		case 2:
			tag[3] = '\0';
			tag[6] = '\0';
			tag[10] = '\0';
			*time = (atoi(&tag[1]) * 60 + atoi(&tag[4])) * 1000 + atoi(&tag[7]);
			result = TRUE;
			break;
	}
	return result;
}

inline gchar* extract_token(LSong* song, gchar* line, guint lineNr)
{
	const gchar old_tags[4][5] = { "[ar:", "[ti:", "[la:", "[by:" };
	gchar *pos, *found = NULL, *tmp = NULL;
	LToken *token = NULL;
	gint time = 0, outlen = 0, tokenlen = 0;
	guint stringlen = strlen(line);
	gchar *new_line = NULL, *new_pos = NULL;

	if (stringlen <= 0) { return g_strdup(line); }
	pos = strstr(line, "[");
	if (pos) {
		new_line = g_new(gchar, stringlen);
		new_pos = new_line;
		tokenlen = pos - line;
		if (tokenlen > 0) {
			memcpy(new_pos, pos, tokenlen);
			new_pos += tokenlen;
		}
	}
	else { return g_strdup(line); }

	while (pos) {
		if (found) {
			tokenlen = pos - found - 1;
			if (tokenlen > 0) {
				memcpy(new_pos, (found+1), tokenlen);
				new_pos += tokenlen;
			}
		}
		found = strstr (pos, "]");
		tokenlen = found - pos;
		switch (tokenlen) {
		case 6:
		case 10:
			if (extrakt_timetag_information(pos, &time) == TRUE) {
				#ifdef CODEDEBUG
				DEBUG(("."), 9);
				#endif
				tokenlen++;
				outlen += tokenlen;
				pos += tokenlen;
				token = g_malloc(sizeof(LToken));
				token->time = time;
				token->line = lineNr;
				token->pos = pos - line - outlen;
				song->first_token = g_list_append(song->first_token, token);
				tmp = pos;
				pos = strstr(tmp, "[");
				break;
			}
		default:
			if (found == NULL) {
				tokenlen = strlen(pos);
				memcpy(new_pos, pos, tokenlen);
				new_pos += tokenlen;
				new_pos[0] = '\0';
				return new_line;
			}
			else {
				if ((found - pos > 5) && (pos == line)) {
					tokenlen = 0;
					while (tokenlen < 4) {
						if (strncmp(&old_tags[tokenlen][0], pos, 4) == 0) {
#							ifdef CODEDEBUG
							DEBUG(("o"), 9);
#							endif
							/* switch (time) {
							default:
							} */
							tokenlen = 4;
							return NULL;
						}
						tokenlen++;
					}
				}
				memcpy(new_pos, pos, found - pos);
				new_pos += (found - pos);
				tmp = found+1;
				pos = strstr(found, "[");
			}
		}
	}
	tokenlen = strlen(tmp);
	if (tokenlen > 0) {
		memcpy(new_pos, tmp, tokenlen);
		new_pos += tokenlen;
	}
	new_pos[0] = '\0';
	return new_line;
}

gboolean l_song_read_text_stream(LSong *song, gchar *buffer)
{
	GSList *string_list = NULL, *slist;

  	gchar *s;
  	guint n = 0;
	gchar *delimiter = "\n";
	guint len;
	gchar *new_string, *line;
	guint delimiter_len = -1;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_read_text_stream]\n"), 9);
	DEBUG(("        'l' = line / '.' = tag / 'o' = old special tag\n"), 9);
#	endif

	if (!buffer || !song) { return FALSE; }

  	s = strstr (buffer, delimiter);
  	if (s) {
      		delimiter_len = strlen (delimiter);
		len = s - buffer;
		if (buffer[len-1] == 13) {
			song->delimiter = g_strdup("    ");
			song->delimiter[0] = 13;
			song->delimiter[1] = '\n';
			song->delimiter[2] = '\0';
		}
		do {
#			ifdef CODEDEBUG
			DEBUG(("l"), 9);
#			endif
			len = s - buffer;
			if (buffer[len-1] == 13) { len--; }
			new_string = g_new (gchar, len + 1);
			strncpy (new_string, buffer, len);
			new_string[len] = 0;
			line = extract_token(song, g_strstrip(new_string), n);
			if (line != NULL) {
				string_list = g_slist_prepend(string_list, line);
				n++;
			}
			g_free(new_string);
			buffer = s + delimiter_len;
			s = strstr (buffer, delimiter);
		}
      		while (s);
	}
	if (*buffer) {
		len = strlen(buffer);
#		ifdef CODEDEBUG
		DEBUG(("l"), 9);
#		endif
		if (buffer[len-1] == 13) {
			len--;
			new_string = g_new (gchar, len + 1);
			strncpy (new_string, buffer, len);
			new_string[len] = 0;
			string_list = g_slist_prepend
				(string_list, g_strdup(extract_token(song, new_string, n)));
			g_free(new_string);
		}
		else
			{ string_list = g_slist_prepend (string_list, g_strdup (extract_token(song, buffer, n))); }
		n++;
	}

	song->lyric_lines = n;
	song->lyrics = g_new (gchar*, n+1);

	song->lyrics[n--] = NULL;
	for (slist = string_list; slist; slist = slist->next)
		song->lyrics[n--] = slist->data;
	g_slist_free (string_list);

	if (song->first_token != NULL) {
		song->first_token = g_list_sort(song->first_token, compare_token_by_time);
		song->first_token = g_list_first(song->first_token);
		song->last_token = g_list_last(song->first_token);

#		ifdef CODEDEBUG
		DEBUG(("\nTagTime-Gap: %.2i:%.2i - %.2i:%.2i\n", tTime(song->first_token) / 60000,
			tTime(song->first_token) % 60000 / 1000, tTime(song->last_token) / 60000,
			tTime(song->last_token) % 60000 / 1000), 9);
#		endif
	}
	else
		{ song->active_token = song->last_token = song->first_token; }

	song->lyric_type = lt_text;

#	ifdef CODEDEBUG
	DEBUG(("\n"), 9);
#	endif
	return TRUE;
}

gboolean l_song_write_text_stream(LSong *song, gchar **buffer, gboolean extended)
{
	gchar **stream, *tmp = NULL, *connected = NULL;
	GList *item;
	gchar timeToken[12];
	gint i, offset = 0, last_line = -1;

#	ifdef CODEDEBUG
	DEBUG(("singit_song.c [l_song_write_text_stream]\n"), 9);
#	endif

	if (!song || !buffer) { return FALSE; }
	if (!l_song_text_found(song)) { return FALSE; }

	timeToken[7] = '\0';
	timeToken[11] = '\0';
	stream = g_new(gchar*, song->lyric_lines + 1);
	stream[song->lyric_lines] = NULL;
	for (i = 0; i < song->lyric_lines; i++)
		{ stream[i] = g_strdup(song->lyrics[i]); }
	item = song->first_token;
	while (item) {
		if (extended) {
			sprintf(timeToken, "[%.2i:%.2i:%.3i]", tTime(item) / 60000, (tTime(item) / 1000) % 60, tTime(item) % 1000);
			if (tLine(item) != last_line) { offset = 0; }
			else { offset += 11; }
		}
		else {
			sprintf(timeToken, "[%.2i:%.2i]", tTime(item) / 60000, (tTime(item) / 1000) % 60);
			if (tLine(item) != last_line) { offset = 0; }
			else { offset += 7; }
		}
		last_line = tLine(item);
		tmp = stream[tLine(item)];
		connected = tools_insert_string(stream[tLine(item)], &timeToken[0], tPos(item) + offset);
		if (connected) {
			stream[tLine(item)] = connected;
			g_free(tmp);
		}
		item = g_list_next(item);
	}

	*buffer = g_strjoinv("\n", stream);
	return TRUE;
}

gboolean l_song_save_to_text_file(LSong *song, gchar *filename, gboolean extended)
{
	FILE *file;
        gchar *buffer, *useFilename;

#	ifdef CODEDEBUG
	DEBUG(("singit_song_text.c [l_song_save_to_text_file]\n"), 9);
#	endif

	if (!song) { return FALSE; }
	if (!l_song_text_found(song)) { return FALSE; }

	if (filename) { useFilename = filename; }
	else { useFilename = song->lyric_filename; }

        if (!(file = fopen(useFilename, "w"))) { return FALSE; }

        l_song_write_text_stream(song, &buffer, extended);
        if (fwrite(buffer, 1, strlen(buffer), file) != strlen(buffer))
        {
                g_free(buffer);
                fclose(file);
                return FALSE;
        }
        fclose(file);
	g_free(buffer);

	if (!song->lyric_filename) { song->lyric_filename = g_strdup(filename); }

	return TRUE;
}

gboolean l_song_load_from_text_file(LSong *song, gchar *filename)
{
	FILE *file;
        gchar *buffer;
        struct stat stats;
	gboolean result;

	SHA_INFO sha;

#	ifdef CODEDEBUG
	DEBUG(("singit_song_text.c [l_song_load_from_text_file] : "), 9);
#	endif

	if (!l_song_attach(song)) { return FALSE; }

	if ((lstat(filename, &stats) == -1) ||
		(!(file = fopen(filename, "r"))))
	{
		l_song_detach(song, TRUE);
#		ifdef CODEDEBUG
		DEBUG(("Unable to read file\n"), 9);
#		endif
		return FALSE;
	}

	if ((stats.st_size > 100000) || (stats.st_size < 1)) {
		fclose(file);
		l_song_detach(song, TRUE);
#		ifdef CODEDEBUG
		DEBUG(("Wrong size (%i)\n", (guint) stats.st_size), 9);
#		endif
		return FALSE;
	}

	buffer = g_malloc(stats.st_size + 1);
	if (fread(buffer, 1, stats.st_size, file) != stats.st_size)
	{
		g_free(buffer);
		fclose(file);
		l_song_detach(song, TRUE);
#		ifdef CODEDEBUG
		DEBUG(("Buffered read failed\n"), 9);
#		endif
		return FALSE;
	}
	fclose(file);
	buffer[stats.st_size] = '\0';

	sha_mem(song->sha_digest, &sha, buffer, stats.st_size);

	result = l_song_read_text_stream(song, buffer);
	if (result) {
		song->file_size = stats.st_size;
		song->lyric_filename = g_strdup(filename);
		song->lyric_type = lt_text;
	}
	else { song->lyric_type = lt_none; }

	g_free(buffer);

	l_song_detach(song, TRUE);
	return result;
}
