
#include "DC_VideoDevice.h"

#include "PixPort.h"

#if EG_MAC
#include <Files.h>
#include <QuickDraw.h>
#include <Video.h>
#include <Devices.h>
#include <Displays.h>
#include <Gestalt.h>
#endif

#if USE_DIRECTX
#include <ddraw.h>
#include "multimon.h"
#endif


DC_VideoDevice::DC_VideoDevice( bool inForceFullRedraws ) :
	VideoDevice( inForceFullRedraws ) {

	mDepth = 0;
	mIsFullscreen = false;
	mSetEntriesCompat = false;

	#if EG_MAC
	mFS_GDevice = nil;

		#if TARGET_API_MAC_CARBON
			mSetEntriesCompat = true;
		#else
			long vers;
			if ( ::Gestalt( gestaltSystemVersion, &vers ) == noErr )
				if ( vers <= 0x906 )
					mSetEntriesCompat = true;
		#endif

	#endif
}



void DC_VideoDevice::SetPort( PP_GrafPtr inPortDC, Rect& inLocalRect, bool inIsFullscreen ) {

	mDC = inPortDC;
	mOutputRect = inLocalRect;
	mDepth = GetPortDepth( mDC );
	mIsFullscreen = inIsFullscreen;

#if EG_MAC
	long dispID = GetDisplayID( mDC );
	::DMGetGDeviceByDisplayID( dispID, &mFS_GDevice, false );
#endif
}

void DC_VideoDevice::TransferBits( PixPort& inFrame, Rect& inSrce, Rect& inDest ) {

	if ( mDC )
		inFrame.CopyBits( mDC, &inSrce, &inDest );
}


int DC_VideoDevice::GetPortDepth( PP_GrafPtr inPortDC ) {
	int depth = 0;

	if ( inPortDC ) {

		#if EG_MAC
		#if TARGET_API_MAC_CARBON
			PixMapHandle pixMapHandle = GetPortPixMap( inPortDC );
			depth = (**pixMapHandle).pixelSize;
		#else
			depth = (**((CGrafPtr) inPortDC ) -> portPixMap).pixelSize;
		#endif
		#endif

		#if EG_WIN
		depth = ::GetDeviceCaps( inPortDC, BITSPIXEL );
		#endif
	}

	return depth;
}

void DC_VideoDevice::SetDeviceEntries( PixPort* inFrame ) {

	if ( inFrame ) {

		#if EG_MAC
		if ( IsFullscreen() && mDepth == 8 && inFrame -> GetDepth() == 8 ) {

			// Get a ptr to the frame's port color data...
			CTabHandle offscreenTable = (**inFrame -> GetPixMap()).pmTable;
			if ( offscreenTable ) {

				// Does the fullscreen display device need an explicit palette change?
				if ( inFrame -> GetCurPaletteSeed() != mCurPaletteSeed ) {
					GDHandle GDev = ( mFS_GDevice ) ? mFS_GDevice : ::GetGDevice();

					if ( GDev && *offscreenTable ) {

						// Set the grafx environment
						::SetGWorld( (CGrafPtr) mDC, GDev );

						if ( mSetEntriesCompat ) {

							// The fullscreen device is in 8 bits, so we need to change its palette to our source's palette
							// Note how the fullscreen palette stays the sys palette (ie, this never gets called) if the source pixmap isn't 8 bit.
							// You'd think we'd pass 256, but the apple docs say, "...values are zero based so setting 3 entries means you pass a 2 in the count parameter."
							::SetEntries( 0, 255, (**offscreenTable).ctTable );  }

						// If we're running 9.1 or later...
						#if ! TARGET_API_MAC_CARBON
						else {


							// Set the device palette to the offscreen port's palette
							// We don't use SetEntries() because MacOS 9.1 essentially forces entry 0 and 255 to be black and white
							//    no matter what we request, totally screwing up apps that need all 256 colors. The old code lies
							//    after this fcn commented out.
							VDSetEntryRecord	colorRange;
							CntrlParam			colorParam;

							colorRange.csTable = (**offscreenTable).ctTable;
							colorRange.csStart = 0;
							colorRange.csCount = 255;

							*((void**) &colorParam.csParam) = &colorRange;
							colorParam.ioCompletion = 0;
							colorParam.ioCRefNum = (*GDev)->gdRefNum;
							colorParam.csCode = 3;

							PBControl( (ParamBlockRec*) &colorParam, 0 );
						}
						#endif
					}

					// Avoid setting the same palette on a device twice
					mCurPaletteSeed = inFrame -> GetCurPaletteSeed();
				}

				// This prevents MacOS from remapping the pixel values in our offscreen GWorld during CopyBits, thinking the colors in our GWorld
				// index to the standard system colors.  It's a hack, but then again, so is MacOS.
				#if ! TARGET_API_MAC_CARBON
				(**offscreenTable).ctSeed = (**(*((CGrafPort*) mDC) -> portPixMap) -> pmTable).ctSeed;
				#endif
			}
		}
		#endif

		#if USE_DIRECTX
		// For Win32, the equivilent of the above block happens in WindowDevice::BeginFrame()
		#endif
	}
}




// Multimonitor support fcns or windows...
#if USE_DIRECTX
typedef struct {
   HMONITOR		mMonitor;
   int  		mNumRemain;
   GUID 		mDeviceGUID;
} DD_ENUM, *PDD_ENUM;


BOOL WINAPI DdEnumCb	(GUID FAR *lpGUID, LPSTR, LPSTR lpDriverName, PVOID lpContext );
BOOL WINAPI DdEnumCbEx	(GUID FAR *lpGUID, LPSTR, LPSTR lpDriverName, PVOID lpContext, HMONITOR inMonitor );


BOOL WINAPI DdEnumCb(GUID FAR *lpGUID, LPSTR, LPSTR lpDriverName, PVOID lpContext ) {
	PDD_ENUM DdE = (PDD_ENUM)lpContext;
	if ( lpGUID && ( UtilStr::StrCmp( lpDriverName, "display", false ) != 0 ) ) {
		DdE -> mDeviceGUID = *lpGUID;
		if ( DdE -> mNumRemain > 0 )
			DdE -> mNumRemain--;
	}

	if ( DdE -> mNumRemain == 0 )
		return -1;
	else
		return 0;
}


BOOL WINAPI DdEnumCbEx(GUID FAR *lpGUID, LPSTR, LPSTR lpDriverName, PVOID lpContext, HMONITOR inMonitor ) {
	PDD_ENUM DdE = (PDD_ENUM) lpContext;
	if ( lpGUID && inMonitor == DdE -> mMonitor && ( UtilStr::StrCmp( lpDriverName, "display", false ) != 0 ) ) {
		DdE -> mDeviceGUID = *lpGUID;
		return -1;
	}

	return 0;
}
#endif

long DC_VideoDevice::GetDisplayID( long inDeviceNum ) {

#if EG_MAC
	OSStatus			err;
	DisplayIDType		id = 0;
	GDHandle theGDevice = DMGetFirstScreenDevice( false );
	while ( theGDevice && inDeviceNum ) {
		inDeviceNum--;

		theGDevice = DMGetNextScreenDevice( theGDevice, false );
	}

	if ( ! theGDevice )
		theGDevice = DMGetFirstScreenDevice( false );

	err = DMGetDisplayIDByGDevice( theGDevice, &id, false );

	return ( err ) ? 0 : id;
#endif

#if USE_DIRECTX
	static DD_ENUM sDDEnum;
	memset( &sDDEnum, 0, sizeof(sDDEnum) );
	sDDEnum.mNumRemain = inDeviceNum;
	::DirectDrawEnumerate( &DdEnumCb, &sDDEnum );
	return (long) ( ( sDDEnum.mNumRemain == 0 ) ? &sDDEnum.mDeviceGUID : nil );
#else
	return nil;
#endif
}

#if USE_DIRECTX
typedef HRESULT (*DDEnumerateExFcnT)( LPDDENUMCALLBACKEX, LPVOID, DWORD );
#endif

long DC_VideoDevice::GetDisplayID( PP_GrafPtr inPort ) {

	long dispID = 0;

	#if EG_MAC
		#if TARGET_API_MAC_CARBON
			Rect r;
			GetPortBounds( inPort, &r );
		#else
			Rect r = inPort -> portRect;
		#endif
		dispID = GetDisplayID( ( r.left + r.right ) / 2, ( r.top + r.bottom ) / 2 );
	#endif

	return dispID;
}

/*

bool DC_VideoDevice::GetDisplaySize( Point& ioSize ) {
	bool ok = false;

	#if EG_MAC
		GDHandle myDevice;

		mFS_GDevice = nil;
	#if TARGET_API_MAC_CARBON
		Rect r;
		GetPortBounds( inPortDC, &r );
	#else
		Rect r = inPortDC -> portRect;
	#endif
	long dispID = GetDisplayID( ( r.left + r.right ) / 2, ( r.top + r.bottom ) / 2 );
	::DMGetGDeviceByDisplayID( dispID, &mFS_GDevice, false );
	#endif

	if ( mFS_GDevice ) {
		ioSize.h = (*(*(mFS_GDevice))->gdPMap)->bounds.right  - (*(*(mFS_GDevice))->gdPMap)->bounds.left;
		ioSize.v = (*(*(mFS_GDevice))->gdPMap)->bounds.bottom - (*(*(mFS_GDevice))->gdPMap)->bounds.top;
	}
	#endif

	return ok;
}*/



bool DC_VideoDevice::GetParentDisplaySize( Point& ioSize ) {
	long dispID = GetDisplayID( mDC );

	return GetDisplaySize( dispID, ioSize );
}


bool DC_VideoDevice::GetDisplaySize( long inDispID, Point& ioSize ) {
	bool ok = false;

	ioSize.h = 0;
	ioSize.v = 0;

	#if EG_MAC
	GDHandle theGDevice;

	if ( DMGetGDeviceByDisplayID( inDispID, &theGDevice, false ) == noErr ) {
		ioSize.h = (*(*(theGDevice))->gdPMap)->bounds.right  - (*(*(theGDevice))->gdPMap)->bounds.left;
		ioSize.v = (*(*(theGDevice))->gdPMap)->bounds.bottom - (*(*(theGDevice))->gdPMap)->bounds.top;

		ok = true;
	}
	#endif

	return ok;
}


long DC_VideoDevice::GetDisplayID( long inX, long inY ) {

	#if EG_MAC
	OSStatus			err;
	DisplayIDType		id = 0;
	Point				inPt;

	inPt.h = inX;
	inPt.v = inY;

	GDHandle theGDevice;

	/*
	** Walk the list of display devices in the system.  DrawSprocket is
	** centered around the DisplayIDType, which is used by the Display
	** Manager.  The GDevice records are going to be in flux with future
	** versions of the system software, so it is best to make the change
	** now and make your software DisplayManager-centric.
	*/
	theGDevice = DMGetFirstScreenDevice( false );
	while( theGDevice && ! id ) {

		if ( ::PtInRect( inPt, &(**theGDevice).gdRect ) ) {

			/* get the display ID */
			err = DMGetDisplayIDByGDevice( theGDevice, &id, false );
			if ( err )
				id = 0;
		}

		/* next device */
		theGDevice = DMGetNextScreenDevice( theGDevice, false );
	}
/*
	err = ::DSpFindContextFromPoint( inPt, &ref );
	if ( ! err )
		err = ::DSpContext_GetDisplayID( ref, &id );
*/
	return ( err ) ? 0 : id;
	#endif

	#ifdef EG_WIN
	#pragma unused( inX, inY )
	/*
	#if USE_DIRECTX
	POINT pt;
	static DD_ENUM sDDEnum;
	memset( &sDDEnum, 0, sizeof(sDDEnum) );

	pt.x = inX;
	pt.y = inY;
	sDDEnum.mMonitor = ::MonitorFromPoint( pt, MONITOR_DEFAULTTONEAREST );

	HMODULE mod = ::GetModuleHandle( "Ddraw.dll" );
	if ( mod ) {

		DDEnumerateExFcnT DDEnumEx = (DDEnumerateExFcnT) ::GetProcAddress( mod, "DirectDrawEnumerateExA" );
		if ( ! DDEnumEx )
			DDEnumEx = (DDEnumerateExFcnT) ::GetProcAddress( mod, "DirectDrawEnumerateExW" );

		if ( DDEnumEx ) {

			( DDEnumEx ) ( &DdEnumCbEx, &sDDEnum, DDENUM_ATTACHEDSECONDARYDEVICES );

			return (long) &sDDEnum.mDeviceGUID;
		}
	} */
	#endif

	return nil;
}


void DC_VideoDevice::PreFrame_Self( PixPort* inFrame ) {


	SetDeviceEntries( inFrame );
}




void DC_VideoDevice::BeginFrame_Self() {


	if ( mDC ) {

		#if EG_MAC
		if ( mFS_GDevice )
			::SetGWorld( (CGrafPtr) mDC, mFS_GDevice );
		else
			::SetPort( mDC );
		#endif

	}
}






void DC_VideoDevice::Erase( bool inUpdateNow ) {

	if ( ! inUpdateNow )
		VideoDevice::Erase( inUpdateNow );

	else if ( mDC ) {

		#if EG_MAC
		RGBColor prev;
		GrafPtr	savePort;
		::GetPort( &savePort );
		::SetPort( mDC );
		::GetForeColor( &prev );

		// Overwrite the window with zeros
		//::RGBForeColor( &mBlackAlias );

		#if GFORCE
		if ( IsFullscreen() && mDepth == 8 && ! mSetEntriesCompat )
			ForeColor( 30 );	// Old-school white -- G-Force uses 0 as black, but that's white in MacOS 8 bit
		else
		#endif
			ForeColor( 33 );	// Old-school black
		Pattern pat;
		::GetIndPattern( &pat, sysPatListID, 1 );
		::FillRect( &mOutputRect, &pat );

		::RGBForeColor( &prev );
		::SetPort( savePort );
		#endif


		#if EG_WIN
		RECT r;
		HRGN winRgn;

		r.top 	= mOutputRect.top;
		r.left	= mOutputRect.left;
		r.right  	= mOutputRect.right;
		r.bottom 	= mOutputRect.bottom;
		winRgn = ::CreateRectRgnIndirect( &r );
		::FillRgn( mDC, winRgn, (HBRUSH) ::GetStockObject( BLACK_BRUSH ) );
		::DeleteObject( winRgn );
		#endif
	}
}
