// file      : xsde/cxx/serializer/name-processor.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2006-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <cxx/serializer/name-processor.hxx>

#include <cxx/serializer/elements.hxx>

#include <xsd-frontend/semantic-graph.hxx>
#include <xsd-frontend/traversal.hxx>

#include <cult/containers/set.hxx>

#include <sstream>

namespace CXX
{
  namespace Serializer
  {
    namespace
    {
      //
      //
      typedef Cult::Containers::Set<String> NameSet;

      class Context
      {
      public:
        Context (SemanticGraph::Schema& schema_,
                 SemanticGraph::Path const& file,
                 Boolean impl_name,
                 String const& skel_suf,
                 String const& impl_suf)
            : schema_path_ (file),
              impl (impl_name),
              skel_suffix_ (skel_suf),
              impl_suffix_ (impl_suf),
              schema (schema_),
              schema_path (schema_path_),
              skel_suffix (skel_suffix_),
              impl_suffix (impl_suffix_),
              ns_ (0),
              ns (ns_),
              global_type_names (global_type_names_)
        {
        }

      protected:
        Context (Context& c)
            : schema (c.schema),
              schema_path (c.schema_path),
              impl (c.impl),
              skel_suffix (c.skel_suffix),
              impl_suffix (c.impl_suffix),
              ns (c.ns),
              global_type_names (c.global_type_names)
        {
        }

      private:
        SemanticGraph::Path const schema_path_;
        String const skel_suffix_;
        String const impl_suffix_;
        SemanticGraph::Namespace* ns_;

        Cult::Containers::Map<String, NameSet> global_type_names_;

      public:
        SemanticGraph::Schema& schema;
        SemanticGraph::Path const& schema_path;
        Boolean const impl;
        String const& skel_suffix;
        String const& impl_suffix;
        SemanticGraph::Namespace*& ns;

        Cult::Containers::Map<String, NameSet>& global_type_names;
      };

      String
      find_name (String const& n, String const& suffix, NameSet& set)
      {
        String name (Serializer::Context::escape (n + suffix));

        for (UnsignedLong i (1); set.find (name) != set.end (); ++i)
        {
          std::wostringstream os;
          os << i;
          name = Serializer::Context::escape (n + os.str () + suffix);
        }

        set.insert (name);
        return name;
      }

      String
      find_name (String const& n, NameSet& set)
      {
        return find_name (n, L"", set);
      }

      // Primary names.
      //
      struct PrimaryParticle: Traversal::Element,
                              Traversal::Any
      {
        PrimaryParticle (NameSet& set, Boolean restriction)
            : set_ (set), restriction_ (restriction)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          using SemanticGraph::Element;

          if (!restriction_)
          {
            e.context ().set ("name", find_name (e.name (), set_));
          }
          else
          {
            Element* prot = e.context ().get<Element*> (
              "xsd-frontend-restriction-correspondence");

            e.context ().set ("name", prot->context ().get<String> ("name"));
          }
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          using SemanticGraph::Any;

          if (!restriction_)
          {
            a.context ().set ("name", find_name (L"any", set_));
          }
          else
          {
            Any* prot = a.context ().get<Any*> (
              "xsd-frontend-restriction-correspondence");

            a.context ().set ("name", prot->context ().get<String> ("name"));
          }
        }

      private:
        NameSet& set_;
        Boolean restriction_;
      };

      struct PrimaryAttribute: Traversal::Attribute,
                               Traversal::AnyAttribute
      {
        PrimaryAttribute (NameSet& set, Boolean restriction)
            : set_ (set), restriction_ (restriction)
        {
        }

        virtual Void
        traverse (SemanticGraph::Attribute& a)
        {
          using SemanticGraph::Attribute;

          if (!restriction_)
          {
            a.context ().set ("name", find_name (a.name (), set_));
          }
          else
          {
            Attribute* prot = a.context ().get<Attribute*> (
              "xsd-frontend-restriction-correspondence");

            a.context ().set ("name", prot->context ().get<String> ("name"));
          }
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          using SemanticGraph::AnyAttribute;

          if (!restriction_)
          {
            a.context ().set ("name", find_name (L"any_attribute", set_));
          }
          else
          {
            AnyAttribute* prot = a.context ().get<AnyAttribute*> (
              "xsd-frontend-restriction-correspondence");

            a.context ().set ("name", prot->context ().get<String> ("name"));
          }
        }

      private:
        NameSet& set_;
        Boolean restriction_;
      };

      // Secondary names.
      //

      struct ParticleTag: Traversal::Element,
                          Traversal::Any,
                          Traversal::Choice,
                          Traversal::Sequence
      {
        ParticleTag (NameSet& set, Boolean restriction)
            : set_ (set), restriction_ (restriction)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          using SemanticGraph::Element;

          if (!restriction_)
          {
            String const base (e.context ().get<String> ("name"));
            e.context ().set ("tag", find_name (base, L"_tag", set_));
          }
          else
          {
            Element* prot = e.context ().get<Element*> (
              "xsd-frontend-restriction-correspondence");

            e.context ().set ("tag", prot->context ().get<String> ("tag"));
          }
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          using SemanticGraph::Any;

          if (!restriction_)
          {
            String const base (a.context ().get<String> ("name"));
            a.context ().set ("tag", find_name (base, L"_tag", set_));
          }
          else
          {
            Any* prot = a.context ().get<Any*> (
              "xsd-frontend-restriction-correspondence");

            a.context ().set ("tag", prot->context ().get<String> ("tag"));
          }
        }

        virtual Void
        traverse (SemanticGraph::Choice& c)
        {
          using SemanticGraph::Compositor;

          if (!restriction_)
          {
            c.context ().set ("tag", find_name (L"choice", L"_tag", set_));
          }
          else
          {
            Compositor* prot = c.context ().get<Compositor*> (
              "xsd-frontend-restriction-correspondence");

            c.context ().set ("tag", prot->context ().get<String> ("tag"));
          }
        }

        virtual Void
        traverse (SemanticGraph::Sequence& s)
        {
          using SemanticGraph::Compositor;

          if (!restriction_)
          {
            s.context ().set ("tag", find_name (L"sequence", L"_tag", set_));
          }
          else
          {
            Compositor* prot = s.context ().get<Compositor*> (
              "xsd-frontend-restriction-correspondence");

            s.context ().set ("tag", prot->context ().get<String> ("tag"));
          }
        }

      private:
        NameSet& set_;
        Boolean restriction_;
      };

      struct SecondaryCompositor: Traversal::All,
                                  Traversal::Choice,
                                  Traversal::Sequence
      {
        SecondaryCompositor (NameSet& set, Boolean restriction)
            : set_ (set),
              restriction_ (restriction),
              particle_tag_ (set, restriction)
        {
          contain_particle_tag_ >> particle_tag_;
        }

        virtual Void
        traverse (SemanticGraph::All& a)
        {
          // For the all compositor, maxOccurs=1 and minOccurs={0,1}
          // and it cannot be used in restriction.
          //
          if (a.contained_compositor ().min () == 0)
          {
            a.context ().set ("present",
                              find_name (L"all", L"_present", set_));
          }

          Traversal::All::traverse (a);
        }

        virtual Void
        traverse (SemanticGraph::Choice& c)
        {
          if (c.contains_begin () != c.contains_end ())
          {
            UnsignedLong min;
            UnsignedLong max;

            if (c.contained_compositor_p ())
            {
              min = c.contained_compositor ().min ();
              max = c.contained_compositor ().max ();
            }
            else
            {
              min = c.contained_particle ().min ();
              max = c.contained_particle ().max ();
            }

            if (!restriction_)
            {
              if (min == 0 && max == 1)
              {
                c.context ().set ("present",
                                  find_name (L"choice", L"_present", set_));
              }
              else if (max != 1)
              {
                c.context ().set ("next",
                                  find_name (L"choice", L"_next", set_));
              }

              // Tags.
              //
              c.context ().set ("arm-tag",
                                find_name (L"choice", L"_arm_tag", set_));

              Traversal::Choice::contains (c, contain_particle_tag_);

              c.context ().set ("arm", find_name (L"choice", L"_arm", set_));
            }
            else
            {
              SemanticGraph::Compositor* prot =
                c.context ().get<SemanticGraph::Compositor*> (
                  "xsd-frontend-restriction-correspondence");

              if (min == 0 && max == 1)
              {
                c.context ().set ("present",
                                  prot->context ().get<String> ("present"));
              }
              else if (max != 1)
              {
                c.context ().set ("next",
                                  prot->context ().get<String> ("next"));
              }

              // Tags.
              //
              c.context ().set ("arm-tag",
                                prot->context ().get<String> ("arm-tag"));

              Traversal::Choice::contains (c, contain_particle_tag_);

              c.context ().set ("arm", prot->context ().get<String> ("arm"));
            }

            Traversal::Choice::traverse (c);
          }
        }

        virtual Void
        traverse (SemanticGraph::Sequence& s)
        {
          UnsignedLong min;
          UnsignedLong max;

          if (s.contained_compositor_p ())
          {
            min = s.contained_compositor ().min ();
            max = s.contained_compositor ().max ();
          }
          else
          {
            min = s.contained_particle ().min ();
            max = s.contained_particle ().max ();
          }

          if (!restriction_)
          {
            if (min == 0 && max == 1)
            {
              s.context ().set ("present",
                                find_name (L"sequence", L"_present", set_));
            }
            else if (max != 1)
            {
              s.context ().set ("next",
                                find_name (L"sequence", L"_next", set_));
            }
          }
          else
          {
            // Root compositor that models inheritance by extension
            // may not have an association.
            //
            if (min == 0 || max != 1)
            {
              SemanticGraph::Compositor* prot =
                s.context ().get<SemanticGraph::Compositor*> (
                  "xsd-frontend-restriction-correspondence");

              if (min == 0 && max == 1)
              {
                s.context ().set ("present",
                                  prot->context ().get<String> ("present"));
              }
              else if (max != 1)
              {
                s.context ().set ("next",
                                  prot->context ().get<String> ("next"));
              }
            }
          }

          Traversal::Sequence::traverse (s);
        }

      private:
        NameSet& set_;
        Boolean restriction_;

        ParticleTag particle_tag_;
        Traversal::ContainsParticle contain_particle_tag_;
      };

      struct SecondaryParticle: Traversal::Element,
                                Traversal::Any
      {
        SecondaryParticle (NameSet& set, Boolean restriction)
            : set_ (set), restriction_ (restriction)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          using SemanticGraph::Element;

          UnsignedLong min (e.contained_particle ().min ());
          UnsignedLong max (e.contained_particle ().max ());

          if (!restriction_)
          {
            String const& base (e.context ().get<String> ("name"));

            if (min == 0 && max == 1)
            {
              e.context ().set ("present",
                                find_name (base + L"_present", set_));
            }
            else if (max != 1)
            {
              e.context ().set ("next", find_name (base + L"_next", set_));
            }

            e.context ().set ("serializer",
                              find_name (base + L"_serializer", set_));

            e.context ().set ("member", find_name (base + L"_", set_));
          }
          else
          {
            Element* prot = e.context ().get<Element*> (
              "xsd-frontend-restriction-correspondence");

            if (min == 0 && max == 1)
            {
              e.context ().set ("present",
                                prot->context ().get<String> ("present"));
            }
            else if (max != 1)
            {
              e.context ().set ("next",
                                prot->context ().get<String> ("next"));
            }

            e.context ().set ("serializer",
                              prot->context ().get<String> ("serializer"));

            e.context ().set ("member",
                              prot->context ().get<String> ("member"));
          }
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          using SemanticGraph::Any;

          UnsignedLong min (a.contained_particle ().min ());
          UnsignedLong max (a.contained_particle ().max ());

          if (!restriction_)
          {
            String const& base (a.context ().get<String> ("name"));

            if (min == 0 && max == 1)
            {
              a.context ().set ("present",
                                find_name (base + L"_present", set_));
            }
            else if (max != 1)
            {
              a.context ().set ("next", find_name (base + L"_next", set_));
            }

            a.context ().set ("serialize",
                              find_name (L"serialize_" + base, set_));
          }
          else
          {
            Any* prot = a.context ().get<Any*> (
              "xsd-frontend-restriction-correspondence");

            if (min == 0 && max == 1)
            {
              a.context ().set ("present",
                                prot->context ().get<String> ("present"));
            }
            else if (max != 1)
            {
              a.context ().set ("next",
                                prot->context ().get<String> ("next"));
            }

            a.context ().set ("serialize",
                              prot->context ().get<String> ("serialize"));
          }
        }

      private:
        NameSet& set_;
        Boolean restriction_;
      };

      struct SecondaryAttribute: Traversal::Attribute,
                                 Traversal::AnyAttribute
      {
        SecondaryAttribute (NameSet& set, Boolean restriction)
            : set_ (set), restriction_ (restriction)
        {
        }

        virtual Void
        traverse (SemanticGraph::Attribute& a)
        {
          using SemanticGraph::Attribute;

          if (!restriction_)
          {
            String const& base (a.context ().get<String> ("name"));

            if (a.optional ())
            {
              a.context ().set ("present",
                                find_name (base + L"_present", set_));
            }

            a.context ().set ("serializer",
                              find_name (base + L"_serializer", set_));

            a.context ().set ("member", find_name (base + L"_", set_));
          }
          else
          {
            Attribute* prot = a.context ().get<Attribute*> (
              "xsd-frontend-restriction-correspondence");

            if (a.optional ())
            {
              a.context ().set ("present",
                                prot->context ().get<String> ("present"));
            }

            a.context ().set ("serializer",
                              prot->context ().get<String> ("serializer"));

            a.context ().set ("member",
                              prot->context ().get<String> ("member"));
          }
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          using SemanticGraph::AnyAttribute;

          if (!restriction_)
          {
            String const& base (a.context ().get<String> ("name"));

            a.context ().set ("next", find_name (base + L"_next", set_));

            a.context ().set ("serialize",
                              find_name (L"serialize_" + base, set_));
          }
          else
          {
            AnyAttribute* prot = a.context ().get<AnyAttribute*> (
              "xsd-frontend-restriction-correspondence");

            a.context ().set ("next", prot->context ().get<String> ("next"));

            a.context ().set ("serialize",
                              prot->context ().get<String> ("serialize"));
          }
        }

      private:
        NameSet& set_;
        Boolean restriction_;
      };

      //
      //
      struct Complex: Traversal::Complex
      {
        virtual Void
        traverse (Type& c)
        {
          // Use processed name.
          //
          String const& name (c.context ().get<String> ("name"));

          NameSet member_set;
          member_set.insert (name);

          // Add our base's members to the initial list unless we are
          // inheriting by restriction in which case we need to have
          // the same names as our base.
          //
          Boolean restriction (false);

          if (c.inherits_p ())
          {
            // @@ What if this types name is the same as one of base's
            //    members?
            //
            SemanticGraph::Type& base (c.inherits ().base ());

            if (base.is_a<SemanticGraph::Complex> ())
            {
              if (!base.context ().count (
                    "cxx-serializer-name-processor-member-set"))
              {
                Complex t;
                t.dispatch (base);
              }

              NameSet const& base_set (
                base.context ().get<NameSet> (
                  "cxx-serializer-name-processor-member-set"));

              member_set.insert (base_set.begin (), base_set.end ());

              // Inheritance by restriction from anyType is a special case.
	      //
              restriction = c.inherits ().is_a<SemanticGraph::Restricts> () &&
	        !c.inherits ().base ().is_a<SemanticGraph::AnyType> ();
            }
          }

          // First assign the "primary" names.
          //
          {
            if (c.contains_compositor_p ())
            {
              PrimaryParticle particle (member_set, restriction);
              Traversal::Compositor compositor;
              Traversal::ContainsCompositor contains_compositor;
              Traversal::ContainsParticle contains_particle;

              contains_compositor >> compositor >> contains_particle;

              contains_particle >> compositor;
              contains_particle >> particle;

              Complex::contains_compositor (c, contains_compositor);
            }

            //
            //
            PrimaryAttribute attribute (member_set, restriction);
            Traversal::Names names (attribute);

            Complex::names (c, names);
          }

          // Assign "secondary" names.
          //
          {
            if (c.contains_compositor_p ())
            {
              SecondaryParticle particle (member_set, restriction);
              SecondaryCompositor compositor (member_set, restriction);
              Traversal::ContainsCompositor contains_compositor;
              Traversal::ContainsParticle contains_particle;

              contains_compositor >> compositor >> contains_particle;

              contains_particle >> compositor;
              contains_particle >> particle;

              Complex::contains_compositor (c, contains_compositor);
            }

            //
            //
            SecondaryAttribute attribute (member_set, restriction);
            Traversal::Names names (attribute);

            Complex::names (c, names);
          }

          // @@ I am not removing this at the moment.
          //
          c.context ().set ("cxx-serializer-name-processor-member-set",
                            member_set);
        }
      };


      //
      //
      struct GlobalType: Traversal::Type, Context
      {
        GlobalType (Context& c, NameSet& set)
            : Context (c), set_ (set)
        {
        }

        virtual Void
        traverse (SemanticGraph::Type& t)
        {
          String name (find_name (t.name () + skel_suffix, set_));
          t.context ().set ("name", name);

          // Note that we do not add this name to the set so that it
          // does not influence other names.
          //
          if (impl)
          {
            String impl (Serializer::Context::escape (t.name () + impl_suffix));
            t.context ().set ("impl", impl);
          }
        }

      private:
        NameSet& set_;
      };


      struct Namespace: Traversal::Namespace, Context
      {
        Namespace (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& ns)
        {
          NameSet& type_set (global_type_names[ns.name ()]);

          GlobalType type (*this, type_set);
          Traversal::Names names (type);

          Context::ns = &ns;

          Traversal::Namespace::names (ns, names);

          Context::ns = 0;
        }
      };

      struct FundType : Traversal::AnyType,
                        Traversal::AnySimpleType,

                        Traversal::Fundamental::Byte,
                        Traversal::Fundamental::UnsignedByte,
                        Traversal::Fundamental::Short,
                        Traversal::Fundamental::UnsignedShort,
                        Traversal::Fundamental::Int,
                        Traversal::Fundamental::UnsignedInt,
                        Traversal::Fundamental::Long,
                        Traversal::Fundamental::UnsignedLong,
                        Traversal::Fundamental::Integer,
                        Traversal::Fundamental::NonPositiveInteger,
                        Traversal::Fundamental::NonNegativeInteger,
                        Traversal::Fundamental::PositiveInteger,
                        Traversal::Fundamental::NegativeInteger,

                        Traversal::Fundamental::Boolean,

                        Traversal::Fundamental::Float,
                        Traversal::Fundamental::Double,
                        Traversal::Fundamental::Decimal,

                        Traversal::Fundamental::String,
                        Traversal::Fundamental::NormalizedString,
                        Traversal::Fundamental::Token,
                        Traversal::Fundamental::Name,
                        Traversal::Fundamental::NameToken,
                        Traversal::Fundamental::NameTokens,
                        Traversal::Fundamental::NCName,
                        Traversal::Fundamental::Language,

                        Traversal::Fundamental::QName,

                        Traversal::Fundamental::Id,
                        Traversal::Fundamental::IdRef,
                        Traversal::Fundamental::IdRefs,

                        Traversal::Fundamental::AnyURI,

                        Traversal::Fundamental::Base64Binary,
                        Traversal::Fundamental::HexBinary,

                        Traversal::Fundamental::Date,
                        Traversal::Fundamental::DateTime,
                        Traversal::Fundamental::Duration,
                        Traversal::Fundamental::Day,
                        Traversal::Fundamental::Month,
                        Traversal::Fundamental::MonthDay,
                        Traversal::Fundamental::Year,
                        Traversal::Fundamental::YearMonth,
                        Traversal::Fundamental::Time,

                        Traversal::Fundamental::Entity,
                        Traversal::Fundamental::Entities

      {
        FundType (String const& skel_suffix, String const& impl_suffix)
            : skel_suffix_ (skel_suffix),
              impl_suffix_ (impl_suffix)
        {
        }

        // anyType & anySimpleType.
        //
        virtual Void
        traverse (SemanticGraph::AnyType& t)
        {
          t.context ().set ("name", make_skel_name ("any_type"));
          t.context ().set ("impl", make_impl_name ("any_type"));
          t.context ().set ("post", String ("post_any_type"));
        }

        virtual Void
        traverse (SemanticGraph::AnySimpleType& t)
        {
          t.context ().set ("name", make_skel_name ("any_simple_type"));
          t.context ().set ("impl", make_impl_name ("any_simple_type"));
          t.context ().set ("post", String ("post_any_simple_type"));
        }

        // Boolean.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Boolean& t)
        {
          t.context ().set ("name", make_skel_name ("boolean"));
          t.context ().set ("impl", make_impl_name ("boolean"));
          t.context ().set ("post", String ("post_boolean"));
        }

        // Integral types.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Byte& t)
        {
          t.context ().set ("name", make_skel_name ("byte"));
          t.context ().set ("impl", make_impl_name ("byte"));
          t.context ().set ("post", String ("post_byte"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::UnsignedByte& t)
        {
          t.context ().set ("name", make_skel_name ("unsigned_byte"));
          t.context ().set ("impl", make_impl_name ("unsigned_byte"));
          t.context ().set ("post", String ("post_unsigned_byte"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Short& t)
        {
          t.context ().set ("name", make_skel_name ("short"));
          t.context ().set ("impl", make_impl_name ("short"));
          t.context ().set ("post", String ("post_short"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::UnsignedShort& t)
        {
          t.context ().set ("name", make_skel_name ("unsigned_short"));
          t.context ().set ("impl", make_impl_name ("unsigned_short"));
          t.context ().set ("post", String ("post_unsigned_short"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Int& t)
        {
          t.context ().set ("name", make_skel_name ("int"));
          t.context ().set ("impl", make_impl_name ("int"));
          t.context ().set ("post", String ("post_int"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::UnsignedInt& t)
        {
          t.context ().set ("name", make_skel_name ("unsigned_int"));
          t.context ().set ("impl", make_impl_name ("unsigned_int"));
          t.context ().set ("post", String ("post_unsigned_int"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Long& t)
        {
          t.context ().set ("name", make_skel_name ("long"));
          t.context ().set ("impl", make_impl_name ("long"));
          t.context ().set ("post", String ("post_long"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::UnsignedLong& t)
        {
          t.context ().set ("name", make_skel_name ("unsigned_long"));
          t.context ().set ("impl", make_impl_name ("unsigned_long"));
          t.context ().set ("post", String ("post_unsigned_long"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Integer& t)
        {
          t.context ().set ("name", make_skel_name ("integer"));
          t.context ().set ("impl", make_impl_name ("integer"));
          t.context ().set ("post", String ("post_integer"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NonPositiveInteger& t)
        {
          t.context ().set ("name", make_skel_name ("non_positive_integer"));
          t.context ().set ("impl", make_impl_name ("non_positive_integer"));
          t.context ().set ("post", String ("post_non_positive_integer"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NonNegativeInteger& t)
        {
          t.context ().set ("name", make_skel_name ("non_negative_integer"));
          t.context ().set ("impl", make_impl_name ("non_negative_integer"));
          t.context ().set ("post", String ("post_non_negative_integer"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::PositiveInteger& t)
        {
          t.context ().set ("name", make_skel_name ("positive_integer"));
          t.context ().set ("impl", make_impl_name ("positive_integer"));
          t.context ().set ("post", String ("post_positive_integer"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NegativeInteger& t)
        {
          t.context ().set ("name", make_skel_name ("negative_integer"));
          t.context ().set ("impl", make_impl_name ("negative_integer"));
          t.context ().set ("post", String ("post_negative_integer"));
        }

        // Floats.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Float& t)
        {
          t.context ().set ("name", make_skel_name ("float"));
          t.context ().set ("impl", make_impl_name ("float"));
          t.context ().set ("post", String ("post_float"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Double& t)
        {
          t.context ().set ("name", make_skel_name ("double"));
          t.context ().set ("impl", make_impl_name ("double"));
          t.context ().set ("post", String ("post_double"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Decimal& t)
        {
          t.context ().set ("name", make_skel_name ("decimal"));
          t.context ().set ("impl", make_impl_name ("decimal"));
          t.context ().set ("post", String ("post_decimal"));
        }

        // Strings.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::String& t)
        {
          t.context ().set ("name", make_skel_name ("string"));
          t.context ().set ("impl", make_impl_name ("string"));
          t.context ().set ("post", String ("post_string"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NormalizedString& t)
        {
          t.context ().set ("name", make_skel_name ("normalized_string"));
          t.context ().set ("impl", make_impl_name ("normalized_string"));
          t.context ().set ("post", String ("post_normalized_string"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Token& t)
        {
          t.context ().set ("name", make_skel_name ("token"));
          t.context ().set ("impl", make_impl_name ("token"));
          t.context ().set ("post", String ("post_token"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NameToken& t)
        {
          t.context ().set ("name", make_skel_name ("nmtoken"));
          t.context ().set ("impl", make_impl_name ("nmtoken"));
          t.context ().set ("post", String ("post_nmtoken"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NameTokens& t)
        {
          t.context ().set ("name", make_skel_name ("nmtokens"));
          t.context ().set ("impl", make_impl_name ("nmtokens"));
          t.context ().set ("post", String ("post_nmtokens"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Name& t)
        {
          t.context ().set ("name", make_skel_name ("name"));
          t.context ().set ("impl", make_impl_name ("name"));
          t.context ().set ("post", String ("post_name"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::NCName& t)
        {
          t.context ().set ("name", make_skel_name ("ncname"));
          t.context ().set ("impl", make_impl_name ("ncname"));
          t.context ().set ("post", String ("post_ncname"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Language& t)
        {
          t.context ().set ("name", make_skel_name ("language"));
          t.context ().set ("impl", make_impl_name ("language"));
          t.context ().set ("post", String ("post_language"));
        }


        // Qualified name.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::QName& t)
        {
          t.context ().set ("name", make_skel_name ("qname"));
          t.context ().set ("impl", make_impl_name ("qname"));
          t.context ().set ("post", String ("post_qname"));
        }


        // ID/IDREF.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Id& t)
        {
          t.context ().set ("name", make_skel_name ("id"));
          t.context ().set ("impl", make_impl_name ("id"));
          t.context ().set ("post", String ("post_id"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::IdRef& t)
        {
          t.context ().set ("name", make_skel_name ("idref"));
          t.context ().set ("impl", make_impl_name ("idref"));
          t.context ().set ("post", String ("post_idref"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::IdRefs& t)
        {
          t.context ().set ("name", make_skel_name ("idrefs"));
          t.context ().set ("impl", make_impl_name ("idrefs"));
          t.context ().set ("post", String ("post_idrefs"));
        }

        // URI.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::AnyURI& t)
        {
          t.context ().set ("name", make_skel_name ("uri"));
          t.context ().set ("impl", make_impl_name ("uri"));
          t.context ().set ("post", String ("post_uri"));
        }

        // Binary.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Base64Binary& t)
        {
          t.context ().set ("name", make_skel_name ("base64_binary"));
          t.context ().set ("impl", make_impl_name ("base64_binary"));
          t.context ().set ("post", String ("post_base64_binary"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::HexBinary& t)
        {
          t.context ().set ("name", make_skel_name ("hex_binary"));
          t.context ().set ("impl", make_impl_name ("hex_binary"));
          t.context ().set ("post", String ("post_hex_binary"));
        }


        // Date/time.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Date& t)
        {
          t.context ().set ("name", make_skel_name ("date"));
          t.context ().set ("impl", make_impl_name ("date"));
          t.context ().set ("post", String ("post_date"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::DateTime& t)
        {
          t.context ().set ("name", make_skel_name ("date_time"));
          t.context ().set ("impl", make_impl_name ("date_time"));
          t.context ().set ("post", String ("post_date_time"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Duration& t)
        {
          t.context ().set ("name", make_skel_name ("duration"));
          t.context ().set ("impl", make_impl_name ("duration"));
          t.context ().set ("post", String ("post_duration"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Day& t)
        {
          t.context ().set ("name", make_skel_name ("gday"));
          t.context ().set ("impl", make_impl_name ("gday"));
          t.context ().set ("post", String ("post_gday"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Month& t)
        {
          t.context ().set ("name", make_skel_name ("gmonth"));
          t.context ().set ("impl", make_impl_name ("gmonth"));
          t.context ().set ("post", String ("post_gmonth"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::MonthDay& t)
        {
          t.context ().set ("name", make_skel_name ("gmonth_day"));
          t.context ().set ("impl", make_impl_name ("gmonth_day"));
          t.context ().set ("post", String ("post_gmonth_day"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Year& t)
        {
          t.context ().set ("name", make_skel_name ("gyear"));
          t.context ().set ("impl", make_impl_name ("gyear"));
          t.context ().set ("post", String ("post_gyear"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::YearMonth& t)
        {
          t.context ().set ("name", make_skel_name ("gyear_month"));
          t.context ().set ("impl", make_impl_name ("gyear_month"));
          t.context ().set ("post", String ("post_gyear_month"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Time& t)
        {
          t.context ().set ("name", make_skel_name ("time"));
          t.context ().set ("impl", make_impl_name ("time"));
          t.context ().set ("post", String ("post_time"));
        }

        // Entity.
        //
        virtual Void
        traverse (SemanticGraph::Fundamental::Entity& t)
        {
          t.context ().set ("name", make_skel_name ("entity"));
          t.context ().set ("impl", make_impl_name ("entity"));
          t.context ().set ("post", String ("post_entity"));
        }

        virtual Void
        traverse (SemanticGraph::Fundamental::Entities& t)
        {
          t.context ().set ("name", make_skel_name ("entities"));
          t.context ().set ("impl", make_impl_name ("entities"));
          t.context ().set ("post", String ("post_entities"));
        }

      private:
        String
        make_skel_name (String const& base)
        {
          return Serializer::Context::escape (base + skel_suffix_);
        }

        String
        make_impl_name (String const& base)
        {
          return Serializer::Context::escape (base + impl_suffix_);
        }

        String skel_suffix_;
        String impl_suffix_;
      };

      // Go into sourced/included/imported schemas while making sure
      // we don't process the same stuff more than once.
      //
      struct Uses: Traversal::Sources,
                   Traversal::Includes,
                   Traversal::Imports
      {
        virtual Void
        traverse (SemanticGraph::Sources& sr)
        {
          SemanticGraph::Schema& s (sr.schema ());

          if (!s.context ().count ("cxx-serializer-name-processor-seen"))
          {
            s.context ().set ("cxx-serializer-name-processor-seen", true);
            Traversal::Sources::traverse (sr);
          }
        }

        virtual Void
        traverse (SemanticGraph::Includes& i)
        {
          SemanticGraph::Schema& s (i.schema ());

          if (!s.context ().count ("cxx-serializer-name-processor-seen"))
          {
            s.context ().set ("cxx-serializer-name-processor-seen", true);
            Traversal::Includes::traverse (i);
          }
        }

        virtual Void
        traverse (SemanticGraph::Imports& i)
        {
          SemanticGraph::Schema& s (i.schema ());

          if (!s.context ().count ("cxx-serializer-name-processor-seen"))
          {
            s.context ().set ("cxx-serializer-name-processor-seen", true);
            Traversal::Imports::traverse (i);
          }
        }
      };

      Void
      process_impl (SemanticGraph::Schema& tu,
                    SemanticGraph::Path const& file,
                    Boolean impl,
                    String const& skel_suffix,
                    String const& impl_suffix)
      {
        // Pass one - assign names to fundamental types.
        //
        {
          Traversal::Schema schema;
          Traversal::Implies implies;

          schema >> implies >> schema;

          Traversal::Names schema_names;
          Traversal::Namespace ns;
          Traversal::Names ns_names;
          FundType fund_type (skel_suffix, impl_suffix);

          schema >> schema_names >> ns >> ns_names >> fund_type;

          schema.dispatch (tu);
        }

        // Pass two - assign names to global types. This pass cannot
        // be combined with pass three because of possible recursive
        // schema inclusions.
        //
        {
          Context ctx (tu, file, impl, skel_suffix, impl_suffix);

          Traversal::Schema schema;
          Uses uses;

          schema >> uses >> schema;

          Traversal::Names schema_names;
          Namespace ns (ctx);

          schema >> schema_names >> ns;

          // Some twisted schemas do recusive self-inclusion.
          //
          tu.context ().set ("cxx-serializer-name-processor-seen", true);

          schema.dispatch (tu);
        }

        // Pass three - assign names inside complex types. Here we don't
        // need to go into included/imported schemas.
        //
        {
          Traversal::Schema schema;
          Traversal::Sources sources;

          schema >> sources >> schema;

          Traversal::Names schema_names;
          Traversal::Namespace ns;
          Traversal::Names ns_names;

          schema >> schema_names >> ns >> ns_names;

          Complex complex;

          ns_names >> complex;

          schema.dispatch (tu);
        }
      }
    }

    Void NameProcessor::
    process (SemanticGraph::Schema& tu,
             SemanticGraph::Path const& file,
             Boolean impl,
             String const& skel_suffix,
             String const& impl_suffix)
    {
      process_impl (tu, file, impl, skel_suffix, impl_suffix);
    }
  }
}
