// file      : xsde/cxx/serializer/serializer-source.cxx
// author    : Boris Kolpackov <boris@codesynthesis.com>
// copyright : Copyright (c) 2005-2007 Code Synthesis Tools CC
// license   : GNU GPL v2 + exceptions; see accompanying LICENSE file

#include <cxx/serializer/serializer-source.hxx>

#include <xsd-frontend/semantic-graph.hxx>
#include <xsd-frontend/traversal.hxx>

namespace CXX
{
  namespace Serializer
  {
    namespace
    {
      struct Enumeration: Traversal::Enumeration,
                          protected virtual Context
      {
        Enumeration (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& e)
        {
          String const& arg (arg_type (e));
          SemanticGraph::Type& base (e.inherits ().base ());

          if (arg != arg_type (base) && arg == L"void")
          {
            String const& name (ename (e));

            os << "// " << name << endl
               << "//" << endl
               << endl;

            os << "void " << name << "::" << endl
               << "pre ()"
               << "{"
               << "}";
          }
        }
      };

      //
      //
      struct List: Traversal::List, protected virtual Context
      {
        List (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& l)
        {
          String const& name (ename (l));
          SemanticGraph::Type& t (l.argumented ().type ());

          String const& ret (ret_type (t));
          String const& arg (arg_type (t));

          String item (unclash (name, "item"));
          String item_next (unclash (name, "item_next"));

          os << "// " << name << endl
             << "//" << endl
             << endl;

          // pre
          //
          if (arg_type (l) == L"void")
          {
            os << "void " << name << "::" << endl
               << "pre ()"
               << "{"
               << "}";
          }

          // item
          //
          os << "bool " << name << "::" << endl
             << item_next << " ()"
             << "{"
             << "return false;"
             << "}";

          if (ret == L"void")
          {
            os << ret << " " << name << "::" << endl
               << item << " ()"
               << "{"
               << "}";
          }

          // serialize_content
          //
          String inst (L"_xsde_" + item + L"_");

          os << "void " << name << "::" << endl
             << "_serialize_content ()"
             << "{";

          os << "bool first = true;"
             << "::xsde::cxx::serializer::context& ctx = this->_context ();"
             << endl;

          if (exceptions)
          {
            os << "while (this->" << item_next << " ())"
               << "{"
               << "if (this->" << inst << ")"
               << "{";

            if (ret == L"void")
              os << "this->" << item << " ();"
                 << "this->" << inst << "->pre ();";
            else
              os << arg << " r = this->" << item << " ();"
                 << "this->" << inst << "->pre (r);";

            os << endl
               << "if (!first)" << endl
               << "this->_characters (\" \", 1);"
               << "else" << endl
               << "first = false;"
               << endl;

            os << "this->" << inst << "->_pre_impl (ctx);"
               << "this->" << inst << "->_serialize_content ();"
               << "this->" << inst << "->_post_impl ();"
               << "this->" << inst << "->post ();";

            os << "}"
               << "}";
          }
          else
          {
            os << "while (this->" << item_next << " ())"
               << "{"
               << "if (this->" << inst << ")"
               << "{";

            if (ret == L"void")
              os << "this->" << item << " ();";
            else
              os << arg << " r = this->" << item << " ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            if (ret == L"void")
              os << "this->" << inst << "->pre ();";
            else
              os << "this->" << inst << "->pre (r);";

            os << endl
               << "if (this->" << inst << "->_error_type ())" << endl
               << "this->" << inst << "->_copy_error (ctx);"
               << endl;

            os << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "if (!first)"
               << "{"
               << "if (!this->_characters (\" \", 1))" << endl
               << "return;"
               << "}"
               << "else" << endl
               << "first = false;"
               << endl;

            os << "this->" << inst << "->_pre_impl (ctx);";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->_serialize_content ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->_post_impl ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->post ();";

            os << endl
               << "if (this->" << inst << "->_error_type ())" << endl
               << "this->" << inst << "->_copy_error (ctx);"
               << endl;

            os << "if (ctx.error_type ())" << endl
               << "return;";

            os << "}" // No check for error here since we return anyway.
               << "}";
          }

          os << "}";
        }
      };

      //
      //
      struct Union: Traversal::Union, protected virtual Context
      {
        Union (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (Type& u)
        {
          if (arg_type (u) == L"void")
          {
            String const& name (ename (u));

            os << "// " << name << endl
               << "//" << endl
               << endl;

            os << "void " << name << "::" << endl
               << "pre ()"
               << "{"
               << "}";
          }
        }
      };

      // Complex serialization code.
      //
      struct Compositor: Traversal::All,
                         Traversal::Choice,
                         Traversal::Sequence,
                         protected virtual Context
      {
        Compositor (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::All& a)
        {
          // For the all compositor, maxOccurs=1 and minOccurs={0,1}.
          //
          UnsignedLong min (a.contained_compositor ().min ());

          if (min == 0)
            os << "if (this->" << epresent (a) << " ())"
               << "{";

          Traversal::All::traverse (a);

          if (min == 0)
          {
            os << "}";

            if (!exceptions)
            {
              os << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl;
            }
          }
        }

        virtual Void
        traverse (SemanticGraph::Choice& c)
        {
          if (c.contains_begin () != c.contains_end ())
          {
            UnsignedLong min;
            UnsignedLong max;

            if (c.contained_compositor_p ())
            {
              min = c.contained_compositor ().min ();
              max = c.contained_compositor ().max ();
            }
            else
            {
              min = c.contained_particle ().min ();
              max = c.contained_particle ().max ();
            }

            if (min == 0 && max == 1)
            {
              os << "if (this->" << epresent (c) << " ())"
                 << "{";
            }
            else if (max != 1)
            {
              os << "while (this->" << enext (c) << " ())"
                 << "{";
            }
            else if (!exceptions &&
                     c.contained_particle_p () &&
                     c.contained_particle ().compositor ().
                     is_a<SemanticGraph::Sequence> ())
            {
              // Only sequence can have several choice compositors in a row.
              //
              os << "{";
            }

            if (exceptions)
              os << "switch (this->" << earm (c) << " ())";
            else
              os << earm_tag (c) << " t = this->" << earm (c) << " ();"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "switch (t)";


              os << "{";

            for (SemanticGraph::Choice::ContainsIterator
                   i (c.contains_begin ()); i != c.contains_end (); ++i)
            {
              os << "case " << etag (i->particle ()) << ":"
                 << "{";

              edge_traverser ().dispatch (*i);

              os << "break;"
                 << "}";
            }

            os << "}";

            if (min == 0 || max != 1)
            {
              os << "}";

              if (!exceptions)
              {
                os << "if (ctx.error_type ())" << endl
                   << "return;"
                   << endl;
              }
            }
            else if (!exceptions &&
                     c.contained_particle_p () &&
                     c.contained_particle ().compositor ().
                     is_a<SemanticGraph::Sequence> ())
            {
              os << "}";
            }
          }
        }

        virtual Void
        traverse (SemanticGraph::Sequence& s)
        {
          UnsignedLong min;
          UnsignedLong max;

          if (s.contained_compositor_p ())
          {
            min = s.contained_compositor ().min ();
            max = s.contained_compositor ().max ();
          }
          else
          {
            min = s.contained_particle ().min ();
            max = s.contained_particle ().max ();
          }

          if (min == 0 && max == 1)
          {
            os << "if (this->" << epresent (s) << " ())"
               << "{";
          }
          else if (max != 1)
          {
            os << "while (this->" << enext (s) << " ())"
               << "{";
          }

          Traversal::Sequence::traverse (s);


          if (min == 0 || max != 1)
          {
            os << "}";

            if (!exceptions)
            {
              os << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl;
            }
          }
        }
      };

      struct Particle: Traversal::Element,
                       Traversal::Any,
                       protected virtual Context
      {
        Particle (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          UnsignedLong min (e.contained_particle ().min ());
          UnsignedLong max (e.contained_particle ().max ());

          if (min == 0 && max == 1)
          {
            os << "if (this->" << epresent (e) << " ())"
               << "{";
          }
          else if (max != 1)
          {
            os << "while (this->" << enext (e) << " ())"
               << "{";
          }

          String const& name (ename (e));
          String const& inst (emember (e));
          String const& ret (ret_type (e.type ()));
          String const& arg (arg_type (e.type ()));

          os << "if (this->" << inst << ")"
             << "{";

          if (exceptions)
          {
            if (ret == L"void")
              os << "this->" << name << " ();"
                 << "this->" << inst << "->pre ();";
            else
              os << arg << " r = this->" << name << " ();"
                 << "this->" << inst << "->pre (r);";

            if (e.qualified ())
              os << "this->_start_element (\"" << e.namespace_ ().name () <<
                "\", \"" << e.name () << "\");";
            else
              os << "this->_start_element (\"" << e.name () << "\");";

            os << "this->" << inst << "->_pre_impl (ctx);"
               << "this->" << inst << "->_serialize_attributes ();"
               << "this->" << inst << "->_serialize_content ();"
               << "this->" << inst << "->_post_impl ();"
               << "this->_end_element ();"
               << "this->" << inst << "->post ();";
          }
          else
          {
            if (ret == L"void")
              os << "this->" << name << " ();";
            else
              os << arg << " r = this->" << name << " ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            if (ret == L"void")
              os << "this->" << inst << "->pre ();";
            else
              os << "this->" << inst << "->pre (r);";

            os << endl
               << "if (this->" << inst << "->_error_type ())" << endl
               << "this->" << inst << "->_copy_error (ctx);"
               << endl;

            os << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            if (e.qualified ())
              os << "if (!this->_start_element (\"" <<
                e.namespace_ ().name () << "\", \"" <<
                e.name () << "\"))" << endl
                 << "return;"
                 << endl;
            else
              os << "if (!this->_start_element (\"" <<
                e.name () << "\"))" << endl
                 << "return;"
                 << endl;

            os << "this->" << inst << "->_pre_impl (ctx);";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->_serialize_attributes ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->_serialize_content ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->_post_impl ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "if (!this->_end_element ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->post ();";

            os << endl
               << "if (this->" << inst << "->_error_type ())" << endl
               << "this->" << inst << "->_copy_error (ctx);"
               << endl;

            os << "if (ctx.error_type ())" << endl
               << "return;";
          }

          os << "}"; // if (inst)

          if (min == 0 || max != 1)
          {
            os << "}";

            if (!exceptions)
            {
              os << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl;
            }
          }
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          UnsignedLong min (a.contained_particle ().min ());
          UnsignedLong max (a.contained_particle ().max ());

          if (min == 0 && max == 1)
          {
            os << "if (this->" << epresent (a) << " ())";

          }
          else if (max != 1)
          {
            os << "while (this->" << enext (a) << " ())";
          }

          os << "{";

          if (stl)
          {
            os << "::std::string ns, name;";

            if (exceptions)
            {
              os << "this->" << ename (a) << " (ns, name);"
                 << endl
                 << "if (ns.empty ())" << endl
                 << "this->_start_element (name.c_str ());"
                 << "else" << endl
                 << "this->_start_element (ns.c_str (), name.c_str ());"
                 << endl
                 << "this->" << eserialize (a) << " ();"
                 << "this->_end_element ();";
            }
            else
            {
              os << "this->" << ename (a) << " (ns, name);"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "if (ns.empty ())"
                 << "{"
                 << "if (!this->_start_element (name.c_str ()))" << endl
                 << "return;"
                 << "}"
                 << "else"
                 << "{"
                 << "if (!this->_start_element (ns.c_str (), " <<
                "name.c_str ()))" << endl
                 << "return;"
                 << "}"
                 << "this->" << eserialize (a) << " ();"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "if (!this->_end_element ())" << endl
                 << "return;";
            }
          }
          else
          {
            os << "const char* ns = 0;"
               << "const char* name;"
               << "bool free;";

            if (exceptions)
            {
              os << "this->" << ename (a) << " (ns, name, free);"
                 << endl
                 << "::xsde::cxx::string auto_ns, auto_name;"
                 << "if (free)"
                 << "{"
                 << "auto_ns.attach (const_cast< char* > (ns));"
                 << "auto_name.attach (const_cast< char* > (name));"
                 << "}"
                 << "if (ns == 0 || *ns == '\\0')" << endl
                 << "this->_start_element (name);"
                 << "else" << endl
                 << "this->_start_element (ns, name);"
                 << endl
                 << "this->" << eserialize (a) << " ();"
                 << "this->_end_element ();";
            }
            else
            {
              os << "this->" << ename (a) << " (ns, name, free);"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "bool r;"
                 << "if (ns == 0 || *ns == '\\0')" << endl
                 << "r = this->_start_element (name);"
                 << "else" << endl
                 << "r = this->_start_element (ns, name);"
                 << endl
                 << "if (free)"
                 << "{"
                 << "delete[] ns;"
                 << "delete[] name;"
                 << "}"
                 << "if (!r)" << endl
                 << "return;"
                 << endl
                 << "this->" << eserialize (a) << " ();"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "if (!this->_end_element ())" << endl
                 << "return;";
            }
          }

          os << "}";

          if (!exceptions && (min == 0 || max != 1))
          {
            os << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;
          }
        }

      };

      struct Attribute: Traversal::Attribute,
                        Traversal::AnyAttribute,
                        protected virtual Context
      {
        Attribute (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Attribute& a)
        {
          if (a.optional ())
          {
            os << "if (this->" << epresent (a) << " ())"
               << "{";
          }

          String const& name (ename (a));
          String const& inst (emember (a));
          String const& ret (ret_type (a.type ()));
          String const& arg (arg_type (a.type ()));

          os << "if (this->" << inst << ")"
             << "{";

          if (exceptions)
          {
            if (ret == L"void")
              os << "this->" << name << " ();"
                 << "this->" << inst << "->pre ();";
            else
              os << arg << " r = this->" << name << " ();"
                 << "this->" << inst << "->pre (r);";

            if (a.qualified ())
              os << "this->_start_attribute (\"" << a.namespace_ ().name () <<
                "\", \"" << a.name () << "\");";
            else
              os << "this->_start_attribute (\"" << a.name () << "\");";

            os << "this->" << inst << "->_pre_impl (ctx);"
               << "this->" << inst << "->_serialize_content ();"
               << "this->" << inst << "->_post_impl ();"
               << "this->_end_attribute ();"
               << "this->" << inst << "->post ();";
          }
          else
          {
            if (ret == L"void")
              os << "this->" << name << " ();";
            else
              os << arg << " r = this->" << name << " ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            if (ret == L"void")
              os << "this->" << inst << "->pre ();";
            else
              os << "this->" << inst << "->pre (r);";

            os << endl
               << "if (this->" << inst << "->_error_type ())" << endl
               << "this->" << inst << "->_copy_error (ctx);"
               << endl;

            os << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            if (a.qualified ())
              os << "if (!this->_start_attribute (\"" <<
                a.namespace_ ().name () << "\", \"" <<
                a.name () << "\"))" << endl
                 << "return;"
                 << endl;
            else
              os << "if (!this->_start_attribute (\"" <<
                a.name () << "\"))" << endl
                 << "return;"
                 << endl;

            os << "this->" << inst << "->_pre_impl (ctx);";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->_serialize_content ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->_post_impl ();";

            os << endl
               << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;

            os << "if (!this->_end_attribute ())" << endl
               << "return;"
               << endl;

            os << "this->" << inst << "->post ();";

            os << endl
               << "if (this->" << inst << "->_error_type ())" << endl
               << "this->" << inst << "->_copy_error (ctx);"
               << endl;

            os << "if (ctx.error_type ())" << endl
               << "return;";
          }

          os << "}"; // if (inst)

          if (a.optional ())
          {
            os << "}";

            if (!exceptions)
            {
              os << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl;
            }
          }
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          os << "while (this->" << enext (a) << " ())"
             << "{";

          if (stl)
          {
            os << "::std::string ns, name;";

            if (exceptions)
            {
              os << "this->" << ename (a) << " (ns, name);"
                 << endl
                 << "if (ns.empty ())" << endl
                 << "this->_start_attribute (name.c_str ());"
                 << "else" << endl
                 << "this->_start_attribute (ns.c_str (), name.c_str ());"
                 << endl
                 << "this->" << eserialize (a) << " ();"
                 << "this->_end_attribute ();";
            }
            else
            {
              os << "this->" << ename (a) << " (ns, name);"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "if (ns.empty ())"
                 << "{"
                 << "if (!this->_start_attribute (name.c_str ()))" << endl
                 << "return;"
                 << "}"
                 << "else"
                 << "{"
                 << "if (!this->_start_attribute (ns.c_str (), " <<
                "name.c_str ()))" << endl
                 << "return;"
                 << "}"
                 << "this->" << eserialize (a) << " ();"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "if (!this->_end_attribute ())" << endl
                 << "return;";
            }
          }
          else
          {
            os << "const char* ns = 0;"
               << "const char* name;"
               << "bool free;";

            if (exceptions)
            {
              os << "this->" << ename (a) << " (ns, name, free);"
                 << endl
                 << "::xsde::cxx::string auto_ns, auto_name;"
                 << "if (free)"
                 << "{"
                 << "auto_ns.attach (const_cast< char* > (ns));"
                 << "auto_name.attach (const_cast< char* > (name));"
                 << "}"
                 << "if (ns == 0 || *ns == '\\0')" << endl
                 << "this->_start_attribute (name);"
                 << "else" << endl
                 << "this->_start_attribute (ns, name);"
                 << endl
                 << "this->" << eserialize (a) << " ();"
                 << "this->_end_attribute ();";
            }
            else
            {
              os << "this->" << ename (a) << " (ns, name, free);"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "bool r;"
                 << "if (ns == 0 || *ns == '\\0')" << endl
                 << "r = this->_start_attribute (name);"
                 << "else" << endl
                 << "r = this->_start_attribute (ns, name);"
                 << endl
                 << "if (free)"
                 << "{"
                 << "delete[] ns;"
                 << "delete[] name;"
                 << "}"
                 << "if (!r)" << endl
                 << "return;"
                 << endl
                 << "this->" << eserialize (a) << " ();"
                 << endl
                 << "if (ctx.error_type ())" << endl
                 << "return;"
                 << endl
                 << "if (!this->_end_attribute ())" << endl
                 << "return;";
            }
          }

          os << "}";

          if (!exceptions)
          {
            os << "if (ctx.error_type ())" << endl
               << "return;"
               << endl;
          }
        }
      };

      // Callbacks.
      //
      struct CompositorCallback: Traversal::All,
                                 Traversal::Choice,
                                 Traversal::Sequence,
                                 protected virtual Context
      {
        CompositorCallback (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::All& a)
        {
          // For the all compositor, maxOccurs=1 and minOccurs={0,1}.
          //
          if (a.contained_compositor ().min () == 0)
          {
            String const& s (ename (scope (a)));

            os << "bool " << s << "::" << endl
               << epresent (a) << " ()"
               << "{"
               << "return false;"
               << "}";
          }

          Traversal::All::traverse (a);
        }

        virtual Void
        traverse (SemanticGraph::Choice& c)
        {
          if (c.contains_begin () != c.contains_end ())
          {
            UnsignedLong min;
            UnsignedLong max;

            if (c.contained_compositor_p ())
            {
              min = c.contained_compositor ().min ();
              max = c.contained_compositor ().max ();
            }
            else
            {
              min = c.contained_particle ().min ();
              max = c.contained_particle ().max ();
            }

            if (min == 0)
            {
              String const& s (ename (scope (c)));

              if (max == 1)
              {
                os << "bool " << s << "::" << endl
                   << epresent (c) << " ()"
                   << "{"
                   << "return false;"
                   << "}";
              }
              else
              {
                os << "bool " << s << "::" << endl
                   << enext (c) << " ()"
                   << "{"
                   << "return false;"
                   << "}";
              }

              os << s << "::" << earm_tag (c) << " " << s << "::" << endl
                 << earm (c) << " ()"
                 << "{"
                 << "return " << etag (c.contains_begin ()->particle ()) << ";"
                 << "}";
            }

            Traversal::Choice::traverse (c);
          }
        }

        virtual Void
        traverse (SemanticGraph::Sequence& s)
        {
          UnsignedLong min;
          UnsignedLong max;

          if (s.contained_compositor_p ())
          {
            min = s.contained_compositor ().min ();
            max = s.contained_compositor ().max ();
          }
          else
          {
            min = s.contained_particle ().min ();
            max = s.contained_particle ().max ();
          }

          if (min == 0)
          {
            String const& es (ename (scope (s)));

            if (max == 1)
            {
              os << "bool " << es << "::" << endl
                 << epresent (s) << " ()"
                 << "{"
                 << "return false;"
                 << "}";
            }
            else
            {
              os << "bool " << es << "::" << endl
                 << enext (s) << " ()"
                 << "{"
                 << "return false;"
                 << "}";
            }
          }

          Traversal::Sequence::traverse (s);
        }

      private:
        SemanticGraph::Scope&
        scope (SemanticGraph::Compositor& c)
        {
          SemanticGraph::Compositor* root (&c);

          while (root->contained_particle_p ())
            root = &root->contained_particle ().compositor ();

          return dynamic_cast<SemanticGraph::Scope&> (
            root->contained_compositor ().container ());
        }
      };

      struct ParticleCallback: Traversal::Element,
                               Traversal::Any,
                               protected virtual Context
      {
        ParticleCallback (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Element& e)
        {
          UnsignedLong min (e.contained_particle ().min ());
          UnsignedLong max (e.contained_particle ().max ());

          String const& s (ename (e.scope ()));

          if (min == 0)
          {
            if (max == 1)
            {
              os << "bool " << s << "::" << endl
                 << epresent (e) << " ()"
                 << "{"
                 << "return false;"
                 << "}";
            }
            else
            {
              os << "bool " << s << "::" << endl
                 << enext (e) << " ()"
                 << "{"
                 << "return false;"
                 << "}";
            }
          }

          // The callback is non-pure-virtual only if the return type
          // is void.
          //
          if (ret_type (e.type ()) == L"void")
          {
            os << "void " << s << "::" << endl
               << ename (e) << " ()"
               << "{"
               << "}";
          }
        }

        virtual Void
        traverse (SemanticGraph::Any& a)
        {
          UnsignedLong min (a.contained_particle ().min ());
          UnsignedLong max (a.contained_particle ().max ());

          if (min == 0)
          {
            String const& s (ename (a.scope ()));

            if (max == 1)
            {
              os << "bool " << s << "::" << endl
                 << epresent (a) << " ()"
                 << "{"
                 << "return false;"
                 << "}";
            }
            else
            {
              os << "bool " << s << "::" << endl
                 << enext (a) << " ()"
                 << "{"
                 << "return false;"
                 << "}";
            }

            if (stl)
            {
              os << "void " << s << "::" << endl
                 << ename (a) << " (::std::string&, ::std::string&)"
                 << "{"
                 << "}";
            }
            else
            {
              os << "void " << s << "::" << endl
                 << ename (a) << " (const char*&, const char*&, bool&)"
                 << "{"
                 << "}";
            }

            os << "void " << s << "::" << endl
               << eserialize (a) << " ()"
               << "{"
               << "}";
          }
        }
      };

      struct AttributeCallback: Traversal::Attribute,
                                Traversal::AnyAttribute,
                                protected virtual Context
      {
        AttributeCallback (Context& c)
            : Context (c)
        {
        }

        virtual Void
        traverse (SemanticGraph::Attribute& a)
        {
          if (a.optional ())
          {
            os << "bool " << ename (a.scope ()) << "::" << endl
               << epresent (a) << " ()"
               << "{"
               << "return false;"
               << "}";
          }

          // The callback is non-pure-virtual only if the return type
          // is void.
          //
          if (ret_type (a.type ()) == L"void")
          {
            os << "void " << ename (a.scope ()) << "::" << endl
               << ename (a) << " ()"
               << "{"
               << "}";
          }
        }

        virtual Void
        traverse (SemanticGraph::AnyAttribute& a)
        {
          String const& s (ename (a.scope ()));

          os << "bool " << s << "::" << endl
             << enext (a) << " ()"
             << "{"
             << "return false;"
             << "}";

          if (stl)
          {
            os << "void " << s << "::" << endl
               << ename (a) << " (::std::string&, ::std::string&)"
               << "{"
               << "}";
          }
          else
          {
            os << "void " << s << "::" << endl
               << ename (a) << " (const char*&, const char*&, bool&)"
               << "{"
               << "}";
          }

          os << "void " << s << "::" << endl
             << eserialize (a) << " ()"
             << "{"
             << "}";
        }
      };

      //
      //
      struct Complex : Traversal::Complex,
                       protected virtual Context
      {
        Complex (Context& c)
            : Context (c),
              compositor_callback_ (c),
              particle_callback_ (c),
              attribute_callback_ (c),
              compositor_ (c),
              particle_ (c),
              attribute_ (c)
        {
          // Callback.
          //
          contains_compositor_callback_ >> compositor_callback_;
          compositor_callback_ >> contains_particle_callback_;
          contains_particle_callback_ >> compositor_callback_;
          contains_particle_callback_ >> particle_callback_;

          names_attribute_callback_ >> attribute_callback_;

          // Serialization code.
          //
          contains_compositor_ >> compositor_;
          compositor_ >> contains_particle_;
          contains_particle_ >> compositor_;
          contains_particle_ >> particle_;

          names_attribute_ >> attribute_;
        }

        virtual Void
        traverse (Type& c)
        {
          Boolean he (has<Traversal::Element> (c));
          Boolean ha (has<Traversal::Attribute> (c));

          Boolean hae (has_particle<Traversal::Any> (c));
          Boolean haa (has<Traversal::AnyAttribute> (c));

          String const& arg (arg_type (c));
          Boolean same (c.inherits_p () &&
                        arg == arg_type (c.inherits ().base ()));

          if (!(he || ha || hae || haa || (!same && arg == L"void")))
            return;

          String const& name (ename (c));

          os << "// " << name << endl
             << "//" << endl
             << endl;

          // pre
          //
          if (!same && arg == L"void")
          {
            os << "void " << name << "::" << endl
               << "pre ()"
               << "{"
               << "}";
          }

          // Member callbacks.
          //
          if (!restriction_p (c))
          {
            if (ha || haa)
              names (c, names_attribute_callback_);

            if (he || hae)
              contains_compositor (c, contains_compositor_callback_);
          }

          // If we are validating, the rest is generated elsewere.
          //
          if (validation)
            return;

          // Don't use restriction_p here since we don't want special
          // treatment of anyType.
          //
          Boolean restriction (
            c.inherits_p () &&
            c.inherits ().is_a<SemanticGraph::Restricts> ());

          // _serialize_attributes
          //
          if (ha || haa)
          {
            os << "void " << name << "::" << endl
               << "_serialize_attributes ()"
               << "{";

            // We need context for wildcards only if we are using error
            // codes.
            //
            if (ha || !exceptions)
              os << "::xsde::cxx::serializer::context& ctx = this->_context ();"
                 << endl;

            // Use unqualified name of our base.
            //
            if (c.inherits_p () && !restriction)
            {
              os << ename (c.inherits ().base ()) <<
                "::_serialize_attributes ();"
                 << endl;

              if (!exceptions)
              {
                os << "if (ctx.error_type ())" << endl
                   << "return;"
                   << endl;
              }
            }

            names (c, names_attribute_);

            os << "}";
          }

          // _serialize_content
          //
          if (he || hae)
          {
            os << "void " << name << "::" << endl
               << "_serialize_content ()"
               << "{";

            // We need context for wildcards only if we are using error
            // codes.
            //
            if (he || !exceptions)
              os << "::xsde::cxx::serializer::context& ctx = this->_context ();"
                 << endl;

            // Use unqualified name of our base.
            //
            if (c.inherits_p () && !restriction)
            {
              os << ename (c.inherits ().base ()) <<
                "::_serialize_content ();"
                 << endl;

              if (!exceptions)
              {
                os << "if (ctx.error_type ())" << endl
                   << "return;"
                   << endl;
              }
            }

            contains_compositor (c, contains_compositor_);

            os << "}";
          }
        }

      private:
        //
        //
        CompositorCallback compositor_callback_;
        ParticleCallback particle_callback_;
        Traversal::ContainsCompositor contains_compositor_callback_;
        Traversal::ContainsParticle contains_particle_callback_;

        AttributeCallback attribute_callback_;
        Traversal::Names names_attribute_callback_;

        //
        //
        Compositor compositor_;
        Particle particle_;
        Traversal::ContainsCompositor contains_compositor_;
        Traversal::ContainsParticle contains_particle_;

        Attribute attribute_;
        Traversal::Names names_attribute_;
      };
    }

    Void
    generate_serializer_source (Context& ctx)
    {
      Traversal::Schema schema;

      Traversal::Sources sources;
      Traversal::Names schema_names;

      Namespace ns (ctx);
      Traversal::Names names;

      schema >> sources >> schema;
      schema >> schema_names >> ns >> names;

      List list (ctx);
      Union union_ (ctx);
      Complex complex (ctx);
      Enumeration enumeration (ctx);

      names >> list;
      names >> union_;
      names >> complex;
      names >> enumeration;

      schema.dispatch (ctx.schema_root);
    }
  }
}

