/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * NOTES:
 * 
 * int QListBox::currentItem() returns -1 if no item is selected.
 * Always check the index before using it as either listbox or
 * list index. When removing listbox items, the highlighted(int)
 * signal gets emitted which may be -1, too.
 * 
 * The current playlist can be modified directly through the
 * pointer.
 * 
 * Quite a few functions are guarded by a null pointer check
 * against the playlist pointer just in case the playlist dialog
 * has not been set up correctly after creation.
 */

#include <sidplay/sidtune.h>

#include <qapplication.h>
#include <qdir.h>
#include <qfiledialog.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qstring.h>

#include "GlobalIcon.h"
#include "Playlist.h"
#include "PlaylistDialog.h"

PlaylistDialog::PlaylistDialog( QWidget* parent,  const char* name, bool modal, WFlags fl )
: PlaylistDialogData( parent, name, modal, fl )
{
    setIcon(*mainIcon);
    setMinimumSize(width(),height());

    // Collect some data that will be used upon resize events.
    // Could automate this by using a list.
    widgetOffs.prevButton.set(this,prevButton);
    widgetOffs.startButton.set(this,startButton);
    widgetOffs.nextButton.set(this,nextButton);
    widgetOffs.listPosLabel.set(this,listPosLabel);
    widgetOffs.subtuneLabel.set(this,subtuneLabel);
    widgetOffs.playtimeLabel.set(this,playtimeLabel);
    widgetOffs.fadeoutLabel.set(this,fadeoutLabel);
    // List box adapts size instead of position.
    widgetOffs.playListBox.set( width()-playListBox->width(), height()-playListBox->height() );
    
    playlist = 0;  // start with no list (*DANGER*)
    
    clear();
}

PlaylistDialog::~PlaylistDialog()
{
    // DO NOT...
    //
    // ... delete 'playlist' here. It is just a copy of a pointer
    // to a list maintained by some other component.
    //
    // ... delete child widgets, Qt does it all for us.
}

void PlaylistDialog::resizeEvent(QResizeEvent*)
{
    int w = width();
    int h = height();
    
    listPosLabel->move(w-widgetOffs.listPosLabel.x(),h-widgetOffs.listPosLabel.y());
    subtuneLabel->move(w-widgetOffs.subtuneLabel.x(),h-widgetOffs.subtuneLabel.y());
    fadeoutLabel->move(w-widgetOffs.fadeoutLabel.x(),h-widgetOffs.fadeoutLabel.y());
    playtimeLabel->move(w-widgetOffs.playtimeLabel.x(),h-widgetOffs.playtimeLabel.y());
    prevButton->move(prevButton->x(),h-widgetOffs.prevButton.y());
    startButton->move(startButton->x(),h-widgetOffs.startButton.y());
    nextButton->move(nextButton->x(),h-widgetOffs.nextButton.y());
    
    playListBox->setGeometry(0,0,w-widgetOffs.playListBox.x(),h-widgetOffs.playListBox.y());
}

void PlaylistDialog::clear()
{
    playListBox->clear();
    listPosLabel->setText(QString("#%1").arg( (int)0 ));
}

void PlaylistDialog::takeList(Playlist* inList)
{
    QApplication::setOverrideCursor( Qt::waitCursor );
    playlist = inList;
    playListBox->clear();
    // Copy each song title into listbox.
    QString tmpBaseDir = playlist->baseDir;
    QListIterator<PlaylistItem> it(playlist->list);
    for ( it.toFirst(); it.current(); ++it ) 
    {
        playListBox->insertItem(it.current()->titleString);
    }
    if ( playListBox->count() )
    {
        playListBox->setCurrentItem(0);
    }
    QApplication::restoreOverrideCursor();
}

void PlaylistDialog::add(PlaylistItem* item)
{
    if ( playlist==0 )
        return;
    playListBox->insertItem(item->titleString);
}

void PlaylistDialog::showPlaylistItem(const PlaylistItem* pli, int i)
{
    listPosLabel->setText(QString("#%1").arg(i+1));
    QString tmp = tr("Subtune: \n%1");
    subtuneLabel->setText(tmp.arg(pli->subtune));
    tmp = tr("Fadeout: \n%1");
    fadeoutLabel->setText(tmp.arg(pli->fadeout));
    tmp = tr("Playtime: \n");
    int rest = pli->time;
    int hours = rest/3600;
    rest -= hours*3600;
    int mins = rest/60;
    rest -= mins*60;
    int secs = rest;
    QString timeTmp;
    if ( hours>0 )
    {
        timeTmp.sprintf("%d:",hours);
        tmp += timeTmp;
    }
    timeTmp.sprintf("%d:%02d",mins,secs);
    tmp += timeTmp;
    playtimeLabel->setText(tmp);
}

// slot (not used)
/*
void PlaylistDialog::deleteButtonClicked()
{
    if ( playlist==0 )
        return;
    if ( playListBox->count() && playListBox->currentItem()>=0 )
    {
        int i = playListBox->currentItem();
        QMessageBox mb( "SIDPLAY",
                        "Do you really want to delete current entry?",
                        QMessageBox::Information,
                        QMessageBox::Yes,
                        QMessageBox::No | QMessageBox::Default | QMessageBox::Escape,
                        0);
        mb.setButtonText( QMessageBox::Yes, "Yes" );
        mb.setButtonText( QMessageBox::No, "No" );
        switch( mb.exec() ) 
        {
         case QMessageBox::Yes:
            break;
         case QMessageBox::No:
            return;
        }
        // We must first modify the list, so the listbox s
        // signals can trigger work on sane list contents.
        playlist->remove(i);
        playListBox->removeItem(i);  // emits itemhighlighted(int)
    }
} */

// slot
void PlaylistDialog::updateListPos(uint index)
{
    playListBox->changeItem(playlist->list.at(index)->titleString,index);
}

// slot
void PlaylistDialog::playlistItemSelected(int index)
{
    if ( playlist==0 )
        return;
    if ( index >= 0 )
    {
        playlist->setCurrentPlayPos(index);
        curPlaylistItem = *(playlist->list.at(index));
        emit playlistPlayRequest(curPlaylistItem);
    }
}

// slot
void PlaylistDialog::newListPos(uint index)
{
    playListBox->blockSignals(true);
    playListBox->setCurrentItem(index);
    PlaylistItem* pli = playlist->list.at(index);
    showPlaylistItem(pli,index);
    playListBox->blockSignals(false);
}
 
// slot
void PlaylistDialog::deleteListPos(uint index)
{
    playListBox->blockSignals(true);
    uint cur = playListBox->currentItem();
    playListBox->removeItem(index);
    if ( cur==index )
        playlistItemHighlighted(index);
    playListBox->blockSignals(false);
}
 
// slot
void PlaylistDialog::playlistItemHighlighted(int index)
{
    if ( playlist==0 )
        return;
    if ( index >= 0 )
    {
        PlaylistItem* pli = playlist->list.at(index);
        showPlaylistItem(pli,index);
    }
}

// slot
void PlaylistDialog::startButtonClicked()
{
    if ( playlist==0 )
        return;
    emit playlistPlayNewPosRequest(0);
}
 
// slot
void PlaylistDialog::nextButtonClicked()
{
    if ( playlist==0 )
        return;
    emit playlistPlayNewPosRequest(1);
}

// slot
void PlaylistDialog::prevButtonClicked()
{
    if ( playlist==0 )
        return;
    emit playlistPlayNewPosRequest(-1);
}
