// TabulatedConstituent:  definition of a constituent that uses tabulated
// equilibrium arguments and node factors.
// Last modified by DWF 1999-10-17

/*
    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

TabulatedConstituent::Argfac::Argfac () {
  arg = Radians (0.0);
  nod = 1.0;
}

TabulatedConstituent::Argfac::Argfac (Angle in_arg, double in_nod) {
  arg = in_arg;
  nod = in_nod;
}

TabulatedConstituent::Argfac &
TabulatedConstituent::ArgfacVector::operator[] (Year in_year) {
  return a[in_year.val() - globalfirstyear];
}

TabulatedConstituent::TabulatedConstituent () {
  myname = "Anonymous";
  readargs = readnods = 0;
  myspeed = DegreesPerHour (0.0);
  myfirstvalidyear = Year(globalfirstyear);
  mylastvalidyear = Year(globallastyear);
}

TabulatedConstituent::TabulatedConstituent (Dstr &in_name) {
  myname = in_name;
  readargs = readnods = 0;
  myspeed = DegreesPerHour (0.0);
  myfirstvalidyear = Year(globalfirstyear);
  mylastvalidyear = Year(globallastyear);
}

TabulatedConstituent::TabulatedConstituent (Constituent &in_con) {
  (*this) = in_con;
}

TabulatedConstituent &
TabulatedConstituent::operator= (Constituent &in_con) {
  myname = in_con.name();
  myspeed = in_con.speed();
  readargs = readnods = 0;
  myfirstvalidyear = in_con.firstvalidyear();
  mylastvalidyear = in_con.lastvalidyear();
  for (unsigned looper=myfirstvalidyear.val(); looper<=mylastvalidyear.val();
  looper++) {
    Year ty (looper);
    myargfacs[ty].arg = in_con.arg(ty);
    myargfacs[ty].nod = in_con.nod(ty);
  }
  return (*this);
}

void
TabulatedConstituent::speed (Speed in_speed) {
  myspeed = in_speed;
}

// This should be redundant -- workaround for bug in
// Sun WorkShop Compiler C++ SPARC Version 5.000
// Error: Could not find a match for TabulatedConstituent::speed(DegreesPerHour).
void
TabulatedConstituent::speed (DegreesPerHour in_speed) {
  myspeed = in_speed;
}

Speed TabulatedConstituent::speed () {
  return myspeed;
}

Year
TabulatedConstituent::firstvalidyear() {
  return myfirstvalidyear;
}

Year
TabulatedConstituent::lastvalidyear() {
  return mylastvalidyear;
}

void
TabulatedConstituent::check_valid (Year in_year) {
  if (readargs != readnods)
    barf (INVALID_CONSTITUENT);
  if (in_year < myfirstvalidyear ||
      in_year > mylastvalidyear) {
    Dstr details ("The years supported by the harmonics file are ");
    details += (int)myfirstvalidyear.val();
    details += " through ";
    details += (int)mylastvalidyear.val();
    details += ".\n";
    details += "The offending year was ";
    details += (int)in_year.val();
    details += ".\n";
    barf (YEAR_NOT_IN_TABLE, details);
  }
}

Angle
TabulatedConstituent::arg (Year in_year) {
  check_valid (in_year);
  return myargfacs[in_year].arg;
}

double
TabulatedConstituent::nod (Year in_year) {
  check_valid (in_year);
  return myargfacs[in_year].nod;
}

void
TabulatedConstituent::readnameandspeed (FILE *fp) {
  myname.scan (fp);
  double tempspeed;
  if (fscanf (fp, "%lf", &tempspeed) != 1) {
    Dstr details ("Can't read speed of constituent.");
    barf (CORRUPT_HARMONICS_FILE, details);
  }
  speed (DegreesPerHour (tempspeed));
}

void
TabulatedConstituent::readnameandspeed (Dstr &in_name_speed) {
  double tempspeed;
  in_name_speed /= myname;
  assert (sscanf (in_name_speed.aschar(), "%lf", &tempspeed) == 1);
  speed (DegreesPerHour (tempspeed));
}

void TabulatedConstituent::readarg (FILE *fp, Year first_year, Year last_year)
{
  Dstr junk;
  if (readargs)
    barf (DOUBLE_READ);
  readargs = 1;
  junk.scan (fp);
  if (junk.isNull())
    barf (END_OF_FILE);
  for (unsigned looper=first_year.val(); looper<=last_year.val(); looper++) {
    double arg;
    if (fscanf (fp, "%lf", &arg) != 1) {
      Dstr details ("Can't read equilibrium argument of constituent.");
      barf (CORRUPT_HARMONICS_FILE, details);
    }
    Year ty (looper);
    myargfacs[ty].arg = Degrees(arg);
  }
  if (first_year > myfirstvalidyear)
    myfirstvalidyear = first_year;
  if (last_year < mylastvalidyear)
    mylastvalidyear = last_year;
}

void TabulatedConstituent::readnod (FILE *fp, Year first_year, Year last_year)
{
  Dstr junk;
  if (readnods)
    barf (DOUBLE_READ);
  readnods = 1;
  junk.scan (fp);
  if (junk.isNull())
    barf (END_OF_FILE);
  for (unsigned looper=first_year.val(); looper<=last_year.val(); looper++) {
    double arg;
    if (fscanf (fp, "%lf", &arg) != 1) {
      Dstr details ("Can't read node factor of constituent.");
      barf (CORRUPT_HARMONICS_FILE, details);
    }
    Year ty (looper);
    myargfacs[ty].nod = arg;
  }
  if (first_year > myfirstvalidyear)
    myfirstvalidyear = first_year;
  if (last_year < mylastvalidyear)
    mylastvalidyear = last_year;
}

// Add a constituent to this one.
TabulatedConstituent &
TabulatedConstituent::operator += (Constituent &in_con) {
  if (readargs != readnods)
    barf (INVALID_CONSTITUENT);
  myspeed += in_con.speed();
  if (in_con.firstvalidyear() > myfirstvalidyear)
    myfirstvalidyear = in_con.firstvalidyear();
  if (in_con.lastvalidyear() < mylastvalidyear)
    mylastvalidyear = in_con.lastvalidyear();
  for (unsigned looper=myfirstvalidyear.val(); looper<=mylastvalidyear.val();
  looper++) {
    Year ty(looper);
    myargfacs[ty].arg += in_con.arg(ty);
    myargfacs[ty].nod *= in_con.nod(ty);
  }
  return (*this);
}

// Multiply this constituent by a factor.
TabulatedConstituent &
TabulatedConstituent::operator *= (double factor) {
  if (readargs != readnods)
    barf (INVALID_CONSTITUENT);
  myspeed *= factor;
  for (unsigned looper=myfirstvalidyear.val(); looper<=mylastvalidyear.val();
  looper++) {
    Year ty(looper);
    myargfacs[ty].arg *= factor;
    myargfacs[ty].nod = pow (myargfacs[ty].nod, fabs (factor));
  }
  return (*this);
}

TabulatedConstituent &
TabulatedConstituent::operator *= (int factor) {
  return ((*this) *= (double)factor);
}

TabulatedConstituent::~TabulatedConstituent() {
}
