/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gui/SMessageBar.h"

#include "gui/Main.h"
#include "swidget/SIconFactory.h"
#include "stoolkit/SCluster.h"


/**
 * a button panel is just a panel of labels.
 */
SMessageBar::SMessageBar (const SProperties props)
{
  xmanagerCludge = true;
  SColor fg = SColor(props["yudit.label.foreground"]);
  SColor tfg = SColor(props["yudit.title.foreground"]);
  SString font = props["yudit.default.font"];
  SString fontSizeString=props["yudit.default.fontsize"];

  double fontSize=0;
  fontSizeString.append ((char)0);
  sscanf (fontSizeString.array(), "%lf", &fontSize);
  if (fontSize < 2.0) fontSize = 2.0;

  /* create the labels and set the preferred size */
  SLabel *l;

  l = new SLabel (translate ("Glyph Info:"));
  l->setFont(font, fontSize);
  l->setForeground (tfg);
  labels[SS_GLYPH_TITLE] = l;

  l = new SLabel (translate (""));
  l->setFont(font, fontSize);
  l->setForeground (fg);
  labels[SS_GLYPH_INFO] = l;

  l = new SLabel (translate ("Font Size:"));
  l->setFont(font, fontSize);
  l->setForeground (tfg);
  labels[SS_FONT_SIZE_TITLE] = l;

  l = new SLabel (translate (""));
  l->setFont(font, fontSize);
  l->setForeground (fg);
  labels[SS_FONT_SIZE] = l;

  l = new SLabel (translate ("Line:"));
  l->setFont(font, fontSize);
  l->setForeground (tfg);
  labels[SS_ROW_TITLE] = l;

  l =  new SLabel ("");
  l->setFont(font, fontSize);
  l->setForeground (fg);
  labels[SS_ROW] = l;

  l = new SLabel (translate ("Column:"));
  l->setFont(font, fontSize);
  l->setForeground (tfg);
  labels[SS_COLUMN_TITLE] = l;

  l =  new SLabel ("");
  l->setFont(font, fontSize);
  l->setForeground (fg);
  labels[SS_COLUMN] = l;

  unsigned int p=0;
  unsigned int h=0;
  for (unsigned int i=0; i<(unsigned int) SS_MAX; i++)
  {
    unsigned int ww = 0;
    switch (i)
    {
    case SS_FONT_SIZE:
      ww = 50;
      break;
    case SS_ROW:
      ww = 50;
      break;
    case SS_COLUMN:
      ww = 50;
      break;
    case SS_GLYPH_INFO:
      ww = 600;
      break;
    default:
      ww = labels[i]->getPreferredSize().width;
    }
    unsigned int hh = labels[i]->getPreferredSize().height;
    labels[i]->resize(SDimension (ww, hh));
    labels[i]->move(SLocation ((int)p, 0));
    add (labels[i]);
    p+= ww +2;
    if (hh > h) h = hh;
  }
  preferredSize = SDimension (p, h);
}
/**
 * This routine was introduced to make this
 * usable on peecees with windows.
 * Seems like XManager has some bug that they dont
 * fix. On unix you could in fact get rid of this whole  method.
 * The presiously set size got lost. I dont really know why XManager is so
 * popular....
 */
void 
SMessageBar::resize(const SDimension& d)
{
  SPanel::resize (d);
  /* we only need to do it once. */
  if (!xmanagerCludge)
  {
    return;
  }
  xmanagerCludge = false;
  unsigned int p=0;
  unsigned int h=0;
  for (unsigned int i=0; i<(unsigned int) SS_MAX; i++)
  {
    unsigned int ww = 0;
    switch (i)
    {
    case SS_FONT_SIZE:
      ww = 50;
      break;
    case SS_ROW:
      ww = 50;
      break;
    case SS_COLUMN:
      ww = 50;
      break;
    case SS_GLYPH_INFO:
      ww = 600;
      break;
    default:
      ww = labels[i]->getPreferredSize().width;
    }
    unsigned int hh = labels[i]->getPreferredSize().height;
    labels[i]->resize(SDimension (ww, hh));
    labels[i]->move(SLocation ((int)p, 0));
    // Seems resize is enough to work around the bug.
    /* No need to labels[i]->move(SLocation ((int)p, 0)); */
    p+= ww +2;
    if (hh > h) h = hh;
  }
}

SMessageBar::~SMessageBar ()
{
}


void
SMessageBar::setRow (unsigned int row)
{
  char a[64];
  sprintf (a, "%u", row);
  SLabel* l = labels[SS_ROW];
  l->setText (a);
}

void
SMessageBar::setColumn (unsigned int column)
{
  char a[64];
  sprintf (a, "%u", column);
  SLabel* l = labels[SS_COLUMN];
  l->setText (a);
}

void
SMessageBar::setFontSize (double d)
{
  char a[64];
  sprintf (a, "%g", d);
  fontSize = d;
  SLabel* l = labels[SS_FONT_SIZE];
  l->setText (a);
}

/**
 * This one draws a glyph info in the form of 
 * compose   [decomp1 decomp2] - if compose will be saved.
 * [compose] decomp1 decomp2 - if decomp1 decomp.. will be saved
 * above P:displayed  is displayed presentation form is used.
 */
void
SMessageBar::setGlyphInfo (const SGlyph* g)
{
  SUniMap shaper ("shape");

  static char* shapes[] = {
     "isolated",
     "initial",
     "medial",
     "final",
  };

  static char* types[] = {
    "",   // 0 unknown
    "Lu", // 01 Lu Letter, Uppercase
    "Ll", // 02 Ll Letter, Lowercase
    "Lt", // 03 Lt Letter, Titlecase
    "Mn", // 04 Mn Mark, Non-Spacing
    "Mc", // 05 Mc Mark, Spacing Combining
    "Me", // 06 Me Mark, Enclosing
    "Nd", // 07 Nd Number, Decimal Digit
    "Nl", // 08 Nl Number, Letter
    "No", // 09 No Number, Other
    "Zs", // 0A Zs Separator, Space
    "Zl", // 0B Zl Separator, Line
    "Zp", // 0C Zp Separator, Paragraph
    "Cc", // 0D Cc Other, Control
    "Cf", // 0E Cf Other, Format
    "Cs", // 0F Cs Other, Surrogate
    "Co", // 10 Co Other, Private Use

// Informative Categories

    "Lm", // 12 Lm Letter, Modifier
    "Lo", // 13 Lo Letter, Other
    "Pc", // 14 Pc Punctuation, Connector
    "Pd", // 15 Pd Punctuation, Dash
    "Ps", // 16 Ps Punctuation, Open
    "Pe", // 17 Pe Punctuation, Close
    "Pi", // 18 Pi Punctuation, Initial quote(may behave like Ps or Pe depending on usage)
    "Pf", // 19 Pf Punctuation, Final quote (may behave like Ps or Pe depending on usage)
    "Po", // 1A Po Punctuation, Other
    "Sm", // 1B Sm Symbol, Math
    "Sc", // 1C Sc Symbol, Currency
    "Sk", // 1D Sk Symbol, Modifier
    "So" // 1E So Symbol, Other
  };
  
  SLabel* l = labels[SS_GLYPH_INFO];
  if (g==0)
  {
    l->setText("");
    return;
  }
  bool savecomp = g->useComposition;
  SS_UCS4 comp = g->getChar();
  SS_UCS4 shape = g->getShapedChar();
  SV_UCS4 decomp;
  for (unsigned int i=0; i<g->size(); i++)
  {
    decomp.append ((*g)[i]);
  }

  SString compStr;
  if (comp!=0)
  {
    unsigned int tp = g->getType();
    /* too much info */
    //if (tp < 0x1f && tp !=0)
    if (false)
    {
       compStr.append (types[tp]);
       compStr.append (" ");
    }
  }
  
  SString decompStr;

  /*
   * shape:xxxx [decomp]
   * In this case we received the shape. We wont modify shape.
   * In this case the shape is 
   * pre-determined and can not be changed
   */
  if (savecomp && decomp.size() == 0 && comp != 0 && shaper.isOK())
   /* check if comp can be a shape */
  {
    SV_UCS4 ret;
    SV_UCS4 v; v.append (comp);
    /* presentation form - get decomposition */
    unsigned int n = shaper.lift (v, 0, true, &ret);
    if (ret.size() > 0)
    {
      decompStr.append ("[");
      decompStr.append (unicodeValueOf (ret));
      decompStr.append ("]");

      SV_UCS4 forms;
      n = shaper.lift (ret, 0, false, &forms);
      if (forms.size()==4)
      {
        for (unsigned int i=0; i<forms.size(); i++)
        {
          if (forms[i] == comp) 
          {
            compStr.append (shapes[i]);
            compStr.append (":");
            compStr.append (unicodeValueOf(v));
            break;
          }
          if (i==3) /* never happens if decoder/encoder is ok */
          {
            compStr.append ("noshape:");
            compStr.append (unicodeValueOf(v));
          }
        }
      }
      else /* never happens if decoder/encoder is ok */
      {
        compStr.append ("moreshape:");
        compStr.append (unicodeValueOf(v));
      }
    }
    if (decompStr.size())
    {
      compStr.append (" ");
      compStr.append (decompStr);
      l->setText(compStr);
      return;
    }
  }
  /*
   * [shape:xxxx] decomp
   * In this case the shape is 
   * pre-determined and can not be changed
   */
  if (g->currentShape>=0) 
  {
    compStr.append ("[");
    compStr.append (shapes[(unsigned int)g->currentShape]);
    compStr.append (":");
    SV_UCS4 v; v.append (shape);
    compStr.append (unicodeValueOf (v));
    compStr.append ("]");

    /* here shape can be a simgle comp or decomp */
    if (decomp.size() > 0)
    {
       decompStr.append (unicodeValueOf (decomp));
    }
    else /* it is comp */
    {
       v.clear(); v.append (comp); /* decomposition is a signle char */
       decompStr.append (unicodeValueOf (v));
    }
    compStr.append (" ");
    compStr.append (decompStr);
    l->setText(compStr);
    return;
  }

  /*
   * decomp ....
   * In this case the shape is 
   * pre-determined and can not be changed
   */
  if (comp==0 && decomp.size() > 0)
  {
    if (g->getShapeArray())
    {
      decompStr.append ("noshape:");
      decompStr.append (unicodeValueOf (decomp));
    }
    else if (g->isCluster())
    {
      decompStr.append ("[cluster:");
      decompStr.append (unicodeValueOf (decomp));
      decompStr.append ("] ");
      decompStr.append (unicodeValueOf (g->getChars()));
    }
    else
    {
      decompStr.append (unicodeValueOf (decomp));
    }
    l->setText(decompStr);
    return;
  }

  /**
   * [xxxx] decomp
   */
  if (!savecomp)
  {
    compStr.append ("[");
    SS_UCS4 lig = comp;
    if (g->isCluster())
    {
      SString ligname ("ligature:");
      const char* chr = getLigatureScript (comp);
      if (chr)
      {
        ligname = SString(chr);
	ligname.append (":");
        lig = lig & 0xffff;
      }
      compStr.append (ligname);
    }
    SV_UCS4 v; v.append (lig);
    compStr.append (unicodeValueOf (v));

    compStr.append ("] ");
    if (g->isCluster())
    {
      // Too much info
      //compStr.append ("[cluster:");
      //compStr.append (unicodeValueOf (decomp));
      //compStr.append ("] ");
      compStr.append (unicodeValueOf (g->getChars()));
    }
    else
    {
      compStr.append (unicodeValueOf (decomp));
    }
    l->setText(compStr);
    return;
  } 

  /**
   * xxxx [decomp]
   */
  if (decomp.size() > 0)
  {
    if (g->getShapeArray()) decompStr.append ("noshape:");
    SV_UCS4 v; v.append (comp);
    compStr.append (unicodeValueOf (v));
    decompStr.append ("[");
    decompStr.append (unicodeValueOf (decomp));
    decompStr.append ("]");
    compStr.append (" ");
    compStr.append (decompStr);
    l->setText(compStr);
    return;
  }

  /**
   * xxxx 
   */
  SV_UCS4 v; v.append (comp);
  compStr.append (unicodeValueOf (v));
  l->setText(compStr);
  return;
}
