/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include "swindow/SFont.h"

#include "swindow/SFontFB.h"
#include "stoolkit/SMatrix.h"
#include "stoolkit/SHashtable.h"
#include "stoolkit/STextData.h"
#include "stoolkit/SExcept.h"

static SFontFB fallbackFont;
typedef SHashtable<SFontImplVector> SFontHashtable;
static SFontHashtable* fontHashtable=0;

static SFontFB::SIGN mapFB (const SGlyph& glyph);

/* write the composing character bellow this */

/**
 * Create a font. The font can have many faces. 
 * @param _name is the logical name of this font.
 */
SFont::SFont (const SString _name) : name(_name), xlfd (SD_XLFD_ANY)
{
 if (fontHashtable == 0 ) fontHashtable = new SFontHashtable();
 if (fontHashtable->get (name) != 0) fontVector = (*fontHashtable)[name];
 setSize (16.0);
}
/**
 * Create a font. The font can have many faces. 
 * @param _name is the logical name of this font.
 */
SFont::SFont (const SString _name, double _size) : name(_name), xlfd (SD_XLFD_ANY)
{
 if (fontHashtable == 0 ) fontHashtable = new SFontHashtable();
 if (fontHashtable->get (name) != 0) fontVector = (*fontHashtable)[name];
 setSize (_size);
}

/**
 * get a point 16 font called _default
 */
SFont::SFont (void) : name("default"), xlfd (SD_XLFD_ANY)
{
  static bool _defaultSet=false;
  if (_defaultSet==false)
  {
    /* FIXME: code2000.ttf and arial.ttf are not free fonts */
    SStringVector m(
        "-*-*-medium-r-normal--16-*-*-*-c-*-iso8859-1,"
        "-*-*-*-*-*--16-*-*-*-c-*-iso8859-1,"
        "cyberbit.ttf,code2000.ttf,arial.ttf,"
        "yudit.ttf"
    );
    SFontImplVector list;
    for (unsigned int i=0; i<m.size(); i++)
    {
      /* encoding is optional */
      SStringVector v(m[i], ":", false);
      SString enc = v[0];

      if (v.size()>1 && v[1].size()!=0) enc = v[1];

      SFontImpl impl (v[0], enc);
      if (v.size()>2 && v[2].size()>0)
      {
         SStringVector pvect(v[2], ";");
         SProperties props;
         for (unsigned int j=0; j<pvect.size(); j++)
         {
            SStringVector vv(pvect[j], "=", true);
            if (vv.size() > 1)
            {
               props.put (vv[0], vv[1]);
            }
            else
            {
               props.put (vv[0], "true");
            }
         }
         impl.setAttributes(props);
      }
      list.append (impl);
    }
    put ("default", list);
  }
  if (fontHashtable == 0 ) fontHashtable = new SFontHashtable();
  fontVector = (*fontHashtable)["default"];
  setSize (16.0);
  
}

/**
 * Copy this font.
 */
SFont::SFont (const SFont& font)
{
  name = font.name;
  xlfd = font.xlfd;
  fontAscent = font.fontAscent;
  fontDescent = font.fontDescent;
  fontWidth = font.fontWidth;
  fontGap = font.fontGap;
  fontVector = font.fontVector;
  fontScale = font.fontScale;
}

/**
 * assign this font.
 */
SFont
SFont::operator = (const SFont& font)
{
  if (&font == this) return *this;
  name = font.name;
  fontAscent = font.fontAscent;
  fontDescent = font.fontDescent;
  fontWidth = font.fontWidth;
  fontGap = font.fontGap;
  xlfd = font.xlfd;
  fontScale = font.fontScale;
  fontVector = font.fontVector;
  return *this;
}

SFont::~SFont()
{
}


/**
 * Go through the list and set the size.
 */
void
SFont::setSize(double points)
{
  fontScale = points;
  SS_Matrix2D m;
  double sc = fallbackFont.scale();
  m.scale (fontScale * sc, fontScale * sc);
  fontGap =  fallbackFont.gap(m);
  fontAscent =  fallbackFont.ascent(m);
  fontDescent =  fallbackFont.descent(m);

  for (unsigned int i=0; i<fontVector.size(); i++)
  {
    SFontImpl im = fontVector[i];
    im.scale (fontScale, fontScale);
    //fontVector.remove (i);
    fontVector.replace (i, im);
    double g = im.gap ();
    double a = im.ascent ();
    double d = im.descent ();
    if (g > fontGap) fontGap = g;
    if (a > fontAscent) fontAscent = a;
    if (d > fontDescent) fontDescent = d;
  }
}

/**
 * Get the size of the font.
 * @return the size in points
 */
double
SFont::getSize () const
{
  return fontScale;
}

/**
 * A static method to build a font list.
 * This is only use at initialization time.
 * @param name is the name of the font.
 * @param gface is the face to add to the name.
 */
void
SFont::put (const SString name, const SFontImplVector& face)
{
  if (fontHashtable == 0 ) fontHashtable = new SFontHashtable();
  fontHashtable->put (name, face);
}

/**
 * clear all the stuff in the list
 */
void
SFont::clear()
{
}

/**
 * map a fallback font sign.
 */
static SFontFB::SIGN
mapFB (const SGlyph& glyph)
{
  if (glyph.size() > 1)
  {
    if (glyph[0] == SGlyph::CR || glyph[1] == SGlyph::LF) return SFontFB::CRLF;
    if (glyph[1] == SGlyph::CR || glyph[0] == SGlyph::LF) return SFontFB::LFCR;
  }
  else
  {
    if (glyph.getChar() == SGlyph::CR) return SFontFB::CR;
    if (glyph.getChar() == SGlyph::LF) return SFontFB::LF;
    if (glyph.getChar() == SGlyph::LS) return SFontFB::LS;
    if (glyph.getChar() == SGlyph::PS) return SFontFB::PS;
    if (glyph.getChar() == SGlyph::TAB) return SFontFB::TAB;
    if (glyph.getChar() < 32) return SFontFB::CTRL;
  }
  return SFontFB::LF;
}

/**
 * Try to draw one single glyph.
 * @param canvas is the canvas to draw to 
 * @param m is the conversion matrix
 * @param uch is the array containing ucs4 
 * @prama len is the length of the array
 */
void
SFont::draw (SCanvas* canvas, const SPen& pen, const SS_Matrix2D& m, 
  const SGlyph& glyph)
{
  double currw = 0.0;
  if (glyph.isSpecial())
  {
    SS_Matrix2D sd;
    double sc = fallbackFont.scale();
    sd.scale (fontScale * sc, fontScale * sc);
    SS_Matrix2D sm =  m * sd;
    SFontFB::SIGN sig = mapFB (glyph);
    fallbackFont.signDraw(canvas, pen, sm, sig, glyph.getChar());
    return;
  }
  /* try to draw it with all fonts. */
  SS_UCS4 comp =  glyph.getShapedChar();
  if (!glyph.lr && glyph.isMirrorable())
     comp = glyph.getMirroredChar();

  /* first try the precomposed */
  if (comp != 0 && comp!= 0x200c && comp != 0x200d)
  {
    unsigned int i;
    /* use mirrored glyphs for Old Hungarian */
    if (comp >= 0xEE00 && comp < 0xEE9F)
    {
      /* Try to use lr and rl attributes */
      for (i=0; i<fontVector.size(); i++)
      {
        SFontImpl im = fontVector[i];
        if (im.isLR() && !glyph.lr) continue;
        if (im.isRL() && glyph.lr) continue;
        /* neutral = lr */
        if (!im.isLR() && !im.isRL() && !glyph.lr) continue;
        if (im.draw (canvas, pen, m, comp)) 
        {
          return;
        }
      }
      SS_Matrix2D mm =  m;
      /* Try to mirror it - we drew neutrals and same directions */
      for (i=0; i<fontVector.size(); i++)
      {
        SFontImpl im = fontVector[i];
        bool used = im.width (comp, &currw);
        if (used)
        {
          /* Mirrorring can be done only if we render on our own. */
          if (im.isTTF())
          {
            mm.x0 = -m.x0; 
            mm.t0 = m.t0 + currw;
          }
          im.draw (canvas, pen, mm, comp); 
          return;
        }
        currw = 0.0;
      }
    }
    else
    {
       /* Try all fonts on it */
      for (i=0; i<fontVector.size(); i++)
      {
        SFontImpl im = fontVector[i];
        if (im.draw (canvas, pen, m, comp)) 
        {
          return;
        }
      }
    }
  }

  /* You reach this point if comp did not work*/
  unsigned int gsize = glyph.size();
  const SS_UCS4* decomp = glyph.getDecompArray(); 
  if (gsize > 0)
   /* there is hope to draw composed chars */
  {
    double* positions = new double[gsize];
    unsigned int* indeces = new unsigned int[gsize];
    CHECK_NEW (positions);
    CHECK_NEW (indeces);
    bool found = false;
    bool overstrike =  (glyph.getShapeArray() ==0 
        && !glyph.isYuditLigature() && !glyph.isCluster());

    /* build positions */
    unsigned int index = 0;
    unsigned int i=0;
    unsigned int fsize = fontVector.size();
    double fullsize = 0;
    while (i<fsize) 
    {
      SFontImpl im = fontVector[i];
      bool used = im.width (decomp[index], &currw);
      /* ZWJ and ZWNJ - use fallback if not present.*/
      if (!used && i+1 == fsize && !overstrike)
      {
         SS_Matrix2D sm;
         double sc = fallbackFont.scale();
         sm.scale (fontScale * sc, fontScale * sc);
         used = true;
         i = fsize;
         currw = fallbackFont.width (sm, decomp[index]);
      }
      /* True Type fonts will need to position 
         non spacing marks *after* moving cursor */
      if (index ==0)
      {
         if (!used || currw==0)
         {
           i++; continue;
         }
         positions[index] = currw; 
         fullsize = currw;
      }
      else if (overstrike) /* may have zero width */
      {
         if (!used)
         {
           i++; continue;
         }
         positions[index] = im.isTTF() ? positions[0] : 0;
      }
      else
      {
         /* clusters can have 0 width stuff. */
         if (!used)
         {
           i++; continue;
         }
         fullsize += currw;
         positions[index] = fullsize;
      }
      indeces[index] = i;
      i=0;
      index++;
      /* found if all found */
      if (index == gsize)
      {
        found = 1;
        break;
      }
    }
    if (found)
    {
      for (i=0; i<gsize; i++)
      {
        SS_Matrix2D mo = m;
        double translatex = 0.0;
        if (glyph.lr)
        {
          if (overstrike)
          {
             translatex = (i==0) ? 0.0 : positions[i];
          }
          else
          {
             translatex = (i==0) ? 0.0 : positions[i-1];
          }
        }
        else
        {
          if (overstrike)
          {
             translatex = (i==0) ? 0.0 : positions[i];
          }
          else
          {
             translatex = fullsize - positions[i];
          }
        }
        if (indeces[i] == fsize)
        {
          SS_Matrix2D sd;
          double sc = fallbackFont.scale();
          sd.scale (fontScale * sc, fontScale * sc);
          SS_Matrix2D sm =  m * sd;
          sm.translate (translatex, (double)0.0);
          fallbackFont.draw(canvas, pen, sm, decomp[i]);
        }
        else
        {
          mo.translate (translatex, (double)0.0);
          SFontImpl im = fontVector[indeces[i]];
          im.draw (canvas, pen, mo, decomp[i]);
        }
      }
    }
    delete positions;
    delete indeces;
    if (found) return;
  }
  else if (glyph.getChar() != comp)
   /* gsize is 0. Is it a single shaped glyph? */
  {
    SS_UCS4 orchar = glyph.getChar();
    /* Try all fonts on it */
    for (unsigned int i=0; i<fontVector.size(); i++)
    {
      SFontImpl im = fontVector[i];
      /* cool. draw it */
      if (im.draw (canvas, pen, m, orchar)) 
      {
         return;
      }
    }
  }

  /* Draw some last resort font. */
  SS_Matrix2D sd;
  double sc = fallbackFont.scale();
  sd.scale (fontScale * sc, fontScale * sc);
  SS_Matrix2D sm =  m * sd;

  if (comp != 0)
  {
    fallbackFont.draw(canvas, pen, sm, comp);
  }
  else
  {
    SS_Matrix2D mo = sm;
    int myindex = 0;
    int inc = 1;
    int limit = gsize;
    if (!glyph.lr)
    {
      myindex = gsize-1;
      inc = -1;
      limit = -1;
    }
    while (myindex!=limit)
    {
      fallbackFont.draw(canvas, pen, mo, decomp[myindex]);
      currw = fallbackFont.width (mo, decomp[myindex]);
      mo.translate (currw, (double)0.0);
      myindex = myindex + inc;
    }
  }
  return;
}


/**
 * return the width of the characters
 * @param m is the conversion matrix
 * @param uch is the array containing ucs4 
 * @prama len is the length of the array
 */
 
double
SFont::width (const SGlyph& glyph)
{
  double maxw = 0.0;
  double currw = 0.0;
  if (glyph.isSpecial ())
  {
    SS_Matrix2D m;
    double sc = fallbackFont.scale();
    m.scale (fontScale * sc, fontScale * sc);
    SFontFB::SIGN sig = mapFB (glyph);
    currw =  fallbackFont.signWidth (m, sig);
    return currw;
  }

  /* try to draw it with all fonts. */
  SS_UCS4 comp =  glyph.getShapedChar();
  if (!glyph.lr && glyph.isMirrorable())
     comp = glyph.getMirroredChar();

  /* first try the precomposed */
  if (comp != 0 && comp!= 0x200c && comp != 0x200d)
  {
    for (unsigned int i=0; i<fontVector.size(); i++)
    {
      SFontImpl im = fontVector[i];
      if (im.width (comp, &currw))
      {
        if (currw < 0.0) return 1.0;
        return currw;
      }
    }
  }
  /* You reach this point if comp did not work */
  unsigned int gsize = glyph.size();
  const SS_UCS4* decomp = glyph.getDecompArray(); 

  /*
   * Shaped glyphs have no overstrikes here, but continuous output. 
   */
  if (gsize > 0)
  {
    bool overstrike =  (glyph.getShapeArray() ==0 
        && !glyph.isYuditLigature() && !glyph.isCluster());
    unsigned int index = 0;
    unsigned int i=0;
    unsigned int fsize = fontVector.size();
    while (i<fsize)
    {
      SFontImpl im = fontVector[i];
      bool used = im.width (decomp[index], &currw);
      /* can draw fallback in the middle */
      if (!used && i+1 == fsize && !overstrike)
      {
         SS_Matrix2D sm;
         double sc = fallbackFont.scale();
         sm.scale (fontScale * sc, fontScale * sc);
         used = true;
         i = fsize;
         currw = fallbackFont.width (sm, decomp[index]);
      }
      if (index==0 && (currw==0.0 || !used))
      {
          i++; continue;
      }
       /* clusters also can have 0 width stuff */
      if (!used)
      {
        i++; continue;
      }
      if (overstrike)
      {
        if (index==0) maxw = currw;
      }
      else
      {
        maxw += currw;
      }
      index++;
      i = 0;
      if (index >= gsize)
      {
        if (maxw > 0.0) return maxw;
        break;
      }
    }
  }
  else if (glyph.getChar() != comp)
   /* gsize is 0. Is it a single shaped glyph? */
  {
    SS_UCS4 ocomp = glyph.getChar();
    for (unsigned int i=0; i<fontVector.size(); i++)
    {
      SFontImpl im = fontVector[i];
      bool used = im.width (ocomp, &currw);
      if (used && currw >0.0)
      {
        return currw;
      }
    }
  }

  /* last resort font */
  SS_Matrix2D sm;
  double sc = fallbackFont.scale();
  sm.scale (fontScale * sc, fontScale * sc);
  if (comp != 0)
  {
    maxw = fallbackFont.width (sm, comp);
  }
  else
  {
    maxw  = 0;
    for (unsigned int i=0; i< glyph.size(); i++)
    {
      maxw = maxw + fallbackFont.width (sm, decomp[i]);
    }
  }
  return maxw;
}

/**
 * return the overall width
 */
double
SFont::width () const
{
  return fontWidth;
}

/**
 * return the overall ascent
 */
double
SFont::ascent () const
{
  return fontAscent;
}


/**
 * return the overall descent
 */
double
SFont::descent () const
{
  return fontDescent;
}

/**
 * return the overall gap
 */
double
SFont::gap () const
{
  return fontGap;
}
