% $Id: PrecCheck.lhs,v 1.7 2004/08/08 12:05:32 berrueta Exp $
%
% Copyright (c) 2001-2003, Wolfgang Lux
% Copyright (c) 2003, Diego Berrueta
% See LICENSE for the full license.
%
\nwfilename{PrecCheck.lhs}
\codesection{Checking Precedences of Infix Operators}
The parser does not know the relative precedences of infix operators
and therefore parses them as if they are all left associative and
having the same precedence. After performing the definition checks,
the compiler is going to process the infix applications in the module
and rearrange infix applications according to their relative
precedences.
\begin{lstlisting}

> module PrecCheck(precDeclEnv,precCheck,precCheckGoal) where
> import Base
> import Ident
> import PEnv
> import CurrySyntax
> import Position
> import List
> import Env
> import TopEnv
> import TypeClassBinding(methodsOfTypeClassDecl)
> import Error(errorAt,internalError)
> import Expr

\end{lstlisting}
In order to assign fixities to each operator defined in the current
module, the compiler first constructs an environment which collects
the information from the infix declarations in the module. This
environment is then used to initialize the environment that maps the
defined operators onto their associated fixity and precedence.

The reason for this two step process is that Curry allows infix
declarations only at the top-level.\footnote{In fact, infix
declarations must follow the import declarations and precede all
other global declarations. However, our parser is a little bit more
liberal in this respect.} In order to declare fixities for functions
in local declaration groups, we allow infix declarations for operators
that are not defined at the top-level. Therefore, we cannot use the
infix declarations directly to associate fixities with the operators;
otherwise, we might shadow the fixity of an imported operator name.

\ToDo{Check that at least one operator is defined for each fixity
declared here.}

\ToDo{Allow infix declarations in local declaration groups. This is
compatible with Haskell, but looses the possibility to associate a
precedence with an argument of a function or $\lambda$-expression,
respectively.}
\begin{lstlisting}

> type PDeclEnv = Env Ident OpPrec

> precDeclEnv :: ModuleIdent -> [Decl] -> PDeclEnv
> precDeclEnv m ds =
>   case linear [PIdent p op | InfixDecl p _ _ ops <- ds, op <- ops] of
>     Linear -> foldr bindPrec emptyEnv ds
>     NonLinear (PIdent p op) -> errorAt p (duplicatePrecedence op)
>   where bindPrec (InfixDecl _ fix p ops) env =
>           foldr (flip bindEnv (OpPrec fix p)) env ops
>         bindPrec _ env = env

\end{lstlisting}
For each scope, we use the environment constructed from the infix
declarations to assign fixities to the operators defined in that
scope. In order to keep this environment small, only those operators
for which an explicit fixity declaration exists are entered into the
environment. For all other operators, the default fixity is assumed.
\begin{lstlisting}

> bindPrecs :: ModuleIdent -> PDeclEnv -> [Decl] -> PEnv -> PEnv
> bindPrecs m pDeclEnv ds env =
>   foldr (definePrec m pDeclEnv) env (opsInScope ds)
>   where opsInScope :: [Decl] -> [Ident]
>         opsInScope ds = [op | d <- ds, isValueDecl d, op <- bv d] ++
>                         (concat [methodsOfTypeClassDecl d | d <- ds,
>			                                      isTypeClassDecl d])

> definePrec :: ModuleIdent -> PDeclEnv -> Ident -> PEnv -> PEnv
> definePrec m pDeclEnv op =
>   maybe id (bindP m op) (lookupEnv (unRenameIdent op) pDeclEnv)

\end{lstlisting}
With the help of these environments, the compiler checks all infix
applications and sections in the program. This pass will modify the
parse tree such that for a nested infix application the operator with
the lowest precedence will be at the top and that two adjacent
operators with the same precedence will not have conflicting
associativities. Note that the top-level precedence environment has to
be returned because it is needed to construct the module's interface.

\emph{The code below is based on a few assumptions about the syntax
tree returned by the parser. In particular, the expression child of an
unary negation is never an infix application, i.e., for the parser
unary negation binds tighter than all infix operators. In addition, we
assume here that for the parser all infix operators have the same
precedence and associate to the right. Thus, the left argument in an
infix application will never be an infix application.}
\begin{lstlisting}

> precCheck :: ModuleIdent -> PDeclEnv -> PEnv -> [Decl] -> (PEnv,[Decl])
> precCheck = precCheckDecls

> precCheckGoal :: PEnv -> Goal -> Goal
> precCheckGoal pEnv (Goal p e ds) =
>   Goal p (precCheckExpr m emptyEnv p pEnv' e) ds'
>   where pEnv' = bindPrecs m emptyEnv ds pEnv
>         ds' = map (precCheckDecl m emptyEnv pEnv') ds
>         m = emptyMIdent 

> precCheckDecls :: ModuleIdent -> PDeclEnv -> PEnv -> [Decl] -> (PEnv,[Decl])
> precCheckDecls m pDeclEnv pEnv ds = (pEnv',ds')
>   where pEnv' = bindPrecs m pDeclEnv ds pEnv
>         ds' = map (precCheckDecl m pDeclEnv pEnv') ds

> precCheckDecl :: ModuleIdent -> PDeclEnv -> PEnv -> Decl -> Decl
> precCheckDecl m pDeclEnv pEnv (InstanceDecl p ctx tc tyexpr decls) =
>   InstanceDecl p ctx tc tyexpr (snd $ precCheckDecls m pDeclEnv pEnv decls)
> precCheckDecl m pDeclEnv pEnv (FunctionDecl p f eqs) =
>   FunctionDecl p f (map (precCheckEqn m pDeclEnv pEnv) eqs)
> precCheckDecl m pDeclEnv pEnv (PatternDecl p t rhs) =
>   PatternDecl p t (precCheckRhs m pDeclEnv pEnv rhs)
> precCheckDecl _ _ _ d = d

> precCheckEqn :: ModuleIdent -> PDeclEnv -> PEnv -> Equation -> Equation
> precCheckEqn m pDeclEnv pEnv (Equation p ts rhs) =
>   Equation p ts (precCheckRhs m pDeclEnv pEnv rhs)

> precCheckRhs :: ModuleIdent -> PDeclEnv -> PEnv -> Rhs -> Rhs
> precCheckRhs m pDeclEnv pEnv (SimpleRhs p e ds) =
>   SimpleRhs p (precCheckExpr m pDeclEnv p pEnv' e) ds'
>   where (pEnv',ds') = precCheckDecls m pDeclEnv pEnv ds
> precCheckRhs m pDeclEnv pEnv (GuardedRhs es ds) =
>   GuardedRhs (map (precCheckCondExpr m pDeclEnv pEnv') es) ds'
>   where (pEnv',ds') = precCheckDecls m pDeclEnv pEnv ds

> precCheckCondExpr :: ModuleIdent -> PDeclEnv -> PEnv -> CondExpr -> CondExpr
> precCheckCondExpr m pDeclEnv pEnv (CondExpr p g e) =
>   CondExpr p (precCheckExpr m pDeclEnv p pEnv g)
>              (precCheckExpr m pDeclEnv p pEnv e)

> precCheckExpr :: ModuleIdent -> PDeclEnv -> Position -> PEnv -> Expression
>               -> Expression
> precCheckExpr _ _ _ _ (Literal l) = Literal l
> precCheckExpr _ _ _ _ (Variable v) = Variable v
> precCheckExpr _ _ _ _ (Constructor c) = Constructor c
> precCheckExpr m pDeclEnv p pEnv (Paren e) =
>   Paren (precCheckExpr m pDeclEnv p pEnv e)
> precCheckExpr m pDeclEnv p pEnv (Typed e ty) =
>   Typed (precCheckExpr m pDeclEnv p pEnv e) ty
> precCheckExpr m pDeclEnv p pEnv (Tuple es) =
>   Tuple (map (precCheckExpr m pDeclEnv p pEnv) es)
> precCheckExpr m pDeclEnv p pEnv (List es) =
>   List (map (precCheckExpr m pDeclEnv p pEnv) es)
> precCheckExpr m pDeclEnv p pEnv (ListCompr e qs) =
>   ListCompr (precCheckExpr m pDeclEnv p pEnv' e) qs'
>   where (pEnv',qs') = mapAccumL (precCheckStmt m pDeclEnv p) pEnv qs
> precCheckExpr m pDeclEnv p pEnv (EnumFrom e) =
>   EnumFrom (precCheckExpr m pDeclEnv p pEnv e)
> precCheckExpr m pDeclEnv p pEnv (EnumFromThen e1 e2) =
>   EnumFromThen (precCheckExpr m pDeclEnv p pEnv e1)
>                (precCheckExpr m pDeclEnv p pEnv e2)
> precCheckExpr m pDeclEnv p pEnv (EnumFromTo e1 e2) =
>   EnumFromTo (precCheckExpr m pDeclEnv p pEnv e1)
>              (precCheckExpr m pDeclEnv p pEnv e2)
> precCheckExpr m pDeclEnv p pEnv (EnumFromThenTo e1 e2 e3) =
>   EnumFromThenTo (precCheckExpr m pDeclEnv p pEnv e1)
>                  (precCheckExpr m pDeclEnv p pEnv e2)
>                  (precCheckExpr m pDeclEnv p pEnv e3)
> precCheckExpr m pDeclEnv p pEnv (UnaryMinus op e) =
>   UnaryMinus op (precCheckExpr m pDeclEnv p pEnv e)
> precCheckExpr m pDeclEnv p pEnv (Apply e1 e2) =
>   Apply (precCheckExpr m pDeclEnv p pEnv e1)
>         (precCheckExpr m pDeclEnv p pEnv e2)
> precCheckExpr m pDeclEnv p pEnv (InfixApply e1 op e2) =
>   fixPrec p pEnv (precCheckExpr m pDeclEnv p pEnv e1) op
>           (precCheckExpr m pDeclEnv p pEnv e2)
> precCheckExpr m pDeclEnv p pEnv (LeftSection e op) =
>   checkLSection p pEnv op (precCheckExpr m pDeclEnv p pEnv e)
> precCheckExpr m pDeclEnv p pEnv (RightSection op e) =
>   checkRSection p pEnv op (precCheckExpr m pDeclEnv p pEnv e)
> precCheckExpr m pDeclEnv p pEnv (Lambda ts e) =
>   Lambda ts (precCheckExpr m pDeclEnv p pEnv' e)
>   where pEnv' = foldr (definePrec m pDeclEnv) pEnv (bv ts)
> precCheckExpr m pDeclEnv p pEnv (Let ds e) =
>   Let ds' (precCheckExpr m pDeclEnv p pEnv' e)
>   where (pEnv',ds') = precCheckDecls m pDeclEnv pEnv ds
> precCheckExpr m pDeclEnv p pEnv (Do sts) =
>   Do (snd (mapAccumL (precCheckStmt m pDeclEnv p) pEnv sts))
> precCheckExpr m pDeclEnv p pEnv (IfThenElse e1 e2 e3) =
>   IfThenElse (precCheckExpr m pDeclEnv p pEnv e1)
>              (precCheckExpr m pDeclEnv p pEnv e2)
>              (precCheckExpr m pDeclEnv p pEnv e3)
> precCheckExpr m pDeclEnv p pEnv (Case e alts) =
>   Case (precCheckExpr m pDeclEnv p pEnv e)
>        (map (precCheckAlt m pDeclEnv pEnv) alts)

> precCheckStmt :: ModuleIdent -> PDeclEnv -> Position -> PEnv -> Statement
>               -> (PEnv,Statement)
> precCheckStmt m pDeclEnv p pEnv (StmtExpr e) =
>   (pEnv,StmtExpr (precCheckExpr m pDeclEnv p pEnv e))
> precCheckStmt m pDeclEnv _ pEnv (StmtDecl ds) = (pEnv',StmtDecl ds')
>   where (pEnv',ds') = precCheckDecls m pDeclEnv pEnv ds
> precCheckStmt m pDeclEnv p pEnv (StmtBind t e) =
>   (pEnv',StmtBind t (precCheckExpr m pDeclEnv p pEnv' e))
>   where pEnv' = foldr (definePrec m pDeclEnv) pEnv (bv t)

> precCheckAlt :: ModuleIdent -> PDeclEnv -> PEnv -> Alt -> Alt
> precCheckAlt m pDeclEnv pEnv (Alt p t rhs) =
>   Alt p t (precCheckRhs m pDeclEnv pEnv rhs)

\end{lstlisting}
The functions \texttt{fixPrec}, \texttt{fixUPrec}, and
\texttt{fixRPrec} check the relative precedences of adjacent infix
operators in nested infix applications and unary negations. The
expressions will be reordered such that the infix operator with the
lowest precedence is at the top of the expression. The functions use
the fact that the parser constructs infix applications in a
right-associative fashion, i.e., the left argument of an infix
application will never be an infix application. In addition, a unary
negation will never have an infix application as its argument.

The function \texttt{fixPrec} checks whether the left argument of an
infix application is a unary negation and eventually reorders the
expression if the precedence of the infix operator is higher than the
of the negation. This will be done with the help of the function
\texttt{fixUPrec}. In any case the function \texttt{fixRPrec} is used
to fix the precedence of the infix operator and that of its right
argument. Note that both arguments already have been checked before
\texttt{fixPrec} is called.
\begin{lstlisting}

> fixPrec :: Position -> PEnv -> Expression -> InfixOp -> Expression
>         -> Expression
> fixPrec p pEnv (UnaryMinus uop e1) op e2
>   | pr < 6 || pr == 6 && fix == InfixL =
>       fixRPrec p pEnv (UnaryMinus uop e1) op e2
>   | pr > 6 = fixUPrec p pEnv uop e1 op e2
>   | otherwise = errorAt p $ ambiguousParse "unary " (qualify uop) (opName op)
>   where OpPrec fix pr = prec op pEnv
> fixPrec p pEnv e1 op e2 = fixRPrec p pEnv e1 op e2

> fixUPrec :: Position -> PEnv -> Ident -> Expression -> InfixOp -> Expression
>          -> Expression
> fixUPrec p pEnv uop  _ op (UnaryMinus _ _) =
>   errorAt p $ ambiguousParse "operator" (opName op) (qualify uop)
> fixUPrec p pEnv uop e1 op1 (InfixApply e2 op2 e3)
>   | pr2 < 6 || pr2 == 6 && fix2 == InfixL =
>       InfixApply (fixUPrec p pEnv uop e1 op1 e2) op2 e3
>   | pr2 > 6 = UnaryMinus uop (fixRPrec p pEnv e1 op1 (InfixApply e2 op2 e3))
>   | otherwise = errorAt p $ ambiguousParse "unary" (qualify uop) (opName op2)
>   where OpPrec fix1 pr1 = prec op1 pEnv
>         OpPrec fix2 pr2 = prec op2 pEnv
> fixUPrec p pEnv uop e1 op e2 = UnaryMinus uop (InfixApply e1 op e2)

> fixRPrec :: Position -> PEnv -> Expression -> InfixOp -> Expression
>          -> Expression
> fixRPrec p pEnv e1 op (UnaryMinus uop e2)
>   | pr < 6 = InfixApply e1 op (UnaryMinus uop e2)
>   | otherwise =
>       errorAt p $ ambiguousParse "operator" (opName op) (qualify uop)
>   where OpPrec _ pr = prec op pEnv
> fixRPrec p pEnv e1 op1 (InfixApply e2 op2 e3)
>   | pr1 < pr2 || pr1 == pr2 && fix1 == InfixR && fix2 == InfixR =
>       InfixApply e1 op1 (InfixApply e2 op2 e3)
>   | pr1 > pr2 || pr1 == pr2 && fix1 == InfixL && fix2 == InfixL =
>       InfixApply (fixPrec p pEnv e1 op1 e2) op2 e3
>   | otherwise =
>       errorAt p $ ambiguousParse "operator" (opName op1) (opName op2)
>   where OpPrec fix1 pr1 = prec op1 pEnv
>         OpPrec fix2 pr2 = prec op2 pEnv
> fixRPrec p pEnv e1 op e2 = InfixApply e1 op e2

\end{lstlisting}
The functions \texttt{checkLSection} and \texttt{checkRSection} are used
to handle the precedences inside left and right sections. These
functions only need to check that an infix operator occurring in the
section has either a higher precedence than the section operator or
both operators have the same precedence and are both left associative
for a left section and right associative for a right section.
\begin{lstlisting}

> checkLSection :: Position -> PEnv -> InfixOp -> Expression -> Expression
> checkLSection p pEnv op e@(UnaryMinus uop _)
>   | pr < 6 || pr == 6 && fix == InfixL = LeftSection e op
>   | otherwise = errorAt p $ ambiguousParse "unary" (qualify uop) (opName op)
>   where OpPrec fix pr = prec op pEnv
> checkLSection p pEnv op1 e@(InfixApply _ op2 _)
>   | pr1 < pr2 || pr1 == pr2 && fix1 == InfixL && fix2 == InfixL =
>       LeftSection e op1
>   | otherwise =
>       errorAt p $ ambiguousParse "operator" (opName op1) (opName op2)
>   where OpPrec fix1 pr1 = prec op1 pEnv
>         OpPrec fix2 pr2 = prec op2 pEnv
> checkLSection p pEnv op e = LeftSection e op

> checkRSection :: Position -> PEnv -> InfixOp -> Expression -> Expression
> checkRSection p pEnv op e@(UnaryMinus uop _)
>   | pr < 6 = RightSection op e
>   | otherwise = errorAt p $ ambiguousParse "unary" (qualify uop) (opName op)
>   where OpPrec _ pr = prec op pEnv
> checkRSection p pEnv op1 e@(InfixApply _ op2 _)
>   | pr1 < pr2 || pr1 == pr2 && fix1 == InfixR && fix2 == InfixR =
>       RightSection op1 e
>   | otherwise =
>       errorAt p $ ambiguousParse "operator" (opName op1) (opName op2)
>   where OpPrec fix1 pr1 = prec op1 pEnv
>         OpPrec fix2 pr2 = prec op2 pEnv
> checkRSection p pEnv op e = RightSection op e

\end{lstlisting}
The function \texttt{prec} returns the fixity and operator precedence
for an entity. As the precedence checking is performed after the
renaming phase and applied only to expressions, the lookup cannot see
any ambiguous identifiers.
\begin{lstlisting}

> prec :: InfixOp -> PEnv -> OpPrec
> prec op env =
>   case qualLookupP (opName op) env of
>     [] -> defaultP
>     [PrecInfo _ p] -> p
>     _ -> internalError "prec"

\end{lstlisting}
Error messages.
\begin{lstlisting}

> duplicatePrecedence :: Ident -> String
> duplicatePrecedence op = "More than one fixity declaration for " ++ name op

> ambiguousParse :: String -> QualIdent -> QualIdent -> String
> ambiguousParse what op1 op2 =
>   "Ambiguous use of " ++ what ++ " " ++ qualName op1 ++
>   " with " ++ qualName op2

\end{lstlisting}
