/*
    This software may only be used by you under license from AT&T Corp.
    ("AT&T").  A copy of AT&T's Source Code Agreement is available at
    AT&T's Internet website having the URL:
    <http://www.research.att.com/sw/tools/graphviz/license/source.html>
    If you received this software without first entering into a license
    with AT&T, you have an infringing copy of this software and cannot use
    it without violating AT&T's intellectual property rights.
*/

/*
 * Written by Stephen North
 * Updated by Emden Gansner
 */

typedef char Agraphinfo_t;
typedef struct Agnodeinfo_t { char mark; } Agnodeinfo_t;
typedef char Agedgeinfo_t;

#include <graph.h>
#include <unistd.h>
#include <string.h>
#include <ingraphs.h>

#ifdef NO_OPTOPT_DECL
#include "getopt.h"
#endif

#define INTERNAL 0
#define EXTERNAL 1
#define SILENT   2

char** Files;
int    verbose;
int    printMode = INTERNAL;
char*  suffix = 0;
char*  outfile = 0;
char*  path = 0;
int    sufcnt = 0;

static char* useString = 
"Usage: ccomps [-svx?] [-o <out template>] <files>\n\
  -s - silent\n\
  -x - external\n\
  -v - verbose\n\
  -o - output file template\n\
  -? - print usage\n\
If no files are specified, stdin is used\n";

static void
usage (int v)
{
    printf (useString);
    exit (v);
}

static void
split (char* name)
{
  char* sfx = 0;
  int   size;

  sfx = strrchr (name, '.');
  if (sfx) {
    suffix = sfx+1;
    size = sfx-name;
    path = (char*)malloc (size+1);
    strncpy (path, name, size);
    *(path+size) = '\0';
  }
  else {
    path = name;
  }
}

static void
init (int argc, char* argv[])
{
  int c;

  aginit ();
  while ((c = getopt(argc, argv, ":o:xsv?")) != -1) {
    switch (c) {
    case 'o':
      outfile = optarg;
      split (outfile);
      break;
    case 'x':
      printMode = EXTERNAL;
      break;
    case 's':
      printMode = SILENT;
      verbose = 1;
      break;
    case 'v':
      verbose = 1;
      break;
    case '?':
      if (optopt == '?') usage(0);
      else fprintf(stderr,"ccomps: option -%c unrecognized - ignored\n", c);
      break;
    }
  }
  argv += optind;
  argc -= optind;
  
  if (argc) Files = argv;
}

static int
dfs(Agraph_t* g, Agnode_t* n, Agraph_t* out, int cnt)
{
  Agedge_t    *e;
  Agnode_t    *other;

  n->u.mark = 1;
  cnt++;
  aginsert(out,n);
  for (e = agfstedge(g,n); e ; e = agnxtedge(g,e,n)) {
    if ((other = e->tail) == n) other = e->head;
    if (other->u.mark == 0) cnt = dfs(g,other,out,cnt);
  }
  return cnt;
}

static int
nodeInduce (Agraph_t* g)
{
  Agnode_t    *n;
  Agedge_t    *e;
  int          e_cnt = 0;

  for (n = agfstnode(g); n; n = agnxtnode(g,n)) {
    for (e = agfstout(g->root,n); e; e = agnxtout(g->root,e)) {
      if (agcontains(g,e->head)) {
        aginsert(g,e);
        e_cnt++;
      }
    }
  }
  return e_cnt;
}

static char*
getName ()
{
  char* name;
  static char* buf = 0;

  if (sufcnt == 0) name = outfile;
  else {
    if (!buf) 
      buf = (char*)malloc(strlen(outfile)+20); /* enough to handle '_number' */
    if (suffix) sprintf (buf, "%s_%d.%s", path, sufcnt, suffix);
    else sprintf (buf, "%s_%d", path, sufcnt);
    name = buf;
  }
  sufcnt++;
  return name;
}

static void
gwrite (Agraph_t* g)
{
  FILE* outf;
  char* name;

  if (!outfile) agwrite (g, stdout);
  else {
    name = getName();
    outf = fopen (name, "w");
    if (!outf) {
      fprintf (stderr, "Could not open %s for writing\n", name);
      perror ("ccomps");
    }
    agwrite (g, outf);
    fclose (outf);
  }
}

static void
process (Agraph_t* g)
{
  long        n_cnt,c_cnt,e_cnt;
  char        name[64];
  Agraph_t    *out;
  Agnode_t    *n;

  c_cnt = 0;
  for (n = agfstnode(g); n; n = agnxtnode(g,n)) {
    if (n->u.mark) continue;
    sprintf(name,"%s_component_%d",g->name,c_cnt);
    out = agsubg(g,name);
    n_cnt = dfs(g,n,out,0);
    e_cnt = nodeInduce(out);
    if (printMode == EXTERNAL) gwrite(out);
    if (printMode != INTERNAL)
      agdelete (g, out);
    if (verbose)
      fprintf(stderr,"(%4d) %7ld nodes %7ld edges\n",
        c_cnt, n_cnt, e_cnt);
    c_cnt++;
  }

  if (printMode == INTERNAL) gwrite(g);

  if(verbose)
    fprintf(stderr,"       %7ld nodes %7ld edges %7ld components %s\n",
      agnnodes(g),agnedges(g),c_cnt,g->name);
}

int
main (int argc, char* argv[])
{
  Agraph_t*     g;
  ingraph_state ig;

  init (argc, argv);
  newIngraph (&ig, Files, agread);
  
  while ((g = nextGraph(&ig)) != 0) {
      process (g);
      agclose (g);
  }

  return 0;
}

