/*
 *   mwmparse.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <stdio.h>
#include <mwave.h>
#include <mwaveapi.h>

#include <mwmparse.h>
#include <mwmparsi.h>
#include <psv8itcb.h>
#include <stddef.h>
#include <sys/timeb.h>
#include <time.h>

#include <port_types.h>
#include <port_functions.h>

#include <mwmspcfc.h>

static char szThisFile[] = "mwmparse.c";

PSTATEINFO psi;
char ud_buffer[UD_BUFFER_SIZE];
int  ud_next;

ULONG mwmParseReadMemoryProfile( STATEINFO *psi );

HANDLE hWritePipe = (HANDLE)NULL;

 /* In Win32 version we put all critical functions in one code*/
 /* segment and lock it in the memory by locking next function in the same seg*/
VOID mwmParseLockedCodeSegmentForInterrupt(VOID);

/*****************************************************************************/
/* MTS 6262                                                                  */
/* 02/21/96 Add usTaskType to the parameter list.  This is so I can use the  */
/* same function to connect to both MCTL and V.34 IPC handlers.              */
/*                                                                           */
/*   0 = MCTL                                                                */
/*   1 = V.34  (actually anything non-zero at this time.                     */
/*****************************************************************************/
ULONG MWM_ENTRY mwmParseConnectIPC(HMTASK hTask , HIPC *phIPC , USHORT usTaskType)
{
  ULONG ulRC = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseConnectIPC entry\n");

  if (usTaskType == 0)
    ulRC = dspConnectIPC(hTask, (PFN)&mwmIPC1InterruptHandler, phIPC);
/*  else*/
/*    ulRC = dspConnectIPC(hTask, (PFN)&mwmIPC1V34InterruptHandler, phIPC);*/

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseConnectIPC exit\n");


  return ulRC;
}




ULONG MWM_ENTRY mwmParseCallback(ULONG mp1, ULONG mp2)

{
  static MODEMMEMORY TempModemMemory;
  FILE   *fMemoryFile;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseCallback entry\n");


  switch (mp1)
  {
    /*************************************************************************/
    /* These are all of the cases which write portions of the profile.       */
    /* Currently, I'm sure the nested switches look kind of stupid.          */
    /* However, I expect future callback routines for functions other than   */
    /* profile writing.                                                      */
    /* Since all profile writing is nearly identical, I am using the same    */
    /* code to open/read/reset/write/close the file.                         */
    /*************************************************************************/
    case MWM_WRITE_PROFILE0:
    case MWM_WRITE_PROFILE1:
    case MWM_WRITE_PROFILE_TO_LOAD:
    case MWM_WRITE_PROFILE_PHONE_NUM0:
    case MWM_WRITE_PROFILE_PHONE_NUM1:
    case MWM_WRITE_PROFILE_PHONE_NUM2:
    case MWM_WRITE_PROFILE_PHONE_NUM3:
      /***********************************************************************/
      /* -Open memory file for read/write                                    */
      /* -Read all info into TempModemMemory                                 */
      /* -Reset file pointer.                                                */
      /* -Switch on the type of profile to write...                          */
      /*   -memcpy new data into a portion of TempModemMemory.               */
      /* -Write TempModemMemory back to memory file.                         */
      /* -Close memory file.                                                 */
      /***********************************************************************/
      /* -Open memory file for read/write                                    */
      /***********************************************************************/
      fMemoryFile = fopen(psi->achMemoryFileName, "rb+");

      if (!fMemoryFile)
      {
        mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                          MWM_OS_ERROR,
                          0 );
        return (1);
      }

      /***********************************************************************/
      /* -Read all info into TempModemMemory                                 */
      /***********************************************************************/
      if (!fread(&TempModemMemory,sizeof(MODEMMEMORY),1,fMemoryFile) )
      {
        if ( ferror(fMemoryFile) )
        {
          mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                            MWM_OS_ERROR,
                            0 );
          return (1);
        }
      }

      /***********************************************************************/
      /* -Reset file pointer.                                                */
      /***********************************************************************/
      rewind(fMemoryFile);

      /***********************************************************************/
      /* -Switch on the type of profile to write...                          */
      /*   -memcpy new data into a portion of TempModemMemory.               */
      /***********************************************************************/
      switch(mp1)
      {
        case MWM_WRITE_PROFILE0:
          /*******************************************************************/
          /* Copy the profile which was passed to us to the Profile 0        */
          /* location of the TempModemMemory structure.                      */
          /*******************************************************************/
          memcpy(&TempModemMemory.aStoredRegs[0],
                 (PVOID)mp2,
                 sizeof(REGISTERS) );
          /*******************************************************************/
          /* Also, copy new settings to psi->ModemMemory.aStoredRegs[0]      */
          /* This updates the structure so that &v will return the updated   */
          /* profile.                                                        */
          /*******************************************************************/
          memcpy(&psi->ModemMemory.aStoredRegs[0],
                 (PVOID)mp2,
                 sizeof(REGISTERS) );
          break;

        case MWM_WRITE_PROFILE1:
          /*******************************************************************/
          /* Copy the profile which was passed to us to the Profile 1        */
          /* location of the TempModemMemory structure.                      */
          /*******************************************************************/
          memcpy(&TempModemMemory.aStoredRegs[1],
                 (PVOID)mp2,
                 sizeof(REGISTERS) );
          /*******************************************************************/
          /* Also, copy new settings to psi->ModemMemory.aStoredRegs[1]      */
          /* This updates the structure so that &v will return the updated   */
          /* profile.                                                        */
          /*******************************************************************/
          memcpy(&psi->ModemMemory.aStoredRegs[1],
                 (PVOID)mp2,
                 sizeof(REGISTERS) );
          break;

        case MWM_WRITE_PROFILE_TO_LOAD:
          /*******************************************************************/
          /* Copy the "profile to load on startup" variable to the           */
          /* TempModemMemory structure.                                      */
          /*******************************************************************/
          TempModemMemory.usStartupProfile = (USHORT)mp2;
          /*******************************************************************/
          /* Also, update the psi version of the "profile to load".          */
          /*******************************************************************/
          psi->ModemMemory.usStartupProfile = (USHORT)mp2;
          break;

        case MWM_WRITE_PROFILE_PHONE_NUM0:
          /*******************************************************************/
          /* Copy the passed phone number variable to the                    */
          /* TempModemMemory phone number 0 location                         */
          /*******************************************************************/
          memset(TempModemMemory.aachStoredNumbers[0],0,AT_CMD_BUFFER_SIZE);
          strncpy(TempModemMemory.aachStoredNumbers[0], (PVOID)mp2 , AT_CMD_BUFFER_SIZE );
          /*******************************************************************/
          /* Also, update the psi version of the stored number.              */
          /*******************************************************************/
          strncpy(psi->ModemMemory.aachStoredNumbers[0], (PVOID)mp2 , AT_CMD_BUFFER_SIZE );
          break;

        case MWM_WRITE_PROFILE_PHONE_NUM1:
          /*******************************************************************/
          /* Copy the passed phone number variable to the                    */
          /* TempModemMemory phone number 1 location                         */
          /*******************************************************************/
          memset(TempModemMemory.aachStoredNumbers[1],0,AT_CMD_BUFFER_SIZE);
          strncpy(TempModemMemory.aachStoredNumbers[1], (PVOID)mp2 , AT_CMD_BUFFER_SIZE );
          /*******************************************************************/
          /* Also, update the psi version of the stored number.              */
          /*******************************************************************/
          strncpy(psi->ModemMemory.aachStoredNumbers[1], (PVOID)mp2 , AT_CMD_BUFFER_SIZE );
          break;

        case MWM_WRITE_PROFILE_PHONE_NUM2:
          /*******************************************************************/
          /* Copy the passed phone number variable to the                    */
          /* TempModemMemory phone number 2 location                         */
          /*******************************************************************/
          memset(TempModemMemory.aachStoredNumbers[2],0,AT_CMD_BUFFER_SIZE);
          strncpy(TempModemMemory.aachStoredNumbers[2], (PVOID)mp2 , AT_CMD_BUFFER_SIZE );
          /*******************************************************************/
          /* Also, update the psi version of the stored number.              */
          /*******************************************************************/
          strncpy(psi->ModemMemory.aachStoredNumbers[2], (PVOID)mp2 , AT_CMD_BUFFER_SIZE );
          break;

        case MWM_WRITE_PROFILE_PHONE_NUM3:
          /*******************************************************************/
          /* Copy the passed phone number variable to the                    */
          /* TempModemMemory phone number 3 location                         */
          /*******************************************************************/
          memset(TempModemMemory.aachStoredNumbers[3],0,AT_CMD_BUFFER_SIZE);
          strncpy(TempModemMemory.aachStoredNumbers[3], (PVOID)mp2 , AT_CMD_BUFFER_SIZE );
          /*******************************************************************/
          /* Also, update the psi version of the stored number.              */
          /*******************************************************************/
          strncpy(psi->ModemMemory.aachStoredNumbers[3], (PVOID)mp2 , AT_CMD_BUFFER_SIZE );
          break;
      }

      /***********************************************************************/
      /* Write File back to disk.                                            */
      /***********************************************************************/
      fwrite(&TempModemMemory, sizeof(MODEMMEMORY), 1, fMemoryFile);

      /***********************************************************************/
      /* -Close memory file.                                                 */
      /***********************************************************************/
      if ( fclose(fMemoryFile) )
      {
        mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                          MWM_OS_ERROR,
                          0 );
        return (1);
      }


      /***********************************************************************/
      /* MTS 3265 ... &W wasn't waiting for profile to be written.           */
      /***********************************************************************/
      mwmParseSetDRVRSTAT(1);
      break;
  }


 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseCallback exit\n");


  return 0;
}


ULONG MWM_ENTRY mwmParseCleanupParser( VOID )


{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseCleanupParser entry\n");

  if (psi)
  {
    free(psi);
  }

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseCleanupParser exit\n");

  return 0;
}



USHORT mwmParseEchoString(STATEINFO *psi,PSZ achString)
{
  USHORT usParserStatus = 0;
  USHORT usNumChars = 0;


 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseEchoString entry\n");


  if (  (psi->usNextPPIndex + strlen(achString)/2 + 5) < PP_BUFFER_THRESHOLD)
  {
    /*************************************************************************/
    /* Add Command to suppress blank line                                    */
    /*************************************************************************/
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x0F | psi->usParserMode);
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0xFFFF;
    /*************************************************************************/
    /* Add the PP command to "Echo"                                          */
    /* Updated 11/1/93...add OR of PP Command and ParserMode.                */
    /*************************************************************************/
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (8 | psi->usParserMode);
    /*************************************************************************/
    /* Add the number of characters that will be echoed...                   */
    /*************************************************************************/
    usNumChars = strlen(achString);
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = usNumChars;
    /*************************************************************************/
    /* Copy in the character string...packing the characters into the        */
    /* word (USHORT) array.                                                  */
    /*************************************************************************/
    strcpy((char *)&psi->ausPPcmdBuffer[psi->usNextPPIndex],achString);

    swab((char *)&psi->ausPPcmdBuffer[psi->usNextPPIndex],
         (char *)&psi->ausPPcmdBuffer[psi->usNextPPIndex],((usNumChars+1)/2)*2);
    /*************************************************************************/
    /* advance the PP Index pointer.                                         */
    /*************************************************************************/
    psi->usNextPPIndex += (usNumChars+1)/2;

  }
  else
  {
    /*************************************************************************/
    /* There is not enough room left in this PP Command buffer to process    */
    /* this command....                                                      */
    /*************************************************************************/
    usParserStatus = MWM_GET_MORE_BUFFERS;
  }


 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseEchoString exit\n");



  return usParserStatus;
}



/*****************************************************************************/
/* mwmParseGetArgFromATString                                                */
/*****************************************************************************/
/* Get numeric Argument from the current position of the AT String.          */
/* If there is no numeric argument, leave the index pointer where it was.    */
/*                                                                           */
/* Returns: the argument from the string, or 0 if none.                      */
/*****************************************************************************/
USHORT mwmParseGetArgFromATString ( char *pchATString, USHORT *pusCurrentIndex )
{

  USHORT  usArg=0;
  USHORT  usEndFlag=0;
  USHORT  usTemp;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseGetArgFromATString entry\n");


  do
  {
    /*************************************************************************/
    /* If the character is not end of string...                              */
    /*************************************************************************/
    if (pchATString[*pusCurrentIndex])
    {
      /***********************************************************************/
      /* if the character is a numeric digit                                 */
      /***********************************************************************/
      if (isdigit(pchATString[*pusCurrentIndex]) )
      {
        /*********************************************************************/
        /* Handle multiple digits...multiply current value by 10, then add   */
        /* the newest digit.                                                 */
        /*********************************************************************/
        usTemp=(USHORT)(pchATString[*pusCurrentIndex]-'0');
        usArg=10*usArg+usTemp;
      }
      else
        usEndFlag=1;
    }
    else
    { /* end of string reached */
      usEndFlag=1;
    } /* endif */

    (*pusCurrentIndex)++;
  } while (!usEndFlag); /* enddo */


  (*pusCurrentIndex)--;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseGetArgFromATString exit\n");

  return usArg;
}



ULONG MWM_ENTRY mwmParseGetTaskInfo( HMTASK  hMCTLTask, HMTASK hmtProtSupr )
{
  ULONG ulRC;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseGetTaskInfo entry\n");


  /***************************************************************************/
  /* The parser needs access to several Public labels in MCTL, ASYNC, MNP    */
  /* Therefore, we must do all of                                            */
  /* the conversions up front.  These labels will be held in the DSPINFO     */
  /* portion of the psi structure.                                           */
  /***************************************************************************/
  psi->dsp.hMCTL = hMCTLTask;




  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "ATCMDBUF", &psi->dsp.dspaddrATCMDBUF);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR,
                      ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* 10/12/94                                                                */
  /* Add Buffer for Class 2 responses.                                       */
  /***************************************************************************/
  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "ATRLCBUF", &psi->dsp.dspaddrATRLCBUF);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR,
                      ulRC );
    return (ulRC);
  }

  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "PP_CMD", &psi->dsp.dspaddrPP_CMD);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "CNTLSTAT", &psi->dsp.dspaddrCNTLSTAT);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "DRVRSTAT", &psi->dsp.dspaddrDRVRSTAT);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "SREGS", &psi->dsp.dspaddrSREGS);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }


  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "OTHRREGS", &psi->dsp.dspaddrOTHRREGS);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "HKSTAT", &psi->dsp.dspaddrHKSTAT);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "IGNORES10", &psi->dsp.dspaddrIGNORES10);
  if (ulRC)
  {
/*    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,*/
/*                      MWM_DSP_ERROR, ulRC );*/
/*    return (ulRC);*/
  }

  /***************************************************************************/
  /* 11/29/95 Get Addresses of MNPITCB, RAMMITCB, and V42ITCB                */
  /***************************************************************************/
  ulRC = dspLabelToAddress(hmtProtSupr, "ITCBMNP", &psi->dsp.dspaddrMNPITCB);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  ulRC = dspLabelToAddress(hmtProtSupr, "ITCBV42", &psi->dsp.dspaddrV42ITCB);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  ulRC = dspLabelToAddress(hmtProtSupr, "ITCBRAMM", &psi->dsp.dspaddrRAMMITCB);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* 12/07/95 Get Address of FCLASS                                          */
  /***************************************************************************/
  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "FCLASS", &psi->dsp.dspaddrFCLASS);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* $4       Get Address of SUBCLASS                                        */
  /***************************************************************************/
  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "SUBCLASS", &psi->dsp.dspaddrSUBCLASS);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /*          Get Address of ITCBPSV8                                        */
  /***************************************************************************/
  ulRC = dspLabelToAddress(hmtProtSupr, "ITCBPSV8", &psi->dsp.dspaddrPSV8ITCB);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* 02/08/95 Get Address of MDMSTAT                                         */
  /***************************************************************************/
  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "MDMSTAT", &psi->dsp.dspaddrMDMSTAT);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* 02/28/96 Get Address of NEG_CNTL                                        */
  /***************************************************************************/
  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "NEG_CNTL", &psi->dsp.dspaddrNEGCNTL);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }


 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseGetTaskInfo exit\n");


  return 0;
}





/*****************************************************************************/
/* mwmParseInitializeModem                                                   */
/*****************************************************************************/
/* Consider this function like a state machine.                              */
/* Each time it is entry, the state information is passed to it via        */
/* psi->usSubCommandInfo.                                                    */
/*                                                                           */
/* The routine then puts as many initialization commands as possible into    */
/* the PP buffer, incrementing the "state" after each small group is placed  */
/* in the buffer.                                                            */
/*                                                                           */
/* When the PP command buffer is full, this function sets the flag requesting*/
/* more buffers, and returns.  It expects to be called again via the         */
/* CNTLSTAT = 2 interrupt from MDMCTL when it is thru processing this buffer.*/
/*****************************************************************************/
USHORT mwmParseInitializeModem(STATEINFO *psi, USHORT usProfileToLoad)
{
  BOOL   bBreakFlag = FALSE;
  USHORT usParserStatus = 0;
  USHORT usQuoteNParameter = 0;
  USHORT usSlashNParameter = 0;
  USHORT usTempS22Value = 0;
  USHORT usFCLASS = 0;
  ULONG  ulRC = 0;
  USHORT usCISend;
  USHORT usSUBCLASS = 0;                                                    /*$6*/
  USHORT usParm;
  ULONG  ulAddress;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseInitializeModem entry\n");


  while (!bBreakFlag)
  {
    switch (psi->usSubCommandInfo)
    {
      /***********************************************************************/
      /* Break the PP commands into small groups.  This allows us to use     */
      /* nearly all of the PP command buffer before requesting another       */
      /* interrupt with a new buffer.                                        */
      /***********************************************************************/
      case 0:                                                       /* PTR 2290*/
        /*********************************************************************/
        /* PTR 2290   11/29/93                                               */
        /* Let the Subcommand info "State variable" start at 0 as well as 1. */
        /* If psi->usSubCommandInfo is 0 when we enter this routine, we      */
        /* simply increment it to 1 before we start processing.              */
        /*********************************************************************/
        psi->usSubCommandInfo++;                                    /* PTR 2290*/
        break;
      case 1:
        /*********************************************************************/
        /* Send a software reset, and the first 3 S Register settings.       */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_S_REG_CMD_SPACE*3 + 1) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* First, add the PP command for software reset...                 */
          /* PTR 2356                                                        */
          /* 12/06/93 Don't ask for a software reset if this is an           */
          /* &F Command.                                                     */
          /*******************************************************************/
          if (psi->usCurrentCommand != MWM_AMP_F_CMD)               /* PTR 2356*/
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 7;

          /*******************************************************************/
          /* Set the first three S Registers.                                */
          /*******************************************************************/
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               0,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[0],
                               psi->ModemMemory.aStoredRegs[usProfileToLoad].S[0] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               2,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[2],
                               psi->ModemMemory.aStoredRegs[usProfileToLoad].S[2] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               3,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[3],
                               psi->ModemMemory.aStoredRegs[usProfileToLoad].S[3] );


          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;
      case 2:
        /*********************************************************************/
        /* Send the next 3 S Register settings...                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_S_REG_CMD_SPACE*3) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the first three S Registers.                                */
          /*******************************************************************/
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          4,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[4],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[4] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          5,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[5],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[5] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          6,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[6],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[6] );


          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;
      case 3:
        /*********************************************************************/
        /* Send the next 3 S Register settings...                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_S_REG_CMD_SPACE*3) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the first three S Registers.                                */
          /*******************************************************************/
          if ( (usProfileToLoad == 2) && (psi->wt.ulWTTCWD != 0)) {
            mwmParseSetSRegister(psi->ausPPcmdBuffer,
                                 &psi->usNextPPIndex,
                                 7, psi->wt.ulWTTCWD,
                                 psi->wt.ulWTTCWD );
          } else {
            mwmParseSetSRegister(psi->ausPPcmdBuffer,
                            &psi->usNextPPIndex,
                            7,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[7],
                              psi->ModemMemory.aStoredRegs[usProfileToLoad].S[7] );
          } /* endif */

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          8,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[8],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[8] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          9,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[9],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[9] );


          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus =  usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;

      case 4:
        /*********************************************************************/
        /* Send the next 3 S Register settings...                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_S_REG_CMD_SPACE*3) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the first three S Registers.                                */
          /*******************************************************************/
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          10,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[10],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[10] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          11,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[11],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[11] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          12,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[12],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[12] );


          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;

      case 5:
        /*********************************************************************/
        /* Send the next 3 S Register settings...                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_S_REG_CMD_SPACE*3) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the first three S Registers.                                */
          /*******************************************************************/
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          14,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[14],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[14] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          18,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[18],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[18] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          21,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[21],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[21] );


          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;

      case 6:
        /*********************************************************************/
        /* Send the next 3 S Register settings...                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_S_REG_CMD_SPACE*4) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the next three S Registers.                                 */
          /*******************************************************************/

          /*******************************************************************/
          /* Query the "Dial Tone Detect Preferred" bit from the WTTCPM word */
          /* of the World Trade Table.                                       */
          /*******************************************************************/
          /*******************************************************************/
          /* MTS 3444 Integrate World Trade Details.                         */

          /*******************************************************************/
          /* MTS 3687                                                        */
          /* Copy the S22 value to a temporary location.  If the World Trade */
          /* parameter "Dial & Busy Detect NOT preferred" is set, then we    */
          /* we will change the S22 value to set X1 instead of X4            */
          /*******************************************************************/
          usTempS22Value = psi->ModemMemory.aStoredRegs[usProfileToLoad].S[22];

          /*******************************************************************/
          /* MTS 3789                                                        */
          /* WTTCPM                                                          */
          /* Bit 8 = Busy Detect NOT Preferred.                              */
          /* Bit 9 = Dial Detect NOT Preferred.                              */
          /*                                                                 */
          /* X Command Defaults...                                           */
          /*                                                                 */
          /*   8  9     X    Mask                                            */
          /* ---------------------                                           */
          /*   0  0     4   0xffbf                                           */
          /*   0  1     3   0xffef                                           */
          /*   1  0     2   0xffdf                                           */
          /*   1  1     1   0xffcf                                           */
          /*******************************************************************/
          if  (psi->wt.ulWTTCPM & 0x200)
          {
            /*****************************************************************/
            /* Bit 9 set                                                     */
            /*****************************************************************/
            if  (psi->wt.ulWTTCPM & 0x100)
            {
              /***************************************************************/
              /* Bit 9 set                                                   */
              /* Bit 8 set                                                   */
              /***************************************************************/
              usTempS22Value =
                psi->ModemMemory.aStoredRegs[usProfileToLoad].S[22] & 0xffcf;
            }
            else
            {
              /***************************************************************/
              /* Bit 9 set                                                   */
              /* Bit 8 NOT set                                               */
              /* Dial Tone Detect NOT preferred.                             */
              /* Busy Tone Detect preferred.                                 */
              /***************************************************************/
              usTempS22Value =
                psi->ModemMemory.aStoredRegs[usProfileToLoad].S[22] & 0xffef;
            }
          }
          else
          {
            if  (psi->wt.ulWTTCPM & 0x100)
            {
              /***************************************************************/
              /* Bit 9 NOT set                                               */
              /* Bit 8 set                                                   */
              /* Dial Tone Detect preferred.                                 */
              /* Busy Tone Detect NOT preferred.                             */
              /***************************************************************/
              usTempS22Value =
                psi->ModemMemory.aStoredRegs[usProfileToLoad].S[22] & 0xffdf;
            }
          }

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                               &psi->usNextPPIndex,
                               22,usTempS22Value,
                               usTempS22Value );

          /*******************************************************************/
          /* This is the speaker S Register...post a message to the main     */
          /* window asking it to update the speaker.                         */
          /*******************************************************************/
#ifndef LINUX_BLD
	  /* if we had an integrated UI we could post a SPEAKER CHANGE message here */
#endif

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          23,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[23],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[23] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          24,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[24],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[24] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          25,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[25],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[25] );


          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;

      case 7:
        /*********************************************************************/
        /* Send the last 2 S Register settings...                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_S_REG_CMD_SPACE*2) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the first three S Registers.                                */
          /*******************************************************************/
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          27,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[27],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[27] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          28,psi->ModemMemory.aStoredRegs[usProfileToLoad].S[28],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].S[28] );

          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;
      case 8:
        /*********************************************************************/
        /* Send the BackSlash A,K, and L commands                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_SLASH_A_CMD_SPACE +
                                   PP_SLASH_K_CMD_SPACE +
                                   PP_SLASH_L_CMD_SPACE )
              < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the Slash A parameter                                       */
          /*******************************************************************/
          usParserStatus = mwmSlashACommand(psi,
                              psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashA );

          /*******************************************************************/
          /* Set the Slash K parameter                                       */
          /*******************************************************************/
          usParserStatus = mwmSlashKCommand(psi,
                              psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashK );

          /*******************************************************************/
          /* Set the Slash L parameter                                       */
          /*******************************************************************/
          usParserStatus = mwmSlashLCommand(psi,
                              psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashL );

          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;
      case 9:
        /*********************************************************************/
        /* Send the BackSlash N and T                                        */
        /* PTR 2497 ... Add Initialization of the &N                         */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_SLASH_N_CMD_SPACE +
                                   PP_SLASH_T_CMD_SPACE +
                                   PP_AMP_N_CMD_SPACE )
              < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the Slash N parameter                                       */
          /* allow anything if both V.42 and MNP are loaded.                 */
          /*******************************************************************/
          if ((psi->ulFeaturesToLoad & FEATURE_V42) &&              /* DCR 2487*/
              (psi->ulFeaturesToLoad & FEATURE_MNP) )               /* DCR 2487*/
          {
            usSlashNParameter = psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashN;
          }
          /*******************************************************************/
          /* MTS 3009 Don't allow \N2 or \N3 if MNP not loaded.              */
          /* 3/30/94                                                         */
          /*******************************************************************/
          else if (psi->ulFeaturesToLoad & FEATURE_V42)             /* MTS 3009*/
          {
            switch (psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashN)
            {
              case 2:
              case 3:
                usSlashNParameter = 0;
                break;
              case 6:
                usSlashNParameter = 4;
                break;
              case 7:
                usSlashNParameter = 5;
                break;
              default:
                usSlashNParameter = psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashN;
            }
          }
          /*******************************************************************/
          /* MTS 3009 Don't allow   >\N3 if V42 not loaded.                  */
          /* 3/30/94                                                         */
          /*******************************************************************/
          else if (psi->ulFeaturesToLoad & FEATURE_MNP)             /* MTS 3009*/
          {
            if (psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashN > 3)
            {
              usSlashNParameter = 3;
            }
            else
              usSlashNParameter = psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashN;
          }
          else
            usSlashNParameter = 0;

          usParserStatus = mwmSlashNCommand(psi,usSlashNParameter);


          /*******************************************************************/
          /* Set the Slash T parameter                                       */
          /*******************************************************************/
          mwmSlashTCommand(psi,
                           psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashT );



          /*******************************************************************/
          /* PTR 2497...Add Initialization of the &N parameter.              */
          /* This should have been a part of PTR 592.  For some reason it    */
          /* Either was not implemented, or was lost at some point.          */
          /*******************************************************************/
          usParserStatus = mwmAmpNCommand(psi,                      /* PTR 2497*/
                           psi->ModemMemory.aStoredRegs[usProfileToLoad].AmpN );

          /*******************************************************************/
          /* PTR 2290...this is no longer the last state, so we are          */
          /* the state variable just as in every other state.                */
          /* Increment the state variable for the next state...              */
          /*******************************************************************/
          psi->usSubCommandInfo++;                                  /* PTR 2290*/
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;
      case 10:                                                      /* PTR 2290*/
        /*********************************************************************/
        /* Send the Percent A and C                                          */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_PERCENT_A_CMD_SPACE +         /* PTR 2290*/
                                   PP_PERCENT_C_CMD_SPACE +
                                   PP_PERCENT_E_CMD_SPACE           /* DCR 2534*/
                                                        )           /* PTR 2290*/
              < PP_BUFFER_THRESHOLD)                                /* PTR 2290*/
        {                                                           /* PTR 2290*/
          /*******************************************************************/
          /* Set the %A parameter                                            */
          /*******************************************************************/
          mwmPercentACommand(psi,                                   /* PTR 2290*/
                             psi->ModemMemory.aStoredRegs[usProfileToLoad].PercentA );

          /*******************************************************************/
          /* Set the %C parameter                                            */
          /*******************************************************************/
          mwmPercentCCommand(psi,                                   /* PTR 2290*/
                           psi->ModemMemory.aStoredRegs[usProfileToLoad].PercentC );

          /*******************************************************************/
          /* Set the %E parameter                                            */
          /*******************************************************************/
          mwmPercentECommand(psi,                                   /* PTR 2534*/
                           psi->ModemMemory.aStoredRegs[usProfileToLoad].PercentE );

          /*******************************************************************/
          /* the state variable just as in every other state.                */
          /* Increment the state variable for the next state...              */
          /*******************************************************************/
          psi->usSubCommandInfo++;                                  /* PTR 2290*/
        }                                                           /* PTR 2290*/
        else                                                        /* PTR 2290*/
        {                                                           /* PTR 2290*/
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;   /* PTR 2290*/
          bBreakFlag = TRUE;                                        /* PTR 2290*/
        }                                                           /* PTR 2290*/
        break;                                                      /* PTR 2290*/
      case 11:                                                      /* PTR 2290*/
        /*********************************************************************/
        /* Send the "H, "N, and "O commands...                               */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_QUOTE_H_CMD_SPACE +           /* PTR 2290*/
                                   PP_QUOTE_N_CMD_SPACE +           /* PTR 2290*/
                                   PP_QUOTE_O_CMD_SPACE )           /* PTR 2290*/
              < PP_BUFFER_THRESHOLD)                                /* PTR 2290*/
        {                                                           /* PTR 2290*/
/*          if (psi->ulInstalledOptions & MWM_V42BIS_INSTALLED)*/
          if (psi->ulFeaturesToLoad & FEATURE_V42BIS)          /* DCRs 2416,2487*/
          {
            /*****************************************************************/
            /* If V.42bis is installed, then we initialize it as it should be*/
            /*****************************************************************/
            /* Set the "H parameter                                          */
            /*****************************************************************/
            mwmQuoteHCommand(psi,                                     /* PTR 2290*/
                             psi->ModemMemory.aStoredRegs[usProfileToLoad].QuoteH );

            /*****************************************************************/
            /* Set the "N parameter                                          */
            /*****************************************************************/
            /*****************************************************************/
            /* MTS 3059                                                      */
            /* Make this command depend on the [FEATURES} V42BIS= INI keyword*/
            /* 03/11/97 DR - Check the flag that determines whether we should*/
            /* force a 512-word V.42bis dictionary.                          */
            /*****************************************************************/
            if (psi->usForce512WordDictionary) {
              usQuoteNParameter = 0;
            } else {
              switch (psi->ulFeaturesToLoad & FEATURE_V42BIS)
              {
                case FEATURE_V42BIS_512:
                  usQuoteNParameter = 0;
                  break;
                case FEATURE_V42BIS_1024:
                  if (psi->ModemMemory.aStoredRegs[usProfileToLoad].QuoteN > 1)
                    usQuoteNParameter = 1;
                  else
                    usQuoteNParameter = psi->ModemMemory.aStoredRegs[usProfileToLoad].QuoteN;
                  break;
                case FEATURE_V42BIS_2048:
                  if (psi->ModemMemory.aStoredRegs[usProfileToLoad].QuoteN > 2)
                    usQuoteNParameter = 2;
                  else
                    usQuoteNParameter = psi->ModemMemory.aStoredRegs[usProfileToLoad].QuoteN;
                  break;
                case FEATURE_V42BIS_4096:
                  if (psi->ModemMemory.aStoredRegs[usProfileToLoad].QuoteN > 3)
                    usQuoteNParameter = 3;
                  else
                    usQuoteNParameter = psi->ModemMemory.aStoredRegs[usProfileToLoad].QuoteN;
                  break;
              }
            } /* endif */

            mwmQuoteNCommand(psi,                                   /* PTR 2290*/
                             usQuoteNParameter );
            /*****************************************************************/
            /* Set the "O parameter                                          */
            /*****************************************************************/
            mwmQuoteOCommand(psi,                                   /* PTR 2290*/
                             psi->ModemMemory.aStoredRegs[usProfileToLoad].QuoteO );
          }
          else
          {
            /*****************************************************************/
            /* if V.42bis is NOT installed, we initialize all of the V.42bis */
            /* parameters to 0                                               */
            /*****************************************************************/
            /*****************************************************************/
            /* Set the "H parameter                                          */
            /*****************************************************************/
            mwmQuoteHCommand(psi, 0 );
            /*****************************************************************/
            /* Set the "N parameter                                          */
            /*****************************************************************/
            mwmQuoteNCommand(psi, 0 );
            /*****************************************************************/
            /* Set the "O parameter                                          */
            /*****************************************************************/
            mwmQuoteOCommand(psi, 0 );
          }

          /*******************************************************************/
          /* the state variable just as in every other state.                */
          /* Increment the state variable for the next state...              */
          /*******************************************************************/
          psi->usSubCommandInfo++;                                  /* DCR 2487*/
        }                                                           /* PTR 2290*/
        else                                                        /* PTR 2290*/
        {                                                           /* PTR 2290*/
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;   /* PTR 2290*/
          bBreakFlag = TRUE;                                        /* PTR 2290*/
        }                                                           /* PTR 2290*/
        break;                                                      /* PTR 2290*/

      case 12:
        /*********************************************************************/
        /* Send the next 5 S Register settings...                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_S_REG_CMD_SPACE*5) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************/
          /* Set the next  5 S Registers.                                    */
          /*******************************************************************/
          /* MTS6384 - S[30..] moved to SV34[0..].  We may later maintain these*/
          /*           registers in the PC but for now send them on*/
          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          30,psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[0],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[0] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          31,psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[1],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[1] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          32,psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[2],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[2] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          33,psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[3],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[3] );

          mwmParseSetSRegister(psi->ausPPcmdBuffer,
                          &psi->usNextPPIndex,
                          34,psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[4],
                            psi->ModemMemory.aStoredRegs[usProfileToLoad].SV34[4] );

          /* CMVC 1411 - S54 and S55 cannot be set in profiles*/
          /* They are always set to zero.*/
          if(!dspLabelToAddress(psi->dsp.hMCTL, "SREG35", &ulAddress))
          {
             usParm = 0;

             dspMemTransfer(psi->dsp.hDSP,
             ulAddress,
             &usParm,
             1,
             DSP_MEMXFER_DATA_WRITE);

             dspMemTransfer(psi->dsp.hDSP,
             ulAddress + 2,
             &usParm,
             1,
             DSP_MEMXFER_DATA_WRITE);
          }

          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;

      case 13:
        /*********************************************************************/
        /* Send the XON/XOFF characters...                                   */
        /*                                                                   */
        /* MTS 6555 - Added 05/20/96                                         */
        /*********************************************************************/
        /* MTS 6937                                                          */
        /* 05/21/96 Tim ... Add Reset of V.23                                */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_XON_XOFF_CMD_SPACE+
                                   2 ) < PP_BUFFER_THRESHOLD)
        {
          /*******************************************************************
          ** Set the locally-maintained S32 and S33 register values.
          ********************************************************************/
          psi->usS32Value = psi->ModemMemory.aStoredRegs[usProfileToLoad].S32;
          psi->usS33Value = psi->ModemMemory.aStoredRegs[usProfileToLoad].S33;

          /*******************************************************************
          ** Send the values via PP Cmd 0x2B, with parameter 1 = 0x10 and
          ** parameter 2 = XON/XOFF characters in the high byte/low byte
          ********************************************************************/
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x2B | psi->usParserMode);
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0x10;
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (psi->usS32Value << 8) | psi->usS33Value;

          /*******************************************************************
          ** Set the locally-maintained W Command register value.
          ********************************************************************/
          psi->usWValue = psi->ModemMemory.aStoredRegs[usProfileToLoad].WCommand;

          /*******************************************************************
          ** Set the locally-maintained S60 register value.
          ** The default value will always come from the Word Trade Table.
          ********************************************************************/
          psi->usS60Value = (psi->wt.ulWTT34FBR & 0x80) >> 7;

          /*******************************************************************/
          /* MTS 6937                                                        */
          /* 05/21/96 Tim ... Reset V.23 if it is enabled.                   */
          /*******************************************************************/
          if (psi->ulFeaturesToLoad & FEATURE_V23)
          {
            /* Turn off V.23 */
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode);
            psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 76;
          }
          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;

      case 14:
        /*********************************************************************/
        /* Reset the default Class2 values...                                */
        /*                                                                   */
        /* 12/09/96 - Class2 Fax Rewrite                                     */
        /*********************************************************************/
        psi->Class2Capabilities.usVR = 0;
        psi->Class2Capabilities.usBR = 7;
        psi->Class2Capabilities.usWD = 0;
        psi->Class2Capabilities.usLN = 2;
        psi->Class2Capabilities.usDF = 0;
        psi->Class2Capabilities.usEC = 0;
        psi->Class2Capabilities.usBF = 0;
        psi->Class2Capabilities.usST = 0;
        psi->Class2Capabilities.usFMINSP = 0;
        psi->Class2Capabilities.usFCR    = 0;
        psi->Class2Capabilities.usFCQ    = 1;
        psi->Class2Capabilities.usFPTS   = 1;
        psi->Class2Capabilities.usFSPL   = 0;
        psi->Class2Capabilities.usFLPL   = 0;
        psi->Class2Capabilities.usFPHCTO = 30;
        psi->TempClass2Capabilities = psi->Class2Capabilities;
        usParserStatus = mwmClss2WriteTaskData( psi, psi->dsp.dspaddrCAPBUFF,
                                                &psi->Class2Capabilities, 15,
                                                szThisFile, __LINE__ );
        if (!usParserStatus) {
          memset( psi->achFLIDString, ' ', sizeof(psi->achFLIDString));
          usParserStatus = mwmClss2WriteTaskData( psi, psi->dsp.dspaddrMYID,
                                                  psi->achFLIDString, 10,
                                                  szThisFile, __LINE__ );
        } /* endif */

        if (!usParserStatus) {
          memset( psi->achFCIGString, ' ', sizeof(psi->achFCIGString));
          usParserStatus = mwmClss2WriteTaskData( psi, psi->dsp.dspaddrMYCIG,
                                                  psi->achFCIGString, 10,
                                                  szThisFile, __LINE__ );
        } /* endif */

        psi->usSubCommandInfo++;
      break;

      case 15:
        /*********************************************************************/
        /* DCR 2823                                                          */
        /* Send the -J command...                                            */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + PP_DASH_J_CMD_SPACE  +           /* DCR 2487*/
                                   PP_SLASH_Q_CMD_SPACE +           /* MTS 3017*/
                                   PP_DASH_SSE_CMD_SPACE +
                                   PP_FCLASS_CMD_SPACE +
                                   3 +
                                   PP_STAR_TH_CMD_SPACE +           /* MTS 4594*/
                                   PP_ES_CMD_SPACE +                /*$6*/
                                   PP_IBC_CMD_SPACE)                /*$6*/
              < PP_BUFFER_THRESHOLD)
        {
          if (psi->ulFeaturesToLoad & FEATURE_V42)            /* DCRs 2416,2487*/
          {
            /*****************************************************************/
            /* If V.42    is installed, then we initialize it as it should be*/
            /*****************************************************************/
            /* Set the -J parameter                                          */
            /*****************************************************************/
            mwmDashJCommand(psi,                                     /* PTR 2487*/
                             psi->ModemMemory.aStoredRegs[usProfileToLoad].DashJ );
          }

          /*******************************************************************/
          /* Set the \Q Command.                                             */
          /*******************************************************************/
          mwmSlashQCommand(psi,                                     /* PTR 3017*/
                           psi->ModemMemory.aStoredRegs[usProfileToLoad].SlashQ );

          /*******************************************************************/
          /* if AMPFTODATA && (DEBUG-CLASS2 = 0) then return to data mode    */
          /* when &f, or z is issued.                                        */
          /*******************************************************************/
          if ( (psi->usAmpFReturnToDataMode) &&
               (!psi->usLeaveClass2Loaded)   &&
               (!psi->usSuspendFlag) )            /* Resume case...*/
          {
            psi->bVoiceModeEntered = FALSE;

            ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrFCLASS,
                                  &usFCLASS, 1,
                                  DSP_MEMXFER_DATA_READ);
            if (ulRC!=DSP_NOERROR)
            {
              mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                                MWM_DSP_ERROR, ulRC );
              return MWM_ATCMD_ERROR;
            }

            usSUBCLASS = 0;                                                    /*$6 beg*/
            if (psi->ulFeaturesToLoad & FEATURE_V80)
            {
              ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrSUBCLASS,
                                    &usSUBCLASS,1,
                                    DSP_MEMXFER_DATA_READ);
              if (ulRC!=DSP_NOERROR)
              {
                mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                                  MWM_DSP_ERROR, ulRC );
                return MWM_ATCMD_ERROR;
              }
            }                                                                  /*$6 end*/

            psi->usV80StateSaved = FALSE;
            if (usFCLASS)
            {
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0x09;
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] = usFCLASS;
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0;
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode);
              psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 36;
            }
            else if (usSUBCLASS == 1)                                          /*$6 beg*/
            {
               /*********************************************************************/
               /* Send the +ES default parms                                        */
               /*********************************************************************/
               mwmv80PESCommand(psi,psi->ModemMemory.aStoredRegs[usProfileToLoad].PlusES, TRUE);
               /*********************************************************************/
               /* Send the +IBC default parms                                       */
               /*********************************************************************/
               mwmv80PIBCCommand(psi,psi->ModemMemory.aStoredRegs[usProfileToLoad].PlusIBC);
            }                                                                  /*$6 end*/
          }


          /*******************************************************************/
          /* MTS 4594                                                        */
          /* Set the *TH Command.                                            */
          /*******************************************************************/
          mwmStarTHCommand(psi,
                           psi->ModemMemory.aStoredRegs[usProfileToLoad].StarTH );

          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;
      case 16:  /* New state added for #CID */
        /*********************************************************************/
        /* Init class 8 Phone-In AGC setting                                 */
        /*********************************************************************/

        /*********************************************************************/
        /* We need enough space in the PP buffer for AGC, SDR & CID resets   */
        /*********************************************************************/
        if ( (psi->usNextPPIndex + 6) < PP_BUFFER_THRESHOLD)
        {
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x24 | psi->usParserMode);
          psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 78;

          /*********************************************************************/
          /* Send the #CID setting...                                          */
          /*********************************************************************/
          mwmCidpPoundCIDCommand(psi,
                           psi->ModemMemory.aStoredRegs[usProfileToLoad].PoundCID );

          /*********************************************************************/
          /* Send the -SDR setting...                                          */
          /*********************************************************************/
          mwmCidpDashSDRCommandSet(psi,
                           psi->ModemMemory.aStoredRegs[usProfileToLoad].DashSDR );

          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;                                         /*$1*/
        }
        else
        {
          /*******************************************************************/
          /* Set the flag to request a new buffer from MDMCTL, and return.   */
          /*******************************************************************/
          usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
          bBreakFlag = TRUE;
        }
        break;

      case 17:  /*$1*/
        if (psi->ulFeaturesToLoad & FEATURE_V80) /*$3*/
        {
          if ( (psi->usNextPPIndex + PP_A8E_CMD_SPACE) < PP_BUFFER_THRESHOLD)
          {
            /*********************************************************************/
            /* Send the +A8E default parms                                       */
            /*********************************************************************/
            mwmv80PA8ECommand(psi,psi->ModemMemory.aStoredRegs[usProfileToLoad].PlusA8E);

            /*****************************************************************
            ** Clear CISEND
            ******************************************************************/
            usCISend = 0;
            ulRC = dspMemTransfer( psi->dsp.hDSP, (psi->dsp.dspaddrPSV8ITCB)+ offsetof(PSV8ITCB, CISEND),
                                   &usCISend, 1,
                                   DSP_MEMXFER_DATA_WRITE);
            if (ulRC!=DSP_NOERROR)
            {
              mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                                MWM_DSP_ERROR, ulRC );
              return MWM_ATCMD_ERROR;
            }

            /*******************************************************************/
            /* Increment the SubcommandInfo                                    */
            /* This effectively increments the state machine to the next state.*/
            /*******************************************************************/
            psi->usSubCommandInfo++;
          }
          else
          {
            /*******************************************************************/
            /* Set the flag to request a new buffer from MDMCTL, and return.   */
            /*******************************************************************/
            usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
            bBreakFlag = TRUE;
          }
        }
        else /* not FEATURE_V80 */
        {
          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        } /* endif */
        break;

      case 18:  /*$1*/
        if (psi->ulFeaturesToLoad & FEATURE_V80) /*$3*/
        {
          if ( (psi->usNextPPIndex + PP_ESA_CMD_SPACE) < PP_BUFFER_THRESHOLD)
          {
            /*********************************************************************/
            /* Send the +ESA default parms                                       */
            /*********************************************************************/
            mwmv80PESACommand(psi,psi->ModemMemory.aStoredRegs[usProfileToLoad].PlusESA);
            /*******************************************************************/
            /* Increment the SubcommandInfo                                    */
            /* This effectively increments the state machine to the next state.*/
            /*******************************************************************/
            psi->usSubCommandInfo++;
          }
          else
          {
            /*******************************************************************/
            /* Set the flag to request a new buffer from MDMCTL, and return.   */
            /*******************************************************************/
            usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
            bBreakFlag = TRUE;
          }
        }
        else /* not FEATURE_V80 */
        {
          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        } /* endif */
        break;

      case 19:  /*$1*/
        if (psi->ulFeaturesToLoad & FEATURE_V80) /*$3*/
        {
          if ( (psi->usNextPPIndex + PP_ITF_CMD_SPACE) < PP_BUFFER_THRESHOLD)
          {
            /*********************************************************************/
            /* Send the +ITF default parms                                       */
            /*********************************************************************/
            mwmv80PITFCommand(psi,psi->ModemMemory.aStoredRegs[usProfileToLoad].PlusITF);
            /*******************************************************************/
            /* Increment the SubcommandInfo                                    */
            /* This effectively increments the state machine to the next state.*/
            /*******************************************************************/
            psi->usSubCommandInfo++;
          }
          else
          {
            /*******************************************************************/
            /* Set the flag to request a new buffer from MDMCTL, and return.   */
            /*******************************************************************/
            usParserStatus = usParserStatus | MWM_GET_MORE_BUFFERS;
            bBreakFlag = TRUE;
          }
        }
        else /* not FEATURE_V80 */
        {
          /*******************************************************************/
          /* Increment the SubcommandInfo                                    */
          /* This effectively increments the state machine to the next state.*/
          /*******************************************************************/
          psi->usSubCommandInfo++;
        } /* endif */
        break;

      case 20:  /*$3*/
        /*******************************************************************/
        /* This is the last state of the Initialize Modem command, so      */
        /* reset the state information.                                    */
        /*******************************************************************/
        psi->usCurrentCommand = 0;
        psi->usSubCommandInfo = 0;

        /***********************************************************************/
        /* Reset #ud data                                                      */
        /***********************************************************************/
        mwmPoundUDClearCallStatus(TRUE);

        /*******************************************************************/
        /* We are finished with the suspend flag, so reset it here.        */
        /* Move this code from the RESUME case of mwmSpcfcNSClientProc     */
        /*******************************************************************/
        psi->usSuspendFlag = 0;
        bBreakFlag = TRUE;
        break;

      default:
      {
        bBreakFlag = TRUE;
      }
    }
  }

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseInitializeModem exit\n");

  return usParserStatus;
}


/*****************************************************************************/
/* mwmParseInitializeParser                                                  */
/*****************************************************************************/
/*                                                                           */
/*                                                                           */
/* Function of this routine...                                               */
/*   - Get fully qualified filename of Modem Memory file from .INI           */
/*   - Load Profile information from stored file into the ModemMemory        */
/*     portion of the psi structure.                                         */
/*   - Get all of the addresses to the public dsp labels needed by the       */
/*     parser at interrupt time.                                             */
/*****************************************************************************/
ULONG MWM_ENTRY mwmParseInitializeParser( HWND ulWindowHandle,
                                          PVOID INIFile )
{
  int        iSuccess;
  ULONG      ulRC;
  USHORT     usSize = 0;
  RMGRINFO  *pMgrInfoBuffer=0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseInitializeParser entry\n");


  /***************************************************************************/
  /* Allocate State Information structure.  This memory block will be shared */
  /* by the entire AT parser.                                                */
  /***************************************************************************/
  psi = (STATEINFO *)calloc(1,sizeof(STATEINFO) );

  if (!psi)
  {
    mwmIPCHandleError(szThisFile,__LINE__,NULL,
                      MWM_OS_ERROR, 0 );
    return (1);
  }


  psi->ulMainWindowHandle = ulWindowHandle;

  strncpy(psi->achINIFileName,(char *)INIFile, MWM_MAX_FILENAME_LENGTH );



  /***************************************************************************/
  /* Determine if we need to use the pmprintf type debug information.        */
  /***************************************************************************/
/*  iSuccess= GetPrivateProfileString("DEBUG","PRINTF", "0",*/
/*                                    psi->achMemoryFileName,*/
/*                                    MWM_MAX_FILENAME_LENGTH,*/
/*                                    psi->achINIFileName);*/
  iSuccess = GetPrivateProfileInt("DEBUG","PRINTF",0,psi->achINIFileName );
  if (iSuccess > 0)
  {
  }

  /* Remove checks for installed features   DCR 2487*/




  /***************************************************************************/
  /* Get Filename of Modem Memory file from the .INI file.                   */
  /* The Name specified in the INI file should be fully qualified.           */
  /***************************************************************************/
  /* paulsch: fixme*/
  iSuccess= GetPrivateProfileString("COMMFILE","MEMFILE", "none",
                                    psi->achMemoryFileName,
                                    MWM_MAX_FILENAME_LENGTH,
                                    psi->achINIFileName);
				    
  if ((iSuccess == 0)||((strcmp("none",psi->achMemoryFileName))==0))
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_OS_ERROR, 0 );
    return (1);
  }

  /* Note: The opeining and reading of the memory profile is now handled
           by mwmParseReadMemoryProfile                                  */
  /***************************************************************************/
  /* Open the Memory file                                                    */
  /***************************************************************************/
/*  fMemoryFile = fopen(psi->achMemoryFileName, "rb");

  if (!fMemoryFile)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_OS_ERROR, 0 );
    return (1);
  }*/

  /***************************************************************************/
  /* Read Modem Memory information into Modem Memory portion of psi struct.  */
  /***************************************************************************/
/*
  if (!fread(&psi->ModemMemory,sizeof(MODEMMEMORY),1,fMemoryFile) )
  {
    if ( ferror(fMemoryFile) )
    {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_OS_ERROR, 0 );
      return (1);
    }
  }
*/
  /***************************************************************************/
  /* Close Memory input file.                                                */
  /***************************************************************************/
/*
  if ( fclose(fMemoryFile) )
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_OS_ERROR, 0 );
    return (1);
  }*/

  ulRC = mwmParseReadMemoryProfile( psi );
  if (ulRC)
    return(ulRC);

  /***************************************************************************/
  /* Get DSP Handle, put it into psi structure....                           */
  /***************************************************************************/
  /* Get the DSP handle                                                      */
  /*   -First get the size of the buffer to be returned.                     */
  /*   -Allocate the buffer                                                  */
  /*   -Query the information again supplying the correct buffer size        */
  /***************************************************************************/

  ulRC = dspQueryManagerInfo(&usSize, pMgrInfoBuffer);

  if (ulRC != DSP_INS_BUFFER)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  /***************************************************************************/
  /* MTS 3525                                                                */
  /* Allocate more than needed...in case someone else is changing the system */
  /* before we get a chance to complete the query.                           */
  /***************************************************************************/
  usSize += 20;
 
  pMgrInfoBuffer = (RMGRINFO *)calloc( 1,usSize );

  if (!pMgrInfoBuffer)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_OS_ERROR, 0 );
    return (1);
  }

  ulRC = dspQueryManagerInfo(&usSize, pMgrInfoBuffer);
  if (ulRC != DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* Put the DSP handle into the psi structure.                              */
  /***************************************************************************/
  psi->dsp.hDSP = pMgrInfoBuffer -> MGRINFO_phDSP[0];

  /***************************************************************************/
  /* Free the Manager Info buffer...we are finished with it.                 */
  /***************************************************************************/
  free(pMgrInfoBuffer);


  /***************************************************************************/
  /* MTS 2944                                                                */
  /* The Modem Version displayed by ATI3 will read: Mwave Modem "......"     */
  /* The text following Mwave Modem will contain this information.           */
  /***************************************************************************/
  iSuccess= GetPrivateProfileString("VERSIONS","MODEM", "unknown",
                                    psi->achModemVersion,
                                    MWM_VERSION_STRING_LENGTH,
                                    psi->achINIFileName);

  /* 06/02/98 DR - Get the ID String from the INI file as well */
  iSuccess= GetPrivateProfileString("VERSIONS", "ID", "IBM Mwave Modem",
                                    psi->achIDString,
                                    MWM_VERSION_STRING_LENGTH,
                                    psi->achINIFileName);

  /***************************************************************************/
  /* MTS 2944                                                                */
  /* The ATI4 command is to be used by OEMs to specify their version of the  */
  /* Mwave Modem.                                                            */
  /***************************************************************************/
  iSuccess= GetPrivateProfileString("VERSIONS","OEM", "Generic Mwave",
                                    psi->achOEMVersion,
                                    MWM_VERSION_STRING_LENGTH,
                                    psi->achINIFileName);

  /*************************************************************************
  ** The speed values reported by ATI0 will be read from the INI file.
  ** Added 02/27/97 DR
  **************************************************************************/
  iSuccess= GetPrivateProfileString("VERSIONS","STRINGPCM", "56000",
                                    psi->achPCMString,
                                    MWM_SPEED_STRING_LENGTH,
                                    psi->achINIFileName);

  iSuccess= GetPrivateProfileString("VERSIONS","STRING336", "33600",
                                    psi->ach336String,
                                    MWM_SPEED_STRING_LENGTH,
                                    psi->achINIFileName);

  iSuccess= GetPrivateProfileString("VERSIONS","STRING288", "28800",
                                    psi->ach288String,
                                    MWM_SPEED_STRING_LENGTH,
                                    psi->achINIFileName);

  iSuccess= GetPrivateProfileString("VERSIONS","STRING144", "14400",
                                    psi->ach144String,
                                    MWM_SPEED_STRING_LENGTH,
                                    psi->achINIFileName);

  iSuccess= GetPrivateProfileString("VERSIONS","STRING2400", "2400",
                                    psi->ach2400String,
                                    MWM_SPEED_STRING_LENGTH,
                                    psi->achINIFileName);

  psi->usCNFGBUF1 = GetPrivateProfileInt("FEATURES","CNFGBUF1",0x1442,psi->achINIFileName );

  psi->usCNFGBUF2 = GetPrivateProfileInt("FEATURES","CNFGBUF2",0x4038,psi->achINIFileName );



  psi->usLeaveClass2Loaded = GetPrivateProfileInt("DEBUG","CLASS2",0,psi->achINIFileName );

  psi->usAmpFReturnToDataMode = GetPrivateProfileInt("DEBUG","AMPFTODATA",0,psi->achINIFileName );

  psi->usAllowA8EActionCommand = GetPrivateProfileInt("FEATURES","A8EACTION",0,psi->achINIFileName );

  psi->usClass0RestoreV80State = GetPrivateProfileInt("FEATURES","CLASS0V80RESTORE",0,psi->achINIFileName );

  psi->bPoundUDDecimal = GetPrivateProfileInt("DEBUG", "POUNDUDDEC", 0, psi->achINIFileName );

  psi->usV80THValue = GetPrivateProfileInt("OPTIONS", "V80THVALUE", 0, psi->achINIFileName );

  /******************************************************************
  ** If SYNTH_SPEED=33600, then we can assume that we need to
  ** reserve some storage space for concurruent BigSynth at 33600,
  ** and should therefore force the V.42bis dictionary size to
  ** limit to 512 words.
  ** Added 03/11/97 DR
  *******************************************************************/
  if ( (GetPrivateProfileInt("STARTUP", "SYNTH_SPEED", 14400, psi->achINIFileName)) == 33600 ) {
    psi->usForce512WordDictionary = TRUE;
  } else {
    psi->usForce512WordDictionary = FALSE;
  } /* endif */

  /***************************************************************************/
  /* MTS 6763  05/06/96  change typo 37 to 73...                             */
  /***************************************************************************/
  iSuccess= GetPrivateProfileString("CLASS1","CAPABILITIES", "3,24,48,72,73,96,97,98,121,122,145,146",
                                    psi->achClass1Capabilities,
                                    sizeof(psi->achClass1Capabilities),
                                    psi->achINIFileName);

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseInitializeParser exit\n");


  return 0;
}

ULONG MWM_ENTRY mwmParseQueryDebugSettings(USHORT *pusDebugWindow)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryDebugSettings entry\n");

  if (psi)
  {
        #ifdef DEBUG
    pusDebugWindow = (SHORT *)psi->usDebugWindow;
        #endif
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryDebugSettings exit\n");

  return 0;
}

ULONG MWM_ENTRY mwmParseQueryNumberDialed(char *pszNumberDialed)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryNumberDialed entry\n");

  if (psi)
  {
    strcpy(pszNumberDialed,psi->achNumberDialed);
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryNumberDialed exit\n");

  return 0;
}

ULONG MWM_ENTRY mwmParseSetNumberDialed(char *pszNumberDialed)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetNumberDialed entry\n");

  if (psi)
  {
    memset(psi->achNumberDialed, 0, AT_CMD_BUFFER_SIZE);
    strcpy(psi->achNumberDialed,pszNumberDialed);
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetNumberDialed exit\n");

  return 0;
}


ULONG MWM_ENTRY mwmParseSetDCSAddress(ULONG ulDCSAddress)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetDCSAddress entry\n");

  if (psi)
  {
    psi->dsp.dspaddrDCS = ulDCSAddress;
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetDCSAddress exit\n");

  return 0;
}

ULONG MWM_ENTRY mwmParseSetCAPBUFFAddress(ULONG ulCAPBUFFAddress)
{
  ULONG ulRC = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetCAPBUFFAddress entry\n");


  if (psi)
  {
    psi->dsp.dspaddrCAPBUFF = ulCAPBUFFAddress;
    if (ulCAPBUFFAddress) {
      ulRC = mwmClss2WriteTaskData( psi, ulCAPBUFFAddress,
                                    &psi->TempClass2Capabilities, 15,
                                    szThisFile, __LINE__ );
    } /* endif */
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetCAPBUFFAddress exit\n");


  return ulRC;
}

ULONG MWM_ENTRY mwmParseSetMYIDAddress(ULONG ulMYIDAddress)
{
  ULONG ulRC = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetMYIDAddress entry\n");


  if (psi)
  {
    psi->dsp.dspaddrMYID = ulMYIDAddress;
    if (ulMYIDAddress) {
      ulRC = mwmClss2WriteTaskData( psi, ulMYIDAddress,
                                    psi->achFLIDString, 10,
                                    szThisFile, __LINE__ );
    } /* endif */
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetMYIDAddress exit\n");


  return ulRC;
}

ULONG MWM_ENTRY mwmParseSetMYCIGAddress(ULONG ulMYCIGAddress)
{
  ULONG ulRC = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetMYCIGAddress entry\n");


  if (psi)
  {
    psi->dsp.dspaddrMYCIG = ulMYCIGAddress;
    if (ulMYCIGAddress) {
      ulRC = mwmClss2WriteTaskData( psi, ulMYCIGAddress,
                                    psi->achFCIGString, 10,
                                    szThisFile, __LINE__ );
    } /* endif */
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetMYCIGAddress exit\n");


  return ulRC;
}

ULONG MWM_ENTRY mwmParseSetFAXERR(USHORT usValue)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetFAXERR entry\n");

  if (psi)
  {
    psi->usFAXERR = usValue;
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetFAXERR exit\n");

  return 0;
}


ULONG MWM_ENTRY mwmParseQueryHookStatus(USHORT *pusHandsetStatus, USHORT *pusHookStatus)
{
  ULONG ulRC;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryHookStatus entry\n");


  ulRC = dspMemTransfer(psi->dsp.hDSP, (psi->dsp.dspaddrHKSTAT)+2,
                        pusHandsetStatus, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrHKSTAT,
                        pusHookStatus, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryHookStatus exit\n");


  return 0;
}




/*****************************************************************************/
/* Export this function so that the power management does not have to        */
/* directly touch psi.                                                       */
/*****************************************************************************/
ULONG MWM_ENTRY mwmParseSaveModemRegisters( void )
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSaveModemRegisters entry\n");
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSaveModemRegisters exit\n");

  return (ULONG)mwmParseQueryModemRegisters(&psi->ModemMemory.aStoredRegs[3]);
}




ULONG MWM_ENTRY mwmParseQueryModemRegisters(REGISTERS *pRegisters)
{
  ULONG ulRC;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryModemRegisters entry\n");


  /***************************************************************************/
  /* Transfer all of the S Registers from MCTL to the REGISTERS structure    */
  /***************************************************************************/
  /* MTS6384 even though SV34[] is split from S[], it is read from to mdm control*/
  /*    as before.  This may subsequently be changed.*/
  ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrSREGS,
                        pRegisters->S,
                        MWM_NUM_S_REGISTERS + MWM_NUM_V34_REGISTERS,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }


  /***************************************************************************/
  /* Transfer the rest of the permanent registers.                           */
  /* 10/5/93 Do this in one shot.  Transfer all of the other registers with  */
  /* one dspMemTransfer.                                                     */
  /***************************************************************************/
  ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrOTHRREGS,
                        &pRegisters->SlashA, MWM_NUM_OTHER_REGISTERS,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

  /***************************************************************************
  ** S32 and S33 are maintained in the PC. Copy their current values into
  ** the REGISTERS structure.
  **
  ** MTS 6555 - Added 05/20/96
  ****************************************************************************/
  pRegisters->S32 = psi->usS32Value;
  pRegisters->S33 = psi->usS33Value;
  pRegisters->WCommand = psi->usWValue;
  pRegisters->S60 = psi->usS60Value;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryModemRegisters exit ulRC 0\n");


  return 0;
}


ULONG MWM_ENTRY mwmParseSetModemRegisters(REGISTERS *pRegisters)
{
  ULONG ulRC;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetModemRegisters entry\n");


  /***************************************************************************/
  /* Transfer all of the S Registers from REGISTERS structure to MCTL        */
  /***************************************************************************/
  /* MTS6384 even though SV34[] is split from S[], it is sent to modem control*/
  /*    as before.  This may subsequently be changed.*/
  ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrSREGS,
                        pRegisters->S,
                        MWM_NUM_S_REGISTERS + MWM_NUM_V34_REGISTERS,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }


  /***************************************************************************/
  /* Transfer the rest of the permanent registers.                           */
  /* 10/5/93 Do this in one shot.  Transfer all of the other registers with  */
  /* one dspMemTransfer.                                                     */
  /***************************************************************************/
  ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrOTHRREGS,
                        &pRegisters->SlashA, MWM_NUM_OTHER_REGISTERS,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetModemRegisters exit\n");


  return 0;
}


ULONG MWM_ENTRY mwmParseQueryBlacklistStatus(USHORT *pusBlackListStatus)
{
  ULONG ulRC;
  ULONG ulAddress;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryBlacklistStatus entry\n");


  ulRC = dspLabelToAddress(psi->dsp.hMCTL, "BLCKSTAT", &ulAddress);
  if (ulRC)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR,
                      ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* Transfer all of the S Registers from MCTL to the REGISTERS structure    */
  /***************************************************************************/
  ulRC = dspMemTransfer(psi->dsp.hDSP, ulAddress,
                        pusBlackListStatus, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }


 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryBlacklistStatus exit\n");


  return 0;
}




/*****************************************************************************/
/* Used by power management, and initial loading to know which mode the      */
/* parser is in.  Currently, there are 2 modes, FAX, and MODEM.              */
/* Since usParserMode is used in mwmFax etc. to return the correct PP cmd,   */
/* assigning new modes could be very dangerous.                              */
/*****************************************************************************/
ULONG MWM_ENTRY mwmParseQueryParserMode(PUSHORT pusParserMode)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryParserMode entry\n");


  *pusParserMode = psi->usParserMode;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQueryParserMode exit\n");

  return 0;
}



/*****************************************************************************/
/* This routine must be used with care.  It should only be called at         */
/* startup time AFTER a power management event.  Otherwise, it should be     */
/* set at the correct setting.                                               */
/*****************************************************************************/
ULONG MWM_ENTRY mwmParseSetParserMode(USHORT usParserMode)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetParserMode entry\n");


  psi->usParserMode = usParserMode;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetParserMode exit\n");

  return 0;
}


/*****************************************************************************/
/* Used to set the features loaded by this run of the parser.                */
/*****************************************************************************/
ULONG MWM_ENTRY mwmParseSetFeatures(ULONG ulInstalledFeatures, ULONG ulFeaturesToLoad)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetFeatures entry\n");

  psi->ulFeaturesToLoad    = ulFeaturesToLoad;
  psi->ulInstalledFeatures = ulInstalledFeatures;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetFeatures exit\n");

  return 0;
}



ULONG MWM_ENTRY mwmParseSetWTParms(PMWM_WTPARMS pwt)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetWTParms entry\n");

  if (psi)
  {
    memcpy(&psi->wt,pwt,sizeof(MWM_WTPARMS));
  }
  else
    return 1;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetWTParms exit\n");

  return 0;
}



ULONG mwmParseQuerySpeaker(PUSHORT pusSpeakerSetting, PUSHORT pusSpeakerOnOff)
{

  USHORT usCurrentValue = 0;
  ULONG  ulRC;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQuerySpeaker entry\n");


   ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrSREGS + 44,
                         &usCurrentValue, 1,
                         DSP_MEMXFER_DATA_READ);
   if (ulRC!=DSP_NOERROR)
   {
     mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                       MWM_DSP_ERROR, ulRC );
     return ulRC;
   }

  *pusSpeakerSetting = (usCurrentValue & 0x0003);
  *pusSpeakerOnOff   = (usCurrentValue & 0x000c)>>2;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseQuerySpeaker exit\n");


  return 0;
}



/*****************************************************************************/
/* mwmParseReadATStringFromDSP                                               */
/*****************************************************************************/
/* This function reads the AT Command String into the State Info structure   */
/* from MDMCTL's ATCMDBUF label.                                             */
/*****************************************************************************/
ULONG mwmParseReadATStringFromDSP( STATEINFO *psi )
{
   ULONG   ulRC;
   int     i;
   USHORT  ausCommandBuffer[AT_CMD_BUFFER_SIZE];
   USHORT  usCommandLength;


 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseReadATStringFromDSP entry\n");


  /***************************************************************************/
  /* Initialize the newly defined array to all 0's.                          */
  /***************************************************************************/
  memset(ausCommandBuffer,0,sizeof(USHORT)*AT_CMD_BUFFER_SIZE);


  /***************************************************************************/
  /* read command buffer into the array ausCommandBuffer                     */
  /*                                                                         */
  /* The AT Command is initially read into an array of USHORT.               */
  /* This routine reads it, then converts it to an ascii string .            */
  /***************************************************************************/

  ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrATCMDBUF,
                        ausCommandBuffer, AT_CMD_BUFFER_SIZE,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC != DSP_NOERROR)
  {
     mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                       MWM_DSP_ERROR, ulRC );
     return ulRC;
  }

  /***************************************************************************/
  /* Clear AT Command Buffer...                                              */
  /***************************************************************************/
  memset(psi->achCommandBuffer,0,AT_CMD_BUFFER_SIZE);

  /***************************************************************************/
  /* Set Next Index to point to beginning of AT Command Buffer.              */
  /***************************************************************************/
  psi->usNextATIndex = 0;


  /***************************************************************************/
  /* The command length is stored in the first word (USHORT) of the incoming */
  /* ausCommandBuffer.                                                       */
  /* However, it is stored in a funny format...get the number of characters. */
  /***************************************************************************/
  usCommandLength = ausCommandBuffer[0]/2 + 1;

  /***************************************************************************/
  /* Convert and copy string from aus to ach                                 */
  /***************************************************************************/
  for (i=1;
       ((i <= usCommandLength) && (i < AT_CMD_BUFFER_SIZE));
       i++)
  {
    /* paulsch: added touper here instead of strupr below */
    psi->achCommandBuffer[i-1] = toupper((char)(ausCommandBuffer[i]));
  } /* endfor */

  /***************************************************************************/
  /* NULL Terminate the new string.                                          */
  /***************************************************************************/
  psi->achCommandBuffer[i]='\0';

  /***************************************************************************/
  /* Convert the string to uppercase                                         */
  /***************************************************************************/
  /* paulsch: use toupper above
  strupr(psi->achCommandBuffer);
  */

 MW_SYSLOG_2(TRACE_MWMPW32,"mwmparse::mwmParseReadATStringFromDSP exit ATcmd %s\n",
	psi->achCommandBuffer);


  return 0;
}




  ULONG MWM_ENTRY mwmParseSetDRVRSTAT(USHORT usValue)
{
  ULONG ulRC;

 MW_SYSLOG_2(TRACE_MWMPW32,"mwmparse::mwmParseSetDRVRSTAT entry usValue %x\n",usValue);


  ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrDRVRSTAT,
                        &usValue,
                        1,
                        DSP_MEMXFER_DATA_WRITE);

  if (ulRC != DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return ulRC;
  }

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetDRVRSTAT exit\n");

  return 0;
}


ULONG MWM_ENTRY mwmParseSetQuoteN(USHORT usQuoteNValue)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetQuoteN entry\n");

  mwmQuoteNCommand(psi, usQuoteNValue );

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetQuoteN exit\n");

  return 0;
}

USHORT MWM_ENTRY mwmParseGetWValue()
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseGetWValue entry\n");
 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseGetWValue exit\n");

  return(psi->usWValue);
}




ULONG MWM_ENTRY mwmParseSetSpeaker(USHORT usAndMask, USHORT usOrMask)
{
  USHORT usCurrentValue = 0;
  ULONG  ulRC;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetSpeaker entry\n");

   ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrSREGS + 44,
                         &usCurrentValue, 1,
                         DSP_MEMXFER_DATA_READ);
   if (ulRC!=DSP_NOERROR)
   {
     mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                       MWM_DSP_ERROR, ulRC );
     return ulRC;
   }

   usCurrentValue = ((usCurrentValue & usAndMask) | usOrMask); /* set the appropriate bits*/

   ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrSREGS + 44,
                         &usCurrentValue, 1,
                         DSP_MEMXFER_DATA_WRITE);
   if (ulRC!=DSP_NOERROR)
   {
     mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                       MWM_DSP_ERROR, ulRC );
     return ulRC;
   }

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetSpeaker exit\n");


  return 0;
}


void mwmParseSetSRegister(USHORT *pusPPBuffer, USHORT *pusPPIndex,
                          USHORT usSReg, USHORT usAndMask, USHORT usOrMask)
{
  USHORT     usDesiredSetting = 0;
  REGISTERS  Registers;

 MW_SYSLOG_4(TRACE_MWMPW32,"mwmparse::mwmParseSetSRegister entry usSReg %x usAndMask %x usOrMask %x\n",usSReg,usAndMask,usOrMask);

/*  char       achDebugString[80] = "\0";*/

  /***************************************************************************/
  /* MTS 5080                                                                */
  /* 05/10/95                                                                */
  /* Certain S Registers have World Trade constraints.                       */
  /* Until now, these S Register settings have simply been ignored when      */
  /* the actions actually occur.  This has been quite confusing to           */
  /* users who see one value in the s register, then the modem function in   */
  /* another way.                                                            */
  /* Now, I will "bound" the S Registers to acceptable settings.  I will not */
  /* return error, instead, I will set the S Register to the best possible   */
  /* setting based on the users request.                                     */
  /*                                                                         */
  /* MTS 5590                                                                */
  /* 02/08/96                                                                */
  /* Bounds checking needs to be done for other S registers as well.         */
  /***************************************************************************/


  /***************************************************************************/
  /* 11/29/95                                                                */
  /* Move S Register setting totally to the host.                            */
  /* Logic is:                                                               */
  /* 1.  Read the current S Reg                                              */
  /* 2.  Apply the AND/OR mask to get "desired setting"                      */
  /* 3.  Bound the register if necessary.                                    */
  /* 4.  Apply the New AND/OR mask                                           */
  /* 5.  Write the register value back to the DSP                            */
  /***************************************************************************/
  mwmParseQueryModemRegisters(&Registers);
  usDesiredSetting = usAndMask & Registers.S[usSReg];
  MW_SYSLOG_2(TRACE_MWMPW32,"mwmparse::mwmParseSetSRegister entry usDesiredSetting %x\n",usDesiredSetting);

  usDesiredSetting = usDesiredSetting | usOrMask;


  switch (usSReg)
  {
    case 0:

      /***********************************************************************/
      /* S0 is being set, so make sure MCTL sends us an IPC 19               */
      /***********************************************************************/
      pusPPBuffer[(*pusPPIndex)++] = (0x24 | psi->usParserMode);
      pusPPBuffer[(*pusPPIndex)++] = 19;

/*      if (psi->wt.ulWTTAF1 & 0x0080)*/
/*      {*/
/*        sprintf(achDebugString,"AND/OR Masks: 0x%04x/0x%04x  DesiredSetting: 0x%04x\n",*/
/*                                usAndMask,usOrMask,usDesiredSetting);*/
/*        if (psi->usDebugWindow)*/
/*          OutputDebugString(achDebugString);*/
/*        sprintf(achDebugString,"ulWTTRNGS: 0x%08x\n",*/
/*                                psi->wt.ulWTTRNGS);*/
/*        if (psi->usDebugWindow)*/
/*          OutputDebugString(achDebugString);*/
        /*********************************************************************/
        /* S0 must be bound between the 2 words of WTTRNGS                   */
        /*  Lo(WTTRNGS) <= S0 <= Hi(WTTRNGS)                                 */
        /*********************************************************************/
/*        sprintf(achDebugString,"ulWTTRNGS (upper): 0x%04x\n",*/
/*                                (psi->wt.ulWTTRNGS & 0xff00)>>8);*/
/*        if (psi->usDebugWindow)*/
/*          OutputDebugString(achDebugString);*/
        /*********************************************************************/
        /* S0=0 is allowed, so we only do this bounding stuff if we are      */
        /* trying to set s0>0                                                */
        /*********************************************************************/
        if (usDesiredSetting)
        {
          if (usDesiredSetting > ((psi->wt.ulWTTRNGS & 0xff00) >> 8))
          {
            usAndMask = (USHORT)((psi->wt.ulWTTRNGS & 0xff00)>> 8);
            usOrMask  = usAndMask;
          }
          if (usDesiredSetting < (psi->wt.ulWTTRNGS & 0x00ff) )
          {
            usAndMask = (USHORT)(psi->wt.ulWTTRNGS & 0x00ff);
            usOrMask  = usAndMask;
          }

        }

/*        sprintf(achDebugString,"AND/OR Masks: 0x%04x/0x%04x  DesiredSetting: 0x%04x\n",*/
/*                                usAndMask,usOrMask,usDesiredSetting);*/
/*        if (psi->usDebugWindow)*/
/*          OutputDebugString(achDebugString);*/
/*      }*/
      break;

    case 3:
      if (usDesiredSetting > 127) {
        usAndMask = 0xFF;
        usOrMask = 0;
      } /* endif */
    break;

    case 4:
      if (usDesiredSetting > 127) {
        usAndMask = 0xFF;
        usOrMask = 0;
      } /* endif */
    break;

    case 5:
      if (usDesiredSetting > 32) {
        usAndMask = 0xFF;
        usOrMask = 0;
      } /* endif */
    break;

    case 6:
/*      if (psi->wt.ulWTTAF1 & 0x0040)*/
/*      {*/
/*        mwmParseQueryModemRegisters(&Registers);*/
/*        usDesiredSetting = usAndMask & Registers.S[6];*/
/*        usDesiredSetting = usDesiredSetting | usOrMask;*/

        /*********************************************************************/
        /* MTS 5203 WTTBDWT will be specified in msec.                       */
        /*********************************************************************/
        if (usDesiredSetting < ((psi->wt.ulWTTBDWT)/1000))
        {
          usAndMask = (USHORT)((psi->wt.ulWTTBDWT)/1000);
          usOrMask  = usAndMask;
        }
        /*********************************************************************/
        /* MTS 5203 WTTBDWT will be specified in msec.                       */
        /*********************************************************************/
        if (usDesiredSetting > ((psi->wt.ulWTTBDWTX)/1000))
        {
          usAndMask = (USHORT)((psi->wt.ulWTTBDWTX)/1000);
	  usOrMask  = usAndMask;
        }
/*      }*/
      break;
    case 7:
/*      if (psi->wt.ulWTTAF1 & 0x0020)*/
/*      {*/
/*        mwmParseQueryModemRegisters(&Registers);*/
/*        usDesiredSetting = usAndMask & Registers.S[7];*/
/*        usDesiredSetting = usDesiredSetting | usOrMask;*/

        if (usDesiredSetting < psi->wt.ulWTTCW)
        {
          usAndMask = (USHORT)(psi->wt.ulWTTCW);
          usOrMask  = usAndMask;
        }
        if (usDesiredSetting > psi->wt.ulWTTCWX)
        {
          usAndMask = (USHORT)(psi->wt.ulWTTCWX);
          usOrMask  = usAndMask;
        }
/*      }*/
      break;

    case 8:
      if (usDesiredSetting < 1) {
        usAndMask = 1;
        usOrMask = usAndMask;
      } /* endif */
    break;

    case 9:
      if (usDesiredSetting < 1) {
        usAndMask = 1;
        usOrMask = usAndMask;
      } /* endif */
    break;

    case 10:
/*      if (psi->wt.ulWTTAF1 & 0x0010)*/
/*      {*/
/*        mwmParseQueryModemRegisters(&Registers);*/
/*        usDesiredSetting = usAndMask & Registers.S[10];*/
/*        usDesiredSetting = usDesiredSetting | usOrMask;*/

        if (usDesiredSetting < psi->wt.ulWTTLCW)
        {
          usAndMask = (USHORT)(psi->wt.ulWTTLCW);
          usOrMask  = usAndMask;
        }
        if (usDesiredSetting > psi->wt.ulWTTLCWX)
        {
          usAndMask = (USHORT)(psi->wt.ulWTTLCWX);
          usOrMask  = usAndMask;
        }
/*      }*/
      break;
    case 11:
      if (psi->wt.ulWTTAF1 & 0x0008)
      {
        /*********************************************************************/
        /* S11 update not allowed...so replace desired s11 setting with      */
        /* WTTDTSD                                                           */
        /*********************************************************************/
        usAndMask = (USHORT)(psi->wt.ulWTTDTSD);
        usOrMask  = usAndMask;
      }
      break;
    case 14:
/*      mwmParseQueryModemRegisters(&Registers);*/
/*      usDesiredSetting = usAndMask & Registers.S[14];*/
/*      usDesiredSetting = usDesiredSetting | usOrMask;*/
      if (usDesiredSetting & 0x20)
      {
        if (!psi->wt.ulWTTDMODE)
        {
          /*******************************************************************/
          /* Pulse is not allowed...reset S14 bit 5 to tone (0)              */
          /*******************************************************************/
          usAndMask &= 0xffdf;
          usOrMask  &= 0xffdf;
        }
      }
      else
      {
        if (psi->wt.ulWTTDTADS == 0xFFFF)
        {
          /*******************************************************************/
          /* Tone Not allowed...set S14 bit 5 to pulse (1)                   */
          /*******************************************************************/
          usAndMask  |= 0x20; /* turn on bit 5*/
          usOrMask   |= 0x20; /* turn on bit 5*/
        }
      }
      break;
    case 28:
      /***********************************************************************/
      /* S28 will also change the S37 value.  We will do that here.          */
      /***********************************************************************/
/*      mwmParseQueryModemRegisters(&Registers);*/
/*      usDesiredSetting = usAndMask & Registers.S[28];*/
/*      usDesiredSetting = usDesiredSetting | usOrMask;*/
      switch (usDesiredSetting)
      {
        case 6:
          psi->usS37Value = 3;
          break;
        case 8:
          psi->usS37Value = 5;
          break;
        case 9:
          psi->usS37Value = 6;
          break;
        case 10:
          psi->usS37Value = 8;
          break;
        case 11:
          psi->usS37Value = 12;
          break;
        case 12:
          psi->usS37Value = 9;
          break;
        case 13:
          psi->usS37Value = 10;
          break;
        case 14:
          psi->usS37Value = 11;
          break;
        default:
          psi->usS37Value = 0;
          break;
      }

      break;

  }


/*   usSReg=2*usSReg;   */       /* Sreg numbers must be doubled */


  /***************************************************************************/
  /* 11/29/95                                                                */
  /* Now we know that the AND/OR Masks are "bounded" so that the register    */
  /* settings will be world trade correct.                                   */
  /* Apply the masks again, and set the Register to its new value.           */
  /* Then, write the register set back to the DSP                            */
  /***************************************************************************/
  usDesiredSetting = usAndMask & Registers.S[usSReg];
  usDesiredSetting = usDesiredSetting | usOrMask;
  Registers.S[usSReg] = usDesiredSetting;

  mwmParseSetModemRegisters(&Registers);


  /***************************************************************************/
  /* Put the PP commands into the PP command array....incrementing the       */
  /* index after each command.                                               */
  /***************************************************************************/
/*  pusPPBuffer[(*pusPPIndex)++] = (5 | psi->usParserMode);*/
/*  pusPPBuffer[(*pusPPIndex)++] = usSReg;*/
/*  pusPPBuffer[(*pusPPIndex)++] = usAndMask;*/
/*  pusPPBuffer[(*pusPPIndex)++] = usOrMask;*/

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetSRegister exit\n");

}



/*****************************************************************************/
/* DCR 2456 Power Management.                                                */
/* The purpose of this routine is to isolate psi from the calling EXE.       */
/* Since the Client Proc of the Power Management stuff is in the .EXE, this  */
/* routine does the work which requires access to the Parser's global memory */
/*****************************************************************************/
ULONG MWM_ENTRY mwmParseSetSuspendFlag(USHORT usSuspendFlag)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetSuspendFlag entry\n");

  psi->usSuspendFlag = usSuspendFlag;
  /***************************************************************************/
  /* If the Suspend flag is being set, save the AT Command registers.        */
  /***************************************************************************/
  if (usSuspendFlag)
  {
    /*************************************************************************/
    /* Save the AT Command settings.                                         */
    /*************************************************************************/
    mwmParseQueryModemRegisters(&psi->ModemMemory.aStoredRegs[3]);
  }

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetSuspendFlag exit\n");

  return 0;
}

/*****************************************************************************
* This flag is set when a country change has occured during an ATDTx, ATH1,
* or ATA.  It allows the country to be changed "under the covers" without the
* user knowing during the dial.  PCR 44.
*****************************************************************************/
ULONG MWM_ENTRY mwmParseSetWTChangeFlag(USHORT usWTChangeFlag)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetWTChangeFlag entry\n");

  DPF("Setting psi->usWTChangeFlag = %d", usWTChangeFlag);
  psi->usWTChangeFlag = usWTChangeFlag;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetWTChangeFlag exit\n");

  return 0;
}

/*****************************************************************************
* Return the current value of the usWTChangeFlag
*****************************************************************************/
ULONG MWM_ENTRY mwmParseWTChangeFlagValue()
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseWTChangeFlagValue entry\n");

  DPF("mwmParseWTChangeFlagValue - usWTChangeFlag = %d", psi->usWTChangeFlag);

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseWTChangeFlagValue exit\n");

  return psi->usWTChangeFlag;
}


/*****************************************************************************/
/* mwmParseWriteCmdToDSP                                                     */
/*****************************************************************************/
/* Write the "End of Command" string with either good or error return codes  */
/* to the PP command string, then dspMemTransfer it to MDMCTL.               */
/*****************************************************************************/
void   mwmParseWriteCmdToDSP(STATEINFO *psi,USHORT usParserStatus)
{
  ULONG  ulRC;

  /* Debug */
  #ifdef DEBUG
  USHORT usCount;
  static  char   pszDebugString[128] = "\0";
  char   pszConvertString[10] = "\0";
  #endif
  /* End Debug */


 MW_SYSLOG_2(TRACE_MWMPW32,"mwmparse::mwmParseWriteCmdToDSP entry usParserStatus %x\n",usParserStatus);


  if (usParserStatus & MWM_GET_MORE_BUFFERS)
  {

    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x0c | psi->usParserMode);
  }
  else
  {
    if (usParserStatus & MWM_ATCMD_ERROR)
    {
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (1 | psi->usParserMode);
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 4;
    }
    else
    {
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (1 | psi->usParserMode);
      psi->ausPPcmdBuffer[psi->usNextPPIndex++] = 0;
    } /* endif */

    /*************************************************************************/
    /* Clear the AT Command buffer, and reset AT index.                      */
    /*************************************************************************/
    memset(psi->achCommandBuffer,0,AT_CMD_BUFFER_SIZE);
    psi->usNextATIndex = 0;
  } /* endif */


#ifdef DEBUG
  /***************************************************************************/
  /* Debug....write the PP command ...                                       */
  /***************************************************************************/
  strncpy(pszDebugString,"PP Cmd: ",128);
  for (usCount = 0;usCount<psi->usNextPPIndex;usCount++)
  {
    sprintf(pszConvertString,"%x,",psi->ausPPcmdBuffer[usCount]);
    strcat(pszDebugString,pszConvertString);
    if (strlen(pszDebugString)>80)
    {
/*      strcat(pszDebugString,"\nMWMODEM: ");*/
/*      if (psi->usDebugWindow)*/
/*      {*/
  	MW_SYSLOG_2(TRACE_MWMPW32,"mwmparse::mwmParseWriteCmdToDsp, ausPPcmdBuffer %s\n",pszDebugString);
        DPF(pszDebugString);
/*      }*/
      *pszDebugString = '\0';
    }
  }

  if (strlen(pszDebugString))
  {
    strcat(pszDebugString,"\n");
/*    if (psi->usDebugWindow)*/
/*    {*/
      MW_SYSLOG_2(TRACE_MWMPW32,"mwmparse::mwmParseWriteCmdToDsp, ausPPcmdBuffer %s\n",pszDebugString);
      DPF(pszDebugString);
/*    }*/
  }
#endif /* DEBUG */


  /***************************************************************************/
  /* Write the PP command buffer to MDMCTL's PPCMDBUF public label.          */
  /***************************************************************************/
  ulRC = dspMemTransfer(psi->dsp.hDSP, psi->dsp.dspaddrPP_CMD,
                            psi->ausPPcmdBuffer,
                            psi->usNextPPIndex,
                            DSP_MEMXFER_DATA_WRITE);

  if (ulRC != DSP_NOERROR)
  {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_DSP_ERROR, ulRC );
    return;
  }

  /***************************************************************************/
  /* Tell MDMCTL that we are finished by writing ffff (-1) to DRVRSTAT.      */
  /***************************************************************************/
  mwmParseSetDRVRSTAT(0xffff);

  /***************************************************************************/
  /* Now, clear out the PP_CMD buffer, and reset the PPIndex counter.        */
  /***************************************************************************/
  memset(psi->ausPPcmdBuffer,0,PP_CMD_BUFFER_SIZE);
  psi->usNextPPIndex = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseWriteCmdToDSP exit\n");


}

/* MTS6384 - new procedure derived from code in mwmSlash.c*/
/*****************************************************************************/
/* mwmParseSetDisconnectTime                                                      */
/*****************************************************************************/
/* Called by...                                                              */
/*   mwmSlashCommand when...                                                 */
/*   Slash T command is encountered in the AT command string.                */
/*   mwmCommandHostEqualCommand when S30= is encountered in AT command       */
/* The check for buffer full should take place before calling this routine.  */
/*****************************************************************************/
void  mwmParseSetDisconnectTime(STATEINFO *psi, USHORT usDisconnectSeconds)
{
  ULONG  ulModemDisconnectTime = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetDisconnectTime entry\n");


  /*************************************************************************/
  /* add the PP command for Slash T to the PP command string.              */
  /*************************************************************************/
  psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (0x1f | psi->usParserMode);

  if (!psi->wt.ulWTTAMD)
  {
    /***********************************************************************/
    /* MTS 3444                                                            */
    /* Allow the command as is since the WTTAMD parameter is 0             */
    /***********************************************************************/
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = usDisconnectSeconds;
  }
  else
  {
    /***********************************************************************/
    /* MTS 3444                                                            */
    /* Since WTTAMD is set, then the AutoDisconnect Timer is restricted to */
    /* this value.                                                         */
    /***********************************************************************/
    psi->ausPPcmdBuffer[psi->usNextPPIndex++] = (USHORT)ulModemDisconnectTime;
  }

  /*************************************************************************/
  /* Add the parameter to the end of the PP command string.                */
  /*************************************************************************/
  psi->ausPPcmdBuffer[psi->usNextPPIndex++] = usDisconnectSeconds;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetDisconnectTime exit\n");

}

/*****************************************************************************/
/* mwmParseGetNextArgument                                                   */
/*****************************************************************************/
/* Gets the next arguments from a string of arguments separated by commas.   */
/*****************************************************************************/
char *mwmParseGetNextArgument( char *pArgumentList, USHORT *pusParmNumber )
{

  static char *achCopyString=NULL;
  char *pStartChar;
  char *pCommaFinder;
  USHORT usCommaCount = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseGetNextArgument entry\n");


  if (achCopyString)
	  	free(achCopyString);
  achCopyString = strdup (pArgumentList);
  pStartChar = achCopyString;
  pCommaFinder = pStartChar;

  do {

    /*************************************************************
    ** Scan from the current start location 'til we find a cursor
    ** or until we reached the end of the string
    **************************************************************/
    while ( *pCommaFinder != ',' && *pCommaFinder != '\0' ) {
      pCommaFinder++;
    } /* endwhile*/

    /*************************************************************
    ** We found a comma, increment the comma counter
    **************************************************************/
    usCommaCount++;

    /***************************************************************
    ** Check to see if we've reached the desired argument position.
    ** or until we reached the end of the string
    ****************************************************************/
    if (usCommaCount == *pusParmNumber+1) {

      /*************************************************
      ** We have found the desired parameter, is there
      ** a comma there? If not, return a pointer to it.
      **************************************************/
      if (*pStartChar != ',') {
        *pCommaFinder = '\0';
        return( pStartChar );

      /*************************************************
      ** There is a comma in the desired parameter. Skip
      ** it and find the next valid parameter.
      **************************************************/
      } else {
        (*pusParmNumber)++;
        pStartChar++;
        pCommaFinder = pStartChar + 1;
      } /* endif */

    /***************************************************************
    ** We're not at the desired argument position. If we haven't
    ** reached the end of the string of arguments, move the start
    ** pointer to the next argument to repeat the process.
    ****************************************************************/
    } else {
      if (*pCommaFinder != '\0') {
        pStartChar = pCommaFinder+1;
      } else {
        pStartChar = pCommaFinder;
      } /* endif */
      pCommaFinder = pStartChar;
    } /* endif */

  } while (*pStartChar != '\0');

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseGetNextArgument exit\n");


  return(NULL);

}

/***************************************************************
** Function: mwmParseMoveATIndex
**
** Moves the index into the AT command buffer past the
** parameters and to the start of the next command
**
** Input: String containing a list of parameters from an
**        AT command
**
****************************************************************/
void mwmParseMoveATIndex( STATEINFO *psi, char *achString)
{

  int i = 0;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseMoveATIndex entry\n");


  do {
    if (achString[i] != '\0' && achString[i] != ';') {
      i++;
      psi->usNextATIndex++;
    } /* endif */
  } while ( achString[i] != '\0' && achString[i] != ';' ); /* enddo */
  if (achString[i] == ';') {
    psi->usNextATIndex++;
  } /* endif */

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseMoveATIndex exit\n");


}

/*****************************************************************************
** Function: mwmParseActivateDebugWindow
**
** Purpose: Called by the loader when the user starts the debug output
**          window. Turns on the flag indicating that there is a debug window
**          present.
******************************************************************************/
void MWM_ENTRY mwmParseActivateDebugWindow( BOOL bActivateFlag )
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseActivateDebugWindow entry\n");

    psi->usDebugWindow = bActivateFlag;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseActivateDebugWindow exit\n");

}

/*****************************************************************************
** Function: mwmParseActivateTimeStamp
**
** Purpose: Called by the loader when the user checks the Show Timestamp
**          option under settings in the debug window.
******************************************************************************/
void MWM_ENTRY mwmParseActivateTimeStamp( BOOL bTimeStampFlag )
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseActivateTimeStamp entry\n");

    if (psi)
      psi->usShowTimeStamp = bTimeStampFlag;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseActivateTimeStamp exit\n");

}

/*****************************************************************************
** Function: mwmParseReadMemoryProfile
**
** Purpose: Reads the memory profile file and converts it to the new
**          format if needed.
******************************************************************************/
ULONG mwmParseReadMemoryProfile( STATEINFO *psi )
{
  FILE   *fMemoryFile;
  MODEMMEMORY tempModemMemory;
  char achVersion[10];
  char achCurrentVersion[10];
  char *pMemory = NULL;
  char *pTempModemMemory = NULL;
  USHORT usSize=0;
  USHORT usStart, usEnd, usBytesToCopy;
  int i;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseReadMemoryProfile entry\n");


  /*****************************************************************
  ** Open the current memory profile file.
  ******************************************************************/
  fMemoryFile = fopen(psi->achMemoryFileName, "rb");

  if (!fMemoryFile) {
    mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                      MWM_OS_ERROR, 0 );
    return (1);
  } /* endif */


  /*****************************************************************
  ** Read the Version, Major and Minor numbers.
  ******************************************************************/
  if (!fread(&tempModemMemory, 3*sizeof(USHORT), 1, fMemoryFile) ) {
    if ( ferror(fMemoryFile) ) {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_OS_ERROR, 0 );
      return (1);
    } /* endif */
  } /* endif */

  /*****************************************************************
  ** Compare the file version with the current version.
  ******************************************************************/
  if ( tempModemMemory.usVersion      == CURRENT_PRF_VERSION &&
       tempModemMemory.usMajorRelease == CURRENT_PRF_MAJOR_RELEASE &&
       tempModemMemory.usMinorRelease == CURRENT_PRF_MINOR_RELEASE) {

 MW_SYSLOG_4(TRACE_MWMPW32,"mwmparse::mwmParseReadMemoryProfile tempModemMemory.usVersion=%hu tempModemMemory.usMajorRelease=%hu tempModemMemory.usMinorRelease=%hu\n", tempModemMemory.usVersion, tempModemMemory.usMajorRelease, tempModemMemory.usMinorRelease);

    /*****************************************************************
    ** They match. Read the rest of the file into memory.
    ******************************************************************/
    if (fseek( fMemoryFile, 0, SEEK_SET )) {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_OS_ERROR, 0 );
      return (1);
    } /* endif */

    if (!fread(&(psi->ModemMemory), sizeof(MODEMMEMORY), 1, fMemoryFile) ) {
      if ( ferror(fMemoryFile) ) {
        mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                          MWM_OS_ERROR, 0 );
        return (1);
      } /* endif */
    } /* endif */
    if ( fclose(fMemoryFile) ) {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_OS_ERROR,
                        0 );
      return (1);
    } /* endif */

  } else {

    /*****************************************************************
    ** They don't match. Convert the file into its new format.
    ******************************************************************/

    sprintf( achCurrentVersion, "%u.%u%u", CURRENT_PRF_VERSION,
                                           CURRENT_PRF_MAJOR_RELEASE,
                                           CURRENT_PRF_MINOR_RELEASE );

    sprintf( achVersion, "%u.%u%u", tempModemMemory.usVersion,
                                    tempModemMemory.usMajorRelease,
                                    tempModemMemory.usMinorRelease );

    /* Get the size of the old structure and allocate a buffer for it */
    if (!strcmp( achVersion, "3.03" )) {
      usSize = sizeof(MODEMMEMORY_V303);
    } else if (!strcmp( achVersion, "3.04" )) {
      usSize = sizeof(MODEMMEMORY_V304);
    } else if (!strcmp( achVersion, "4.00")) {
      usSize = sizeof(MODEMMEMORY_V400);
    } else if (!strcmp( achVersion, "4.01")) {
      usSize = sizeof(MODEMMEMORY);
    } else if (!strcmp( achVersion, "4.02")) {
      usSize = sizeof(MODEMMEMORY);
    } else {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_DEBUG_MESSAGE,
                        (ULONG)"Unrecognized Modem Profile Version");
    } /* endif */

    pMemory = calloc( usSize, 1 );

    /* Read the old structure into the buffer */
    if (fseek( fMemoryFile, 0, SEEK_SET )) {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_OS_ERROR, 0 );
      return (1);
    } /* endif */

    if (!fread(pMemory, usSize, 1, fMemoryFile) ) {
      if ( ferror(fMemoryFile) ) {
        mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                          MWM_OS_ERROR, 0 );
        return (1);
      } /* endif */
    } /* endif */

    if ( fclose(fMemoryFile) ) {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_OS_ERROR,
                        0 );
      return (1);
    } /* endif */

    do {

      /*************************************************
      ** Convert Version 3.03 to Version 3.04
      **************************************************/
      if (!strcmp(achVersion, "3.03")) {

        DPF("Updating modem profile from Version 3.03 to 3.04");

        pTempModemMemory = (char *)calloc( sizeof(MODEMMEMORY_V304), 1 );

        /* Structure sizes are the same, just copy the old to the new */
        memcpy( pTempModemMemory, pMemory, sizeof(MODEMMEMORY_V304));

        /* Now set the default S32 and S33 registers */
        for (i=0; i<=2; i++) {
          ((MODEMMEMORY_V304 *)pTempModemMemory)->aStoredRegs[i].S32 = DEFAULT_S32_VALUE;
          ((MODEMMEMORY_V304 *)pTempModemMemory)->aStoredRegs[i].S33 = DEFAULT_S33_VALUE;
        } /* endfor */

        /* Set the new Version number */
        ((MODEMMEMORY_V304 *)pTempModemMemory)->usVersion      = 3;
        ((MODEMMEMORY_V304 *)pTempModemMemory)->usMajorRelease = 0;
        ((MODEMMEMORY_V304 *)pTempModemMemory)->usMinorRelease = 4;

      /*************************************************
      ** Convert Version 3.04 to Version 4.00
      **************************************************/
      } else if (!strcmp(achVersion, "3.04")) {

        DPF("Updating modem profile from Version 3.04 to 4.00");

        pTempModemMemory = (char *)calloc( sizeof(MODEMMEMORY_V400), 1 );

        /* Registers thru PoundCID are the same, so we can block copy those */
        usEnd   = (USHORT)&(((MODEMMEMORY_V304 *)pMemory)->aStoredRegs[0].PoundCID);
        usStart = (USHORT)&(((MODEMMEMORY_V304 *)pMemory)->aStoredRegs[0]);
        usBytesToCopy = usEnd - usStart;

        for (i=0; i<=2; i++) {
          memcpy( &(((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i]),
                  &(((MODEMMEMORY_V304 *)pMemory)->aStoredRegs[i]),
                  usBytesToCopy);
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].S32 = ((MODEMMEMORY_V304 *)pMemory)->aStoredRegs[i].S32;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].S33 = ((MODEMMEMORY_V304 *)pMemory)->aStoredRegs[i].S33;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusA8E[0] = DEFAULT_PLUSA8E_0_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusA8E[1] = DEFAULT_PLUSA8E_1_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusA8E[2] = DEFAULT_PLUSA8E_2_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusES[0]  = DEFAULT_PLUSES_0_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusES[1]  = DEFAULT_PLUSES_1_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusES[2]  = DEFAULT_PLUSES_2_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusESA[0] = DEFAULT_PLUSESA_0_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusESA[1] = DEFAULT_PLUSESA_1_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusESA[2] = DEFAULT_PLUSESA_2_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusESA[3] = DEFAULT_PLUSESA_3_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusESA[4] = DEFAULT_PLUSESA_4_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusESA[5] = DEFAULT_PLUSESA_5_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusESA[6] = DEFAULT_PLUSESA_6_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusESA[7] = DEFAULT_PLUSESA_7_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusITF[0] = DEFAULT_PLUSITF_0_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusITF[1] = DEFAULT_PLUSITF_1_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusITF[2] = DEFAULT_PLUSITF_2_VALUE;
          ((MODEMMEMORY_V400 *)pTempModemMemory)->aStoredRegs[i].PlusIBC[0] = DEFAULT_PLUSIBC_0_VALUE;
        } /* endfor */

        /* Copy the stored phone numbers over */
        for (i=0; i<=3; i++) {
          strcpy( ((MODEMMEMORY_V400 *)pTempModemMemory)->aachStoredNumbers[i],
                  ((MODEMMEMORY_V304 *)pMemory)->aachStoredNumbers[i] );
        } /* endfor */

        /* Copy the startup profile number over */
        ((MODEMMEMORY_V400 *)pTempModemMemory)->usStartupProfile =
                                ((MODEMMEMORY_V304 *)pMemory)->usStartupProfile;

        /* Set the new Version number */
        ((MODEMMEMORY_V400 *)pTempModemMemory)->usVersion = 4;
        ((MODEMMEMORY_V400 *)pTempModemMemory)->usMajorRelease = 0;
        ((MODEMMEMORY_V400 *)pTempModemMemory)->usMinorRelease = 0;

      /*************************************************
      ** Convert Version 4.00 to Version 4.01
      **************************************************/
      } else if (!strcmp(achVersion, "4.00")) {

        DPF("Updating modem profile from Version 4.00 to 4.01");

        pTempModemMemory = (char *)calloc( sizeof(MODEMMEMORY), 1 );

        memcpy( pTempModemMemory, pMemory, sizeof(MODEMMEMORY));

        for (i=0; i<=2; i++) {
          ((MODEMMEMORY *)pTempModemMemory)->aStoredRegs[i].WCommand = DEFAULT_WCOMMAND_VALUE;
          ((MODEMMEMORY *)pTempModemMemory)->aStoredRegs[i].S[7] = DEFAULT_S7_VALUE;
        } /* endfor */

        /* Copy the stored phone numbers over */
        for (i=0; i<=3; i++) {
          strcpy( ((MODEMMEMORY *)pTempModemMemory)->aachStoredNumbers[i],
                  ((MODEMMEMORY_V400 *)pMemory)->aachStoredNumbers[i] );
        } /* endfor */

        /* Force the startup profile to 2 (Factory Default) */
        ((MODEMMEMORY *)pTempModemMemory)->usStartupProfile = 2;

        /* Set the new Version number */
        ((MODEMMEMORY *)pTempModemMemory)->usVersion = 4;
        ((MODEMMEMORY *)pTempModemMemory)->usMajorRelease = 0;
        ((MODEMMEMORY *)pTempModemMemory)->usMinorRelease = 1;

      /*************************************************
      ** Convert Version 4.01 to Version 4.02
      **************************************************/
      } else if (!strcmp(achVersion, "4.01")) {

        DPF("Updating modem profile from Version 4.01 to 4.02");

        pTempModemMemory = (char *)calloc( sizeof(MODEMMEMORY), 1 );

        memcpy( pTempModemMemory, pMemory, sizeof(MODEMMEMORY));

        for (i=0; i<=2; i++) {
          ((MODEMMEMORY *)pTempModemMemory)->aStoredRegs[i].WCommand = DEFAULT_WCOMMAND_VALUE;
        } /* endfor */

        /* Set the new Version number */
        ((MODEMMEMORY *)pTempModemMemory)->usVersion = 4;
        ((MODEMMEMORY *)pTempModemMemory)->usMajorRelease = 0;
        ((MODEMMEMORY *)pTempModemMemory)->usMinorRelease = 2;

      /*************************************************
      ** Convert Version 4.02 to Version 4.03
      **************************************************/
      } else if (!strcmp(achVersion, "4.02")) {

        DPF("Updating modem profile from Version 4.02 to 4.03");

        pTempModemMemory = (char *)calloc( sizeof(MODEMMEMORY), 1 );

        memcpy( pTempModemMemory, pMemory, sizeof(MODEMMEMORY));

        for (i=0; i<=2; i++) {
          ((MODEMMEMORY *)pTempModemMemory)->aStoredRegs[i].WCommand = DEFAULT_WCOMMAND_VALUE;
        } /* endfor */

        /* Set the new Version number */
        ((MODEMMEMORY *)pTempModemMemory)->usVersion = 4;
        ((MODEMMEMORY *)pTempModemMemory)->usMajorRelease = 0;
        ((MODEMMEMORY *)pTempModemMemory)->usMinorRelease = 3;

      /*************************************************
      ** Unrecognized memory profile
      **************************************************/
      } else {
        DPF("WARNING! The memory profile format is not recognized by this driver!");

        pTempModemMemory = (char *)calloc( sizeof(MODEMMEMORY), 1 );

        memcpy( pTempModemMemory, pMemory, sizeof(MODEMMEMORY));

        /* Set the new Version number */
        ((MODEMMEMORY *)pTempModemMemory)->usVersion =  CURRENT_PRF_VERSION;
        ((MODEMMEMORY *)pTempModemMemory)->usMajorRelease = CURRENT_PRF_MAJOR_RELEASE;
        ((MODEMMEMORY *)pTempModemMemory)->usMinorRelease = CURRENT_PRF_MINOR_RELEASE;
      } /* endif */

      sprintf( achVersion, "%u.%u%u", ((MODEMMEMORY *)pTempModemMemory)->usVersion,
                                      ((MODEMMEMORY *)pTempModemMemory)->usMajorRelease,
                                      ((MODEMMEMORY *)pTempModemMemory)->usMinorRelease );
      free(pMemory);
      pMemory = pTempModemMemory;

    } while ( strcmp(achVersion, achCurrentVersion) );

    /*****************************************************************
    ** Copy the updated profile to stateinfo.
    ******************************************************************/
    memcpy( &(psi->ModemMemory), pMemory, sizeof(MODEMMEMORY) );
    free(pMemory);

    /*****************************************************************
    ** Open the new memory profile file.
    ******************************************************************/
    fMemoryFile = fopen(psi->achMemoryFileName, "wb");

    if (!fMemoryFile) {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_OS_ERROR, 0 );
      return (1);
    } /* endif */

    /*****************************************************************
    ** Write the new structure to the profile file.
    ******************************************************************/
    if (!fwrite( &(psi->ModemMemory), sizeof(MODEMMEMORY), 1, fMemoryFile) ) {
      if ( ferror(fMemoryFile) ) {
        mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                          MWM_OS_ERROR, 0 );
        return (1);
      } /* endif */
    } /* endif */

    /*****************************************************************
    ** Close the new memory profile file.
    ******************************************************************/
    if ( fclose(fMemoryFile) ) {
      mwmIPCHandleError(szThisFile,__LINE__,&psi->mwmError,
                        MWM_OS_ERROR,
                        0 );
      return (1);
    } /* endif */
  } /* endif */

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseReadMemoryProfile exit\n");


  return(0);

}


//void mwmParseDebugLog(UCHAR * pString) /*$2*/
/*{*/
/*   FILE *fDebugFile;*/
/*   fDebugFile=fopen("c:\mww\modem\debug.log","at+");fprintf(fDebugFile,pString);fclose(fDebugFile);*/
/*}*/

void MWM_ENTRY mwmParseSetPipeHandle( HANDLE hPipe )
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetPipeHandle entry\n");

  hWritePipe = hPipe;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetPipeHandle exit\n");

}

void MWM_ENTRY mwmParsePostMessage( UINT uMsg, LONG wParam, LONG lParam )
{
  
  MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParsePostMessage entry\n");
  
  mwmPostMessage(uMsg,wParam,lParam);

  MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParsePostMessage exit\n");
}

ULONG MWM_ENTRY mwmParseSetADPCMaddr(ULONG ulDSPaddr_stepsize,
                                     ULONG ulDSPaddr_indexval,
                                     ULONG ulDSPaddr_newSample)
{

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetADPCMaddr entry\n");

  psi->dsp.dspaddrADPCMstepsize  = ulDSPaddr_stepsize;
  psi->dsp.dspaddrADPCMindexval  = ulDSPaddr_indexval;
  psi->dsp.dspaddrADPCMnewSample = ulDSPaddr_newSample;

 MW_SYSLOG_1(TRACE_MWMPW32,"mwmparse::mwmParseSetADPCMaddr exit\n");


  return 0L;
}


