#ifndef KADU_CHAT_MANAGER_H
#define KADU_CHAT_MANAGER_H

#include <qobject.h>
#include <qtimer.h>

#include "chat.h"
#include "chat_message.h"
#include "toolbar.h"
#include "usergroup.h"

class ChatManagerSlots;
class Protocol;

/**
	Klasa pozwalajca zarzdza otwartymi oknami rozmw: otwiera,
	zamykac, szuka okna ze wgldu na list uytkownikw itp.
	\class ChatManager
	\brief Klasa zarzdzajca oknami Chat
**/

class ChatManager : public QObject
{
	Q_OBJECT

	private:
		ChatList Chats; /*!< lista okien*/

		static ChatManagerSlots *chatslots;

		/**
			\fn int openPendingMsg(int index, ChatMessage &msg)
			Funkcja otwierajca zakolejkowan wiadomo
			\param index nr wiadomoci z kolejki
			\param msg wiadomo z kolejki
			\return zwracany jest nr okna w ktrym zostaa otwarta wiadomo
		**/
		int openPendingMsg(int index, ChatMessage &msg);

		/**
			\struct ChatInfo
			Struktura przechowuje informacje o danym oknie
		**/
		struct ChatInfo
		{
			UserListElements users;/*!< lista uytkownikw
					identyfikujca okno */
			QMap<QString, QVariant> map; /*!< parametry
							danego okna */
			ChatInfo() : users(), map() {}
		};
		QValueList<ChatInfo> addons; /*!< lista parametrw okien */
		QTimer refreshTitlesTimer;

	private slots:

		void autoSendActionActivated(const UserGroup* users, const QWidget* source, bool is_on);
		void scrollLockActionActivated(const UserGroup* users, const QWidget* source, bool is_on);
		void clearActionActivated(const UserGroup* users);
		void historyActionActivated(const UserGroup* users);
		void boldActionActivated(const UserGroup* users, const QWidget* source, bool is_on);
		void italicActionActivated(const UserGroup* users, const QWidget* source, bool is_on);
		void underlineActionActivated(const UserGroup* users, const QWidget* source, bool is_on);
		void colorActionActivated(const UserGroup* users, const QWidget* source);
		void insertEmoticonActionActivated(const UserGroup* users, const QWidget* source);
		void insertEmoticonActionAddedToToolbar(const UserGroup*, ToolButton* button, ToolBar* toolbar);
		void whoisActionActivated(const UserGroup* users);
		void ignoreUserActionActivated(const UserGroup* users);
		void blockUserActionActivated(const UserGroup* users);
		void insertImageActionActivated(const UserGroup* users);
		void sendActionActivated(const UserGroup* users);
		void chatActionActivated(const UserGroup* users);

	public:
		/**
			\fn ChatManager(QObject* parent=NULL, const char* name=NULL)
			Konstruktor tworzcy obiekt zarzdzajcy oknami
			\param parent rodzic okna
			\param name nazwa obiektu
		**/
		ChatManager(QObject* parent=NULL, const char* name=NULL);

		/**
			\fn ~ChatManager()
			Destruktor zamyka wszystkie otwarte okna
		**/
		~ChatManager();

		/**
			\fn static void initModule()
			Rejestruje opcje modulu Chat w oknie konfiguracji
		**/
		static void initModule();

		/**
			\fn static void closeModule()
			Wyrejestrowuje opcje moduu z okna konfiguracji
		**/
		static void closeModule();

		/**
			\fn const ChatList& chats() const
			Funkcja zwraca list otwartych okien Chat
		**/
		const ChatList& chats() const;

		/**
			\fn Chat* findChat(const UserGroup *group) const;
			Funkcja zwraca wskanik do okna z list
			uytkownikw group
			\param group lista uytkownikw
			\return wskanik do okna jeli istnieje w przeciwnym
			 wypadku zwraca NULL
		**/
		Chat* findChat(const UserGroup *group) const;

		/**
			\fn Chat* findChat(UserListElements users) const;
			Funkcja zwraca wskanik do okna z list
			uytkownikw group
			\param users lista uytkownikw
			\return wskanik do okna jeli istnieje w przeciwnym
			 wypadku zwraca NULL
		**/
		Chat* findChat(UserListElements users) const;

		/**
			\fn QVariant& getChatProperty(const UserGroup *group, const QString &name)
			Funkcja zwraca warto wasnoci "name" okna
			okrelonego przez group
			\param group grupa uytkownikw identyfikujca okno
			\param name nazwa wasnoci
			\return zwraca warto wasnoci jeli okrelone okno
			istnieje,\n jeli nie to tworzy tak
			wasno (ustawia na pust)
		**/
		QVariant& getChatProperty(const UserGroup *group, const QString &name);

	public slots:

		/**
			\fn void chatMsgReceived(const QString &protocolName, UserListElements senders, const QString& msg, time_t time, bool& grab)
			Slot informujcy o odebraniu wiadomoci
			\param protocolName nazwa protokou
			\param senders lista uytkownikw identyfikujcych okno
			\param msg tre otrzymanej wiadomoci
			\param time czas otrzymania wiadomoci
			\param grab
		**/
		void chatMsgReceived(Protocol *protocol, UserListElements senders, const QString& msg, time_t time, bool& grab);

		/**
			\fn int openChat(QString initialProtocol, UserListElements users, time_t time = 0)
			Funkcja otwiera nowe okno Chat z wymienionymi rozmowcami.
			\param initialProtocol protok pocztkowy
			\param users lista uytkownikw identyfikujcych okno
			\param time parametr sluzy do sprawdzenia ile wiadomosci
			z historii ma sie pojawic w oknie.
			\return zwracany jest numer otwartego okna
		**/
		int openChat(QString initialProtocol, UserListElements users, time_t time = 0);

		/**
			\fn void openPendingMsgs(UserListElements users)
			Funkcja wpisuje zakolejkowane wiadomoci do okna
			z uytkownikami "users"
			\param users lista uytkownikw identyfikujcych okno
		**/
		void openPendingMsgs(UserListElements users);

		/**
			\fn void openPendingMsgs()
			Funkcja wpisuje wszystkie zakolejkowane wiadomoci
			do odpowiednich okien
		**/
		void openPendingMsgs();

		/**
			\fn void deletePendingMsgs(UserListElements users)
			Funkcja usuwa zakolejkowane wiadomoci
			z uytkownikami "users"
			\param users lista uytkownikw identyfikujcych okno
		**/
		void deletePendingMsgs(UserListElements users);


		//TODO: opisac funkcje sendMessage(..)
		/*
			Niebardzo rozumiem tej funkcji (czemu jest uin i uins)
		*/
		void sendMessage(UserListElement user, UserListElements selected_users);

		/**
			\fn void closeAllWindows()
			Funkcja zamyka wszystkie okna chat
		**/
		void closeAllWindows();

		/**
			\fn int registerChat(Chat* chat)
			Dodaje okno do menadera
			\param chat wskanik do okna ktore chcemy doda
			\return zwraca numer naszego okna po zarejestrowaniu
		**/
		int registerChat(Chat* chat);

		/**
			\fn void unregisterChat(Chat* chat)
			Funkcja wyrejestrowuje okno z managera \n
			Zapisuje wasnoci okna \n
			wysya sygna chatDestroying i chatDestroyed
			\param chat okno ktre bdzie wyrejestrowane
		**/
		void unregisterChat(Chat* chat);

		/**
			\fn void refreshTitles()
			Funkcja odwiea tytuy wszystkich okien
		**/
		void refreshTitles();
		void refreshTitlesLater();

		/**
			\fn void refreshTitlesForUser(UserListElement user)
			Funkcja odwiea tytuy okien ktre zawieraj uin
			\param user uytkownik, ktrego
			opis/status bdzie odwieany
		**/
		void refreshTitlesForUser(UserListElement user);

		/**
			\fn void changeAppearance()
			Funkcja odwiea kolory i czcionki we wszystkich oknach
		**/
		void changeAppearance();

		/**
			\fn void setChatProperty(const UserGroup *group, const QString &name, const QVariant &value)
			Funkcja pozwala przypisa okrelonemu czatowi
			(nawet jeeli on jeszcze nie istnieje) pewne wasnoci
			\param group grupa uytkownikw identyfikujcych okno
			\param name nazwa wasnoci
			\param value warto wasnoci
		**/
		void setChatProperty(const UserGroup *group, const QString &name, const QVariant &value);

	signals:

		/**
			\fn void chatCreated(const UserGroup *group)
		 	Sygna ten jest wysyany po utworzeniu nowego okna chat
			\param group lista uytkownikw, z ktrymi tworzymy nowy chat
		**/
		void chatCreated(const UserGroup *group);

		/**
			\fn void chatDestroying(const UserGroup *group)
		 	Sygna ten jest wysyany przed zamnkniciem okna chat
			\param group lista uytkownikww, ktrzy wystpuj w tym chacie
		**/
		void chatDestroying(const UserGroup *group);

		/**
			\fn void chatDestroyed(const UserGroup *group)
		 	Sygna ten jest wysyany po zamnkniciem okna chat
			\param group lista uytkownikw, ktrzy
			wystpoway w tym chacie
		**/
		void chatDestroyed(const UserGroup *group);

		/**
			\fn void chatOpen(UserListElements users)
			Sygna ten jest wysyaniy podczas kadej prby
			otwarcia nowego okna chat nawet jeli ju taki istnieje
			\param users lista uytkownikw
		**/
		void chatOpen(UserListElements users);

		/**
			\fn void messageSentAndConfirmed(UserListElements receivers, const QString& message)
			This signal is emited when message was sent
			and it was confirmed.
			When confirmations are turned off signal is
			emited immediately after message was send.
			\param receivers list of receivers
			\param message the message
		**/
		void messageSentAndConfirmed(UserListElements receivers, const QString& message);
};

/**
	wskanik do obiektu ChatManager'a
**/
extern ChatManager* chat_manager;

#endif
