#ifndef XMMS_H
#define XMMS_H

#include <qobject.h>

#include "chat.h"
#include "modules.h"
#include "misc.h"
#include "config_file.h"

class QTimer;
class UserGroup;
class ToolBar;
class ToolButton;

class Xmms : public QObject
{
	Q_OBJECT

	public:
		Xmms();
		~Xmms();

		/*
			Returns song title (author and title) as string.
			Optional argument 'position' defines XMMS playlist
			entry, which title should be returned of.
			Value -1 of 'position' means current playing track.
		*/
		QString getTitle(int position = -1);

		/*
			Returns song file name as string.
			Optional argument 'position' similar as in
			getTitle() method.
		*/
		QString getFile(int position = -1);

		/*
			Returns track length in miliseconds.
			Optional argument 'position' similar as in
			getTitle() method.
		*/
		int getLength(int position = -1);

		/*
			Returns current position of played track in miliseconds.
		*/
		int getCurrentPos();

		/*
			Looks for special tags in string 'str' and replaces it by
			assigned values. Then returns string in new form.

			Tag:	  Replaces by:
			%t	  Song title
			%f	  File name
			%l	  Length in format MM:SS
			%c	  Current playing position in format MM:SS
			%p	  Current playing position in percents
			%d	  Current user description (if any).
		*/
		QString parse(QString str);

		/*
			Returns 1 if XMMS is playing, or 0 otherwise.
		*/
		bool isPlaying();

		/*
			Converts given string 'str' from CP1250 to ISO8859-2
			and returns modified string.
		*/
		QString convert(QString str);

		/*
			Returns formatted string of track length, which should be
			value returned by getLength() or getCurrentPosition() method.
		*/
		QString formatLength(int length);

		QStringList getPlayListTitles();
		QStringList getPlayListFiles();
		uint getPlayListLength();

	private:
		QTimer* timer;
		QString currentTitle;
		int pos();
		QPopupMenu* menu;
		int popups[6];
		bool winKeyPressed;
		ConfigFile* config;
		QMap<Chat*,QPushButton*> chatButtons;

		/*
			Returns Chat window, which already has a focus.
		*/
		Chat* getCurrentChat();

		/*
			Applies XMMS functions to newly created Chat window.
		*/
		void handleCreatedChat(Chat* chat);

		/*
			Removes XMMS functions from Chat window being destroyed.
		*/
		void handleDestroyingChat(Chat* chat);

	private slots:
		void chatCreated(const UserGroup*);
		void chatDestroying(const UserGroup*);
		void checkTitle();
		void toggleStatuses(int);
		void checkStatusDesc(UserStatus&, bool&);
		void chatKeyPressed(QKeyEvent*, CustomInput*, bool&);
		void chatKeyReleased(QKeyEvent*, CustomInput*, bool&);

	public slots:

		/*
			Puts song title into current chat edit field.
		*/
		void putSongTitle(int);

		/*
			Puts whole playlist into current chat edit field.
		*/
		void putPlayList(int);

		/*
			Shows currently played title in hint (Pseudo-OSD).
		*/
		void putTitleHint(QString title);

		/*
			Orders XMMS to play the next one track.
		*/
		void nextTrack();

		/*
			Orders XMMS to play the previous one track.
		*/
		void prevTrack();

		/*
			Orders XMMS to start playing.
		*/
		void play();

		/*
			Orders XMMS to stop playing.
		*/
		void stop();

		/*
			Orders XMMS to pause playing.
		*/
		void pause();

		/*
			Orders XMMS to sset volume to 'vol' (in percents).
		*/
		void setVolume(int vol);

		/*
			Orders XMMS to increment volume about 2%.
		*/
		void incrVolume();

		/*
			Orders XMMS to decrement volume about 2%.
		*/
		void decrVolume();

		/*
			Slots for ConfigDialog class.
		*/
		void onCreateConfig();
		void onUpdateConfig();
		void onDestroyConfig();

	// XMMS Button
	//void onButtonAction(const UserGroup* users, bool is_on);
	//void onAddedButton(ToolButton *button, ToolBar *toolbar, const UserListElements& users);
	void xmmsActionActivated(const UserGroup* grp, const QWidget* source);
};

#endif
