/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qapplication.h>
#include <qlistview.h>
#include "git.h"
#include "mainimpl.h"
#include "domain.h"

using namespace QGit;

void StateInfo::S::clear() {

	sha = fn = dtSha = "";
	isM = allM = false;
	sel = true;
}

StateInfo::S& StateInfo::S::operator=(const StateInfo::S& st) {

	if (&st != this) {
		sha   = st.sha;
		fn    = st.fn;
		dtSha = st.dtSha;
		sel   = st.sel;
		isM   = st.isM;
		allM  = st.allM;
	}
	return *this;
}

bool StateInfo::S::operator==(const StateInfo::S& st) const {

	if (&st == this)
		return true;

	return (   sha   == st.sha
	        && fn    == st.fn
	        && dtSha == st.dtSha
	        && sel   == st.sel
	        && isM   == st.isM
	        && allM  == st.allM );
}

bool StateInfo::S::operator!=(const StateInfo::S& st) const {

	return !(StateInfo::S::operator==(st));
}

void StateInfo::clear() {

	nextS.clear();
	curS.clear();
	prevS.clear();
	isLocked = false;
}

StateInfo& StateInfo::operator=(const StateInfo& newState) {

	if (&newState != this) {
		if (isLocked)
			nextS = newState.curS;
		else
			curS = newState.curS; // prevS is mot modified to allow a rollback
	}
	return *this;
}

bool StateInfo::operator==(const StateInfo& newState) const {

	if (&newState == this)
		return true;

	return (curS == newState.curS); // compare is made on curS only
}

bool StateInfo::operator!=(const StateInfo& newState) const {

	return !(StateInfo::operator==(newState));
}

// ************************* Domain ****************************

Domain::Domain(MainImpl* m, Git* g) : QObject(m), git(g) {

	d = this;
	st.clear();
	busy = readyToDrag = dragging = dropping = false;
	popupType = 0;
}

MainImpl* Domain::m() const { return static_cast<MainImpl*>(parent()); }

bool Domain::setReadyToDrag(bool b) {

	readyToDrag = (b && !busy && !dragging && !dropping);
	return readyToDrag;
}

bool Domain::setDragging(bool b) {

	bool dragFinished = (!b && dragging);

	dragging = (b && readyToDrag && !dropping);

	if (dragging)
		readyToDrag = false;

	if (dragFinished)
		flushQueue();

	return dragging;
}

bool Domain::flushQueue() {
// during dragging any state update is queued, so try to flush pending now

	if (!busy && st.flushQueue()) {
		UPDATE_DOMAIN(d);
		return true;
	}
	return false;
}

void Domain::customEvent(QCustomEvent* e) {

	switch (e->type()) {
	case UPD_DOMAIN_EV:
		update();
		break;
	case MSG_EV:
		if (!busy && !st.requestPending())
			QApplication::postEvent(m(), new MessageEvent(((MessageEvent*)e)->data()));
		else // waiting for the end of updating
			statusBarRequest = ((MessageEvent*)e)->data();
		break;
	default:
		break;
	}
	QObject::customEvent(e);
}

void Domain::populateState() {

	const Rev* r = git->revLookup(st.sha());
	if (r)
		st.setIsMerge(r->parentsCount() > 1);
}

void Domain::update() {

	if (st.sha().isEmpty()) {
		dbs("ASSERT update called with empty sha");
		return;
	}
	if (busy || dragging)
		return;

	busy = true;

	setReadyToDrag(false); // do not start any drag while updating
	populateState(); // complete any missing state information

	st.setLock(true); // any state change will be queued now

	if (doUpdate())
		st.commit();
	else
		st.rollBack();

	st.setLock(false);

	busy = false;

	bool nextRequestPending = flushQueue();

	if (!nextRequestPending && !statusBarRequest.isEmpty()) {
		// update status bar when we are sure no more work is pending
		QApplication::postEvent(m(), new MessageEvent(statusBarRequest));
		statusBarRequest = "";
	}
	if (!nextRequestPending && popupType)
		sendPopupEvent();
}

void Domain::sendPopupEvent() {

	// call an async context popup, must be executed
	// after returning to event loop
	DeferredPopupEvent* e = new DeferredPopupEvent(popupData, popupType);
	QApplication::postEvent(m(), e);
	popupType = 0;
}

void Domain::on_contextMenu(const QString& data, int type) {

	popupType = type;
	popupData = data;

	if (busy)
		return; // we are in the middle of an update

	// if list view is already updated pop-up
	// context menu, otherwise it means update()
	// has still not been called, a pop-up request,
	// will be fired up at the end of next update()
	if ((type == POPUP_LIST_EV) && (data != st.sha()))
		return;

	sendPopupEvent();
}
