/* -*- Mode: C++; c-file-style: "stroustrup"; indent-tabs-mode: nil -*- */
/*
 * ddtc.cc
 *   User client main file.
 *
 * $Id: ddtc.cc,v 1.13 2002/04/08 07:11:00 benoit Exp $
 *
 * Copyright (c) 2000-2001 Remi Lefebvre <remi@ddts.org>
 *
 * DDT comes with ABSOLUTELY NO WARRANTY and is licenced under the
 * GNU General Public License (version 2 or later). This license
 * can be retrieved from http://www.gnu.org/copyleft/gnu.html.
 *
 */


#if defined(FREEBSD2) || defined(FREEBSD3) || defined(FREEBSD4) \
    || defined(BSDI2) || defined(BSDI3) || defined(BSDI4) \
    || defined(OPENBSD2) || defined(NETBSD1)
#include <sys/cdefs.h>
#endif

#include <iostream>
#include <sstream>
#include <getopt.h>

#include "ddtcd.h" // for the fifo names

using std::cerr;
using std::ostringstream;

#define C_NOTSET 0
#define C_ONLINE 1
#define C_OFFLINE 2
#define C_STATUS 3

int parseOptions(int argc, char **argv);
int usage();

/* global parameters. one might say its bad but that proggy is so small. */
int action;
int batchmode;
unsigned long ip;

int main(int argc, char **argv)
{
    int infifofd;
    int outfifofd;
    
    parseOptions(argc, argv);

    /* open both input and output fifos */
    
    // open O_RDWR to avoid select() bug.
    if ((infifofd = open(OUTFIFO, O_RDONLY | O_NONBLOCK)) == -1)
    {
        cerr << "Failed to open input FIFO.\n";
        exit(1);
    }
    
    if ((outfifofd = open(INFIFO, O_WRONLY)) == -1)
    {
        cerr << "Failed to open output FIFO.\n";
        exit(1);
    }
    
    switch (action)
    {
    case C_ONLINE:
    {
        ostringstream online;
        online << "online " << (ip == 0 ? 0 : htonl(ip));
        write(outfifofd, online.str().c_str(), strlen(online.str().c_str()));
        break;
    }
    case C_OFFLINE:
        write(outfifofd, "offline", strlen("offline"));
        break;
            
    case C_STATUS:
        write(outfifofd, "status", strlen("status"));
        break;
            
    case C_NOTSET:
    default:
        fprintf(stderr, "Invalid action request\n");
        exit(1);
        break;
    }

    if (batchmode)
    {
        exit (0);
    }

    fd_set rset;
    struct timeval tv;

    FD_ZERO(&rset);

    /* wait up to fifteen secs for a reply, otherwise, we abandon */
    tv.tv_sec = 15;
    tv.tv_usec = 0;

    FD_SET(infifofd, &rset);

    select(infifofd + 1, &rset, NULL, NULL, &tv);

    int n;
    char string[256];

    /* there's stuff available, let's read it */
    while (1)
    {
        n = read(infifofd, string, sizeof(string));
        if ((n == -1) && (errno == EAGAIN))
        {
            /* avoid consuming all cpu */
            usleep(100);
            continue;
        }
        if (n == -1)
        {
            printf("read failed: %s\n", strerror (errno));
            break;
        }
                    
        if (n == 0)
        {
//            usleep (100);
//            continue;
            break;
        }
        
        string[n] = '\0';
        printf(string);
    }
}

int parseOptions(int argc, char **argv)
{
    int c;
    int option_index = 0;
    static struct option options[] = {
        { "help", 0, NULL, 'h' },
        { "online", 2, NULL, 'o'},
        { "offline", 0, NULL, 'f'},
        { "status", 0, NULL, 'S'},
        { "batchmode", 0, NULL, 'b'},
        { 0, 0, 0, 0 }
    };

    while ((c = getopt_long(argc, argv, "ho::fbS",
                            options, &option_index)) != EOF)
    {
        switch (c)
        {
        case 'o':
            if (optarg)
                ip = inet_addr(optarg);
            else
                ip = 0;
            action = C_ONLINE;
            break;
                
        case 'f':
            action = C_OFFLINE;
            break;
                
        case 'S':
            action = C_STATUS;
            break;

        case 'b':
            batchmode = 1;
            break;
                
        case 'h':
        case '?':
        default:
            usage();
            break;
        }
    }
    return 0;
}

int usage ()
{
    fprintf(stderr, "ddtc -o[ADDRESS]|-f|-S [-b] [-h]\n");
    return 0;
}
