/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.1, (the "License"); you may not use this file except in
 * compliance with the License. You should have received a copy of the
 * Erlang Public License along with this software. If not, it can be
 * retrieved via the world wide web at http://www.erlang.org/.
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Original Code is Ericsson Utvecklings AB.
 * Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
 * AB. All Rights Reserved.''
 * 
 *     $Id$
 */
/*
 * Function:
 * ei_print_term to print out a binary coded term
 */

#ifdef VXWORKS
#include <vxWorks.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#ifdef VRTX
#define __READY_EXTENSIONS__
#include <errno.h>
#endif

#include "ei.h"
#include "ei_printterm.h"
#include "erl_error.h"
#include "erl_malloc.h"


#define BINPRINTSIZE 30

/*
 * PRINT out a binary term (hacked from erl_print_term)
 */

static int print_string(FILE* fp, ei_x_buff* x, char* s);
static int print_term(FILE* fp, ei_x_buff* x,
			       const char* buf, int* index);

static void xputc(char c, FILE* fp, ei_x_buff* x)
{
    if (fp != NULL)
	putc(c, fp);
    else
	ei_x_append_buf(x, &c, 1);
}

static void xputs(const char* s, FILE* fp, ei_x_buff* x)
{
    if (fp != NULL)
	fputs(s, fp);
    else
	ei_x_append_buf(x, s, strlen(s));
}

static int xprintf(FILE* fp, ei_x_buff* x, const char* fmt, ...)
{
    int r = 0;
    va_list ap;
    va_start(ap, fmt);
    if (fp != NULL)
	r = vfprintf(fp, fmt, ap);
    else {
	char tmpbuf[2000];
	r = vsprintf(tmpbuf, fmt, ap);
	ei_x_append_buf(x, tmpbuf, strlen(tmpbuf));
    }
    return r;
}

static int print_term(FILE* fp, ei_x_buff* x,
			       const char* buf, int* index)
{
    int i, doquote, n, m, ty;
    char a[MAXATOMLEN], *p;
    int ch_written = 0;		/* counter of written chars */
    erlang_pid pid;
    erlang_port port;
    erlang_ref ref;
    double d;
    long l;
    unsigned long u;

    if (fp == NULL && x == NULL) return -1;

    doquote = 0;
    ei_get_type(buf, index, &ty, &n);
    switch (ty) {
    case ERL_ATOM_EXT:
	if (ei_decode_atom(buf, index, a) < 0)
	   goto err;
	doquote = !islower(a[0]);
	for (p = a; !doquote && *p != '\0'; ++p)
	    doquote = !(isalnum(*p) || *p == '_' || *p == '@');
	if (doquote) {
	    xputc('\'', fp, x); ++ch_written; 
	}
	xputs(a, fp, x); ch_written += strlen(a);
	if (doquote) {
	    xputc('\'', fp, x); ++ch_written;
	}
	break;
    case ERL_PID_EXT:
	if (ei_decode_pid(buf, index, &pid) < 0) goto err;
	ch_written += xprintf(fp, x, "<%s.%d.%d>", pid.node,
			      pid.num, pid.serial);
	break;
    case ERL_PORT_EXT:
	if (ei_decode_port(buf, index, &port) < 0) goto err;
	ch_written += xprintf(fp, x, "#Port<%d.%d>", port.id, port.creation);
	break;
    case ERL_NEW_REFERENCE_EXT:
    case ERL_REFERENCE_EXT:
	if (ei_decode_ref(buf, index, &ref) < 0) goto err;
	ch_written += xprintf(fp, x, "#Ref<");
	for (i = 0; i < ref.len; ++i) {
	    ch_written += xprintf(fp, x, "%d", ref.n[i]);
	    if (i < ref.len - 1) {
		xputc('.', fp, x); ++ch_written;
	    }
	}
	xputc('>', fp, x); ++ch_written;
	break;
    case ERL_NIL_EXT:
	if (ei_decode_list_header(buf, index, &n) < 0) goto err;
	ch_written += xprintf(fp, x, "[]");
	break;
    case ERL_LIST_EXT:
	if (ei_decode_list_header(buf, index, &n) < 0) goto err;
	xputc('[', fp, x); ch_written++;
	for (i = 0; i < n; ++i) {
	    ch_written += print_term(fp, x, buf, index);
	    if (i < n - 1) {
		xputs(", ", fp, x); ch_written += 2;
	    }
	}
	if (ei_get_type(buf, index, &ty, &n) < 0) goto err;
	if (ty != ERL_NIL_EXT) {
	    xputs(" | ", fp, x); ch_written += 3;
	    ch_written += print_term(fp, x, buf, index);
	} else {
	    if (ei_decode_list_header(buf, index, &n) < 0) goto err;
	}
	xputc(']', fp, x); ch_written++;
	break;
    case ERL_STRING_EXT:
	p = erl_malloc(n+1);
	if (p == NULL) goto err;
	if (ei_decode_string(buf, index, p) < 0) {
	    erl_free(p);
	    goto err;
	}
	ch_written += print_string(fp, x, p);
	erl_free(p);
	break;
    case ERL_SMALL_TUPLE_EXT:
    case ERL_LARGE_TUPLE_EXT:
	if (ei_decode_tuple_header(buf, index, &n) < 0) goto err;	
	xputc('{', fp, x); ch_written++;
	for (i = 0; i < n; ++i) {
	    ch_written += print_term(fp, x, buf, index);
	    if (i < n-1) {
		xputs(", ", fp, x); ch_written += 2;
	    }
	}
	xputc('}', fp, x); ch_written++;
	break;
    case ERL_BINARY_EXT:
	p = erl_malloc(n);
	if (p == NULL) goto err;
	if (ei_decode_binary(buf, index, p, (long*)&n) < 0) {
	    erl_free(p);
	    goto err;
	}
	ch_written += xprintf(fp, x, "#Bin<");
	m = n;
	if (n > BINPRINTSIZE)
	    m = BINPRINTSIZE;
	--m;
	for (i = 0; i < m; ++i) {
	    ch_written += xprintf(fp, x, "%d,", p[i]);
	}
	ch_written += xprintf(fp, x, "%d", p[i]);
	if (n > BINPRINTSIZE)
	    ch_written += xprintf(fp, x, ",...");
	xputc('>', fp, x); ++ch_written;
	erl_free(p);
	break;
    case ERL_SMALL_INTEGER_EXT:
    case ERL_INTEGER_EXT:
	if (ei_decode_long(buf, index, &l) < 0) goto err;
	ch_written += xprintf(fp, x, "%ld", l);
	break;
    case ERL_SMALL_BIG_EXT:
	if (ei_decode_ulong(buf, index, &u) < 0) goto err;
	ch_written += xprintf(fp, x, "%lu", u);
	break;
    case ERL_LARGE_BIG_EXT:
	ch_written += xprintf(fp, x, "BigNum");
	*index += n;
	break;
    case ERL_FLOAT_EXT:
	if (ei_decode_double(buf, index, &d) < 0) goto err;
	ch_written += xprintf(fp, x, "%f", d);
	break;
    default:
	goto err;
    }
    return ch_written;
 err:
    erl_err_msg("<ERROR> ei_print_term: Bad type of term !");
    return -1;
}

static int print_string(FILE* fp, ei_x_buff* x, char* s)
{
    int ch_written = 0;		/* counter of written chars */
  
    xputc('"', fp, x);
    ++ch_written;
    for (; *s != '\0'; ++s) {
	int c = *s;
	if (c >= ' ') {
	    xputc(c, fp, x); ++ch_written; }
	else {
	    switch (c) {
	    case '\n': xputs("\\n", fp, x); ch_written += 2; break;
	    case '\r': xputs("\\r", fp, x); ch_written += 2; break;
	    case '\t': xputs("\\t", fp, x); ch_written += 2; break;
	    case '\v': xputs("\\v", fp, x); ch_written += 2; break;
	    case '\b': xputs("\\b", fp, x); ch_written += 2; break;
	    case '\f': xputs("\\f", fp, x); ch_written += 2; break;
		break;
	    default:
		ch_written += xprintf(fp, x, "\\x%x", c);
		break;
	    }
	}
    }
    xputc('"', fp, x); ch_written++;
    return ch_written;
}


/* ------------------------------------------ */

/*
 * skip a binary term
 */


int ei_print_term(FILE *fp, const char* buf, int* index)
{
    return print_term(fp, NULL, buf, index);
}

int ei_s_print_term(char** s, const char* buf, int* index)
{
    int r;
    ei_x_buff x;
    if (*s != NULL) {
	x.buff = *s;
	x.index = 0;
	x.buffsz = BUFSIZ;
    } else
	ei_x_new(&x);
    r = print_term(NULL, &x, buf, index);
    ei_x_append_buf(&x, "", 1); /* append '\0' */
    *s = x.buff;
    return r;
}

int ei_skip_term(const char* buf, int* index)
{
    int i, n, ty;

    /* ASSERT(ep != NULL); */

    ei_get_type(buf, index, &ty, &n);
    switch (ty) {
    case ERL_ATOM_EXT:
	/* XXX: what if some weird locale is in use? */
	if (ei_decode_atom(buf, index, NULL) < 0)
	   goto err;
	break;
    case ERL_PID_EXT:
	if (ei_decode_pid(buf, index, NULL) < 0) goto err;
	break;
    case ERL_PORT_EXT:
	if (ei_decode_port(buf, index, NULL) < 0) goto err;
	break;
    case ERL_NEW_REFERENCE_EXT:
    case ERL_REFERENCE_EXT:
	if (ei_decode_ref(buf, index, NULL) < 0) goto err;
	break;
    case ERL_NIL_EXT:
	if (ei_decode_list_header(buf, index, &n) < 0) goto err;
	break;
    case ERL_LIST_EXT:
	if (ei_decode_list_header(buf, index, &n) < 0) goto err;
	for (i = 0; i < n; ++i)
	    ei_skip_term(buf, index);
	if (ei_get_type(buf, index, &ty, &n) < 0) goto err;
	if (ty != ERL_NIL_EXT)
	    ei_skip_term(buf, index);
	else
	    if (ei_decode_list_header(buf, index, &n) < 0) goto err;
	break;
    case ERL_STRING_EXT:
	if (ei_decode_string(buf, index, NULL) < 0) goto err;
	break;
    case ERL_SMALL_TUPLE_EXT:
    case ERL_LARGE_TUPLE_EXT:
	if (ei_decode_tuple_header(buf, index, &n) < 0) goto err;	
	for (i = 0; i < n; ++i)
	    ei_skip_term(buf, index);
	break;
    case ERL_BINARY_EXT:
	if (ei_decode_binary(buf, index, NULL, (long*)&n) < 0)
	    goto err;
	break;
    case ERL_SMALL_INTEGER_EXT:
    case ERL_INTEGER_EXT:
	if (ei_decode_long(buf, index, NULL) < 0) goto err;
	break;
    case ERL_SMALL_BIG_EXT:
	if (ei_decode_ulong(buf, index, NULL) < 0) goto err;
	break;
    case ERL_LARGE_BIG_EXT:
	*index += n; /* !! Funkar detta? */
	break;
    case ERL_FLOAT_EXT:
	if (ei_decode_double(buf, index, NULL) < 0) goto err;
	break;
    case ERL_FUN_EXT:
	if (ei_decode_fun(buf, index, NULL) < 0) goto err;
	break;
    default:
	goto err;
    }
    return 0;
 err:
    erl_err_msg("<ERROR> ei_skip_term: unknown type of term !");
    return -1;
}

