/* Copyright (C) 2001-2020 Artifex Software, Inc.
   All Rights Reserved.

   This software is provided AS-IS with no warranty, either express or
   implied.

   This software is distributed under license and may not be copied,
   modified or distributed except as expressly authorized under the terms
   of the license contained in the file LICENSE in this distribution.

   Refer to licensing information at http://www.artifex.com or contact
   Artifex Software, Inc.,  1305 Grant Avenue - Suite 200, Novato,
   CA 94945, U.S.A., +1(415)492-9861, for further information.
*/


/* Transparency parameter definitions */

#ifndef gstparam_INCLUDED
#  define gstparam_INCLUDED

#include "gsccolor.h"
#include "gsrefct.h"
#include "gscspace.h"
#include "stdint_.h"
#include "gsfunc.h"

/* Define the names of the known blend modes. */
typedef enum {
    BLEND_MODE_Normal,
    BLEND_MODE_Multiply,
    BLEND_MODE_Screen,
    BLEND_MODE_Difference,
    BLEND_MODE_Darken,
    BLEND_MODE_Lighten,
    BLEND_MODE_ColorDodge,
    BLEND_MODE_ColorBurn,
    BLEND_MODE_Exclusion,
    BLEND_MODE_HardLight,
    BLEND_MODE_Overlay,
    BLEND_MODE_SoftLight,
    BLEND_MODE_Luminosity,
    BLEND_MODE_Hue,
    BLEND_MODE_Saturation,
    BLEND_MODE_Color,
    BLEND_MODE_CompatibleOverprint,
#define MAX_BLEND_MODE BLEND_MODE_CompatibleOverprint

    /* For compatibility with old PDFs */
    BLEND_MODE_Compatible
} gs_blend_mode_t;
#define GS_BLEND_MODE_NAMES\
  "Normal", "Multiply", "Screen", "Difference",\
  "Darken", "Lighten", "ColorDodge", "ColorBurn", "Exclusion",\
  "HardLight", "Overlay", "SoftLight", "Luminosity", "Hue",\
  "Saturation", "Color", "CompatibleOverprint", "Compatible"

#define blend_is_idempotent(B)       \
    (((((1<<BLEND_MODE_Multiply)   | \
        (1<<BLEND_MODE_Screen)     | \
        (1<<BLEND_MODE_Overlay)    | \
        (1<<BLEND_MODE_ColorDodge) | \
        (1<<BLEND_MODE_ColorBurn)  | \
        (1<<BLEND_MODE_HardLight)  | \
        (1<<BLEND_MODE_SoftLight)  | \
        (1<<BLEND_MODE_Difference) | \
        (1<<BLEND_MODE_Exclusion)) >> (B)) & 1) == 0)

/* Define the parameter structure for a transparency group. */
/* (Update gs_trans_group_params_init if these change.) */
typedef struct gs_transparency_group_params_s {
    const gs_color_space *ColorSpace;
    bool Isolated;
    bool Knockout;
    float group_shape;
    float group_opacity;
    bool image_with_SMask;
    int text_group;
    bool shade_group;
    bool page_group;
    bool idle;
    uint mask_id;
    int group_color_numcomps;
    gs_transparency_color_t group_color_type;
    int64_t icc_hashcode;                    /* Needed when we are doing clist reading */
    cmm_profile_t *iccprofile;               /* The profile  */
} gs_transparency_group_params_t;

/* Define the parameter structure for a transparency mask. */
typedef enum {
    TRANSPARENCY_MASK_Alpha,
    TRANSPARENCY_MASK_Luminosity,
    TRANSPARENCY_MASK_None          /* special case for removing a SMask from the tos */
} gs_transparency_mask_subtype_t;

#define GS_TRANSPARENCY_MASK_SUBTYPE_NAMES\
  "Alpha", "Luminosity", "None"

/* See the gx_transparency_mask_params_t type below */
/* (Update gs_trans_mask_params_init if these change.) */
typedef struct gs_transparency_mask_params_s {
    const gs_color_space *ColorSpace;
    gs_transparency_mask_subtype_t subtype;
    int Background_components;
    int Matte_components;
    float Background[GS_CLIENT_COLOR_MAX_COMPONENTS];
    float Matte[GS_CLIENT_COLOR_MAX_COMPONENTS];
    float GrayBackground;
    int (*TransferFunction)(double in, float *out, void *proc_data);
    gs_function_t *TransferFunction_data;
    bool replacing;
    int64_t icc_hashcode;                    /* Needed when we are doing clist reading */
    cmm_profile_t *iccprofile;               /* The profile  */
} gs_transparency_mask_params_t;

#define MASK_TRANSFER_FUNCTION_SIZE 256

/* The post clist version of transparency mask parameters */
typedef struct gx_transparency_mask_params_s {
    gs_transparency_mask_subtype_t subtype;
    int group_color_numcomps;
    gs_transparency_color_t group_color_type;
    int Background_components;
    int Matte_components;
    float Background[GS_CLIENT_COLOR_MAX_COMPONENTS];
    float Matte[GS_CLIENT_COLOR_MAX_COMPONENTS];
    float GrayBackground;
    bool function_is_identity;
    bool idle;
    bool replacing;
    uint mask_id;
    byte transfer_fn[MASK_TRANSFER_FUNCTION_SIZE*2+2];
    int64_t icc_hashcode;                    /* Needed when we are doing clist reading */
    cmm_profile_t *iccprofile;               /* The profile  */
} gx_transparency_mask_params_t;

/*
   The longest compositor command includes transparency mask parameters
   (see gdevp14.c c_pdf14trans_write() the "case PDF14_BEGIN_TRANS_MASK").
   Note that 'idle' is not written because it is generated by the clist interpreter.
*/
#define MAX_CLIST_TRANSPARENCY_BUFFER_SIZE ( \
             4 /* the clist cmd prefix byte cmd_opv_extend + extended function + 2 bytes for band_id */ + \
             1 + sizeof(float) * 6 /* See sput_matrix. */ + \
             sizeof(((gs_pdf14trans_params_t *)0)->subtype) + \
             sizeof(((gs_pdf14trans_params_t *)0)->group_color_numcomps) + \
             5 /* group color, replacing, function_is_identity, Background_components, Matte_components */ + \
             sizeof(((gs_pdf14trans_params_t *)0)->bbox) + \
             sizeof(((gs_pdf14trans_params_t *)0)->mask_id) + \
             sizeof(((gs_pdf14trans_params_t *)0)->Background) + \
             sizeof(((gs_pdf14trans_params_t *)0)->Matte) + \
             sizeof(float)*4 + /* If cmyk background */ \
             sizeof(((gs_pdf14trans_params_t *)0)->GrayBackground) + \
             sizeof(int64_t)) /* ICC band information */
#define MAX_CLIST_TRANSPARENCY_COMPOSITOR_SIZE (MAX_CLIST_TRANSPARENCY_BUFFER_SIZE + \
             sizeof(((gs_pdf14trans_params_t *)0)->transfer_fn))
#define MAX_CLIST_COMPOSITOR_SIZE MAX_CLIST_TRANSPARENCY_COMPOSITOR_SIZE

/* Select the opacity or shape parameters. */
typedef enum {
    TRANSPARENCY_CHANNEL_Opacity = 0,
    TRANSPARENCY_CHANNEL_Shape = 1
} gs_transparency_channel_selector_t;

#endif /* gstparam_INCLUDED */
