<?php

/**
 * Matomo - free/libre analytics platform
 *
 * @link    https://matomo.org
 * @license https://www.gnu.org/licenses/gpl-3.0.html GPL v3 or later
 */

namespace Piwik\Plugins\LanguagesManager\TranslationWriter\Filter;

class ByParameterCount extends FilterAbstract
{
    protected $baseTranslations = array();

    /**
     * Sets base translations
     *
     * @param array $baseTranslations
     */
    public function __construct($baseTranslations = array())
    {
        $this->baseTranslations = $baseTranslations;
    }

    /**
     * Removes all translations where the placeholder parameter count differs to base translation
     *
     * @param  array $translations
     *
     * @return array   filtered translations
     */
    public function filter($translations)
    {
        $cleanedTranslations = array();

        foreach ($translations as $pluginName => $pluginTranslations) {
            foreach ($pluginTranslations as $key => $translation) {
                if (isset($this->baseTranslations[$pluginName][$key])) {
                    $baseTranslation = $this->baseTranslations[$pluginName][$key];
                } else {
                    // english string was deleted, do not error
                    continue;
                }

                // ensure that translated strings have the same number of %s as the english source strings
                $baseCount = $this->getParametersCountToReplace($baseTranslation);
                $translationCount = $this->getParametersCountToReplace($translation);

                if ($baseCount != $translationCount) {
                    $this->filteredData[$pluginName][$key] = $translation;
                    continue;
                }

                $cleanedTranslations[$pluginName][$key] = $translation;
            }
        }

        return $cleanedTranslations;
    }

    /**
     * Counts the placeholder parameters n given string
     *
     * @param string $string
     * @return array
     */
    protected function getParametersCountToReplace($string)
    {
        $sprintfParameters = array('%s', '%1$s', '%2$s', '%3$s', '%4$s', '%5$s', '%6$s', '%7$s', '%8$s', '%9$s');
        $count = array();
        foreach ($sprintfParameters as $parameter) {
            $placeholderCount = substr_count($string, $parameter);
            if ($placeholderCount > 0) {
                $count[$parameter] = $placeholderCount;
            }
        }
        return $count;
    }
}
