package hardware

import (
	"encoding/json"
	"io/ioutil"

	hhardware "github.com/jouyouyun/hardware"
	hbat "github.com/jouyouyun/hardware/battery"
	hcpu "github.com/jouyouyun/hardware/cpu"
	hdisk "github.com/jouyouyun/hardware/disk"
	hdmi "github.com/jouyouyun/hardware/dmi"
	hmem "github.com/jouyouyun/hardware/memory"
	hnet "github.com/jouyouyun/hardware/network"
)

type Hardware struct {
	ID              string   `json:"id"`
	Hostname        string   `json:"hostname"`
	Username        string   `json:"username"`
	OS              string   `json:"os"`
	CPU             string   `json:"cpu"`
	Laptop          bool     `json:"laptop"`
	Memory          int64    `json:"memory"`
	DiskTotal       int64    `json:"disk_total"`
	NetworkCardList string   `json:"network_cards"`
	DiskList        string   `json:"disk_list"`
	DMI             hdmi.DMI `json:"dmi"`
}

const (
	fileHostname = "/etc/hostname"
)

func GetHardware() (*Hardware, error) {
	os, err := getOSFromDeepin(fileOS)
	if err != nil {
		return nil, err
	}
	hostname, err := readFile(fileHostname)
	if err != nil {
		return nil, err
	}

	cpu, _ := hcpu.NewCPU()
	mem, _ := hmem.GetMemoryList()
	dmi, _ := hdmi.GetDMI()
	netList, _ := hnet.GetNetworkList()
	diskList, _ := hdisk.GetDiskList()
	batList, _ := hbat.GetBatteryList()

	var info = &Hardware{
		Hostname:        hostname,
		OS:              os,
		Laptop:          len(batList) != 0,
		DiskTotal:       calcDiskTotal(diskList),
		NetworkCardList: string(jsonMarshal(netList)),
		DiskList:        string(jsonMarshal(diskList)),
	}
	if cpu != nil {
		info.CPU = cpu.Name
	}
	if len(mem) > 0 {
		info.Memory = mem[0].Capacity
	}
	if dmi != nil {
		info.DMI = *dmi
	}

	info.ID, err = hhardware.GenMachineID()
	if err != nil {
		return nil, err
	}
	return info, nil
}

func calcDiskTotal(diskList hdisk.DiskList) int64 {
	if len(diskList) == 0 {
		return 0
	}
	var total int64
	for _, d := range diskList {
		total += d.Size
	}
	return total / 1024 //kb
}

func (hw *Hardware) Bytes() []byte {
	data, _ := json.Marshal(hw)
	return data
}

func jsonMarshal(info interface{}) []byte {
	data, _ := json.Marshal(info)
	return data
}

func readFile(filename string) (string, error) {
	data, err := ioutil.ReadFile(filename)
	if err != nil {
		return "", err
	}
	l := len(data)
	if data[l-1] == '\n' {
		return string(data[:l-1]), nil
	}
	return string(data), nil
}
