package background

import (
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"

	"pkg.deepin.io/daemon/sync/infrastructure/storage"
	"pkg.deepin.io/daemon/sync/infrastructure/utils"
	. "pkg.deepin.io/daemon/sync/modules/model"
)

type File struct {
	URI       string `json:"uri"`
	CloudName string `json:"cloud_name"`
	Md5       string `json:"md5"`
}
type FileList []*File

type Data struct {
	Version           string   `json:"version"`
	BackgroundURIs    []string `json:"background_uris"`
	GreeterBackground string   `json:"greeter_background"`
	SlideShow         string   `json:"slide_show"`

	Workspace FileList `json:"workspace"`
	Greeter   *File    `json:"greeter"`
}

const (
	Name = "background"

	backgroundDir    = ".config/deepin/dde-daemon/appearance/custom-wallpapers"
	cloudNamePrefix  = "workspace"
	backgroundMaxNum = 4
)

func (core *Data) GenCache(current DataIFC) DataIFC {
	curInfo := current.(*Data)
	length := len(curInfo.Workspace)
	coreLen := len(core.Workspace)
	var info *Data
	if length == 0 {
		length = len(core.Workspace)
		info = core
		goto fix
	}

	info = &Data{
		Version:   core.Version,
		SlideShow: core.SlideShow,
		Greeter:   core.Greeter,
	}
	if coreLen >= length {
		info.Workspace = core.Workspace[:length]
		goto fix
	}

	info.Workspace = core.Workspace
	info.Workspace = append(info.Workspace, curInfo.Workspace[coreLen:]...)

fix:
	// fix background uri name
	info.BackgroundURIs = info.Workspace.Files()[:length]
	if info.Greeter != nil {
		info.GreeterBackground = info.Greeter.URI
	}
	return info
}

func (core *Data) Merge(current, cache DataIFC) {
	curInfo := current.(*Data)
	cacheInfo := cache.(*Data)
	if curInfo.SlideShow != cacheInfo.SlideShow {
		core.SlideShow = curInfo.SlideShow
	}
	if len(core.SlideShow) != 0 {
		return
	}

	if cacheInfo.Greeter != nil && curInfo.Greeter != nil {
		if !curInfo.Greeter.Equal(cacheInfo.Greeter) {
			core.Greeter = curInfo.Greeter
		} else {
			core.Greeter.URI = curInfo.Greeter.URI
		}
	} else if curInfo.Greeter != nil {
		core.Greeter = curInfo.Greeter
		core.GreeterBackground = core.Greeter.URI
	}

	curLen := len(curInfo.Workspace)
	cacheLen := len(cacheInfo.Workspace)
	rLen := len(core.Workspace)
	var idx = 0
	for _, v := range curInfo.Workspace {
		if idx >= cacheLen {
			break
		}
		if idx >= rLen {
			break
		}
		if !v.Equal(cacheInfo.Workspace[idx]) {
			core.Workspace[idx] = v
		} else {
			// fix uri with local uri
			core.Workspace.ChangeURI(core.Workspace[idx].URI, v.URI)
		}
		idx++
	}
	for ; idx < curLen; idx++ {
		if idx < rLen {
			core.Workspace[idx] = curInfo.Workspace[idx]
		} else {
			core.Workspace = append(core.Workspace, curInfo.Workspace[idx])
		}
	}
	if curLen != 0 {
		core.BackgroundURIs = core.Workspace.Files()
	}
}

func (core *Data) Name() string {
	return Name
}

func (core *Data) Model() interface{} {
	return &Data{}
}

func (core *Data) ToIFC(v interface{}) DataIFC {
	return v.(*Data)
}

func (core *Data) Equal(target DataIFC) bool {
	info := target.(*Data)
	if len(core.SlideShow) != 0 || len(info.SlideShow) != 0 {
		return core.SlideShow == info.SlideShow
	}
	if !core.Workspace.Equal(info.Workspace) {
		return false
	}
	return core.Greeter.Equal(info.Greeter)
}

func (core *Data) Upload(cloud *utils.CloudStorage) error {
	for _, bg := range core.Workspace {
		err := core.uploadFile(cloud, bg)
		if err != nil {
			return err
		}
	}
	if core.Greeter == nil {
		return nil
	}
	return core.uploadFile(cloud, core.Greeter)
}

func (core *Data) Download(cloud *utils.CloudStorage) error {
	for _, bg := range core.Workspace {
		err := core.downloadFile(cloud, bg)
		if err != nil {
			return err
		}
	}
	core.BackgroundURIs = core.Workspace.Files()
	if core.Greeter == nil {
		return nil
	}
	err := core.downloadFile(cloud, core.Greeter)
	if err != nil {
		return err
	}
	core.GreeterBackground = core.Greeter.URI
	return nil
}

func (core *Data) uploadFile(cloud *utils.CloudStorage, info *File) error {
	prefix := utils.GetFilePath(Name, info.CloudName)
	// limit file size < 5M
	filename := utils.TrimFileURI(info.URI)
	if err := checkFileSize(filename); err != nil {
		return err
	}
	data, err := ioutil.ReadFile(filename)
	if err != nil {
		return err
	}
	_, err = cloud.Set(prefix, data, &storage.SetOptions{NewVersion: true})
	return err
}

func (core *Data) downloadFile(cloud *utils.CloudStorage, info *File) error {
	filename, exists := checkFileExist(info.URI)
	if exists {
		info.URI = "file://" + filename
		return nil
	}

	prefix := utils.GetFilePath(Name, info.CloudName)
	ret, err := cloud.Get(prefix,
		&storage.GetOptions{Version: storage.VersionLatest})
	if err != nil {
		return err
	}
	uri, err := writeBackground(ret.Content, info.URI)
	if err != nil {
		return err
	}
	info.URI = uri
	return nil
}

func (core *Data) Parse() {
	if len(core.Workspace) != 0 && (len(core.GreeterBackground) == 0 || core.Greeter != nil) {
		return
	}

	var files FileList
	for idx, uri := range core.BackgroundURIs {
		if idx >= backgroundMaxNum {
			break
		}
		md5, _ := utils.SumFileMd5(utils.TrimFileURI(uri))
		if len(md5) == 0 {
			md5 = filepath.Base(uri)
		}
		files = append(files, &File{
			URI:       uri,
			CloudName: fmt.Sprintf("%s%d", cloudNamePrefix, idx),
			Md5:       md5,
		})
	}
	core.Workspace = files

	if len(core.GreeterBackground) == 0 {
		core.Greeter = &File{}
		return
	}

	md5, _ := utils.SumFileMd5(utils.TrimFileURI(core.GreeterBackground))
	if len(md5) == 0 {
		md5 = filepath.Base(core.GreeterBackground)
	}
	core.Greeter = &File{
		URI:       core.GreeterBackground,
		CloudName: "greeter",
		Md5:       md5,
	}
}

func (list FileList) Equal(infos FileList) bool {
	len1 := len(list)
	len2 := len(infos)
	if len1 != len2 {
		return false
	}
	for i := 0; i < len1; i++ {
		if !list[i].Equal(infos[i]) {
			return false
		}
	}
	return true
}

func (list FileList) Files() []string {
	var ret []string
	for _, v := range list {
		ret = append(ret, v.URI)
	}
	return ret
}

func (list FileList) ChangeURI(src, dst string) {
	for _, v := range list {
		if v.URI == src {
			v.URI = dst
		}
	}
}

func (d *File) Equal(info *File) bool {
	return d.CloudName == info.CloudName && d.Md5 == info.Md5
}

func writeBackground(data []byte, uri string) (string, error) {
	uri = utils.TrimFileURI(uri)
	var perr *os.PathError
	err := os.MkdirAll(filepath.Dir(uri), 0755)
	if err == nil {
		err = ioutil.WriteFile(uri, data, 0644)
		if err != nil {
			perr = err.(*os.PathError)
			if perr.Err.Error() == os.ErrPermission.Error() {
				goto create
			}
			return "", err
		}
		return "file://" + uri, nil
	}

	perr = err.(*os.PathError)
	if perr.Err.Error() != os.ErrPermission.Error() {
		return "", err
	}
create:
	dir := filepath.Join(os.Getenv("HOME"), backgroundDir)
	err = os.MkdirAll(dir, 0755)
	if err != nil {
		return "", err
	}
	newfile := filepath.Join(dir, filepath.Base(uri))
	return "file://" + newfile, ioutil.WriteFile(newfile, data, 0755)
}

func checkFileExist(uri string) (string, bool) {
	filename := utils.TrimFileURI(uri)
	if utils.IsFileExist(filename) {
		return filename, true
	}
	filename = filepath.Join(os.Getenv("HOME"), backgroundDir,
		filepath.Base(uri))
	return filename, utils.IsFileExist(filename)
}

func checkFileSize(filename string) error {
	info, err := os.Stat(filename)
	if err != nil {
		return err
	}
	// limit file size < 10M
	if info.Size() > 10*1024*1024 {
		return fmt.Errorf("file is too large: %s", filename)
	}
	return nil
}
