/*
 * Copyright (C) 2014-2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "24: Flash Memory Module (FTFA)"
 */

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */
#ifdef STATE

struct {
	/* Flash Status Register (FTFA_FSTAT) */
	uint8_t ccif;
	uint8_t rdcolerr;
	uint8_t accerr;
	uint8_t fpviol;
	uint8_t mgstat0;

	/* Flash Configuration Register (FTFA_FCNFG) */
	uint8_t ccie;
	uint8_t rdcollie;
	uint8_t ersareq;
	uint8_t erssusp;

	/* Flash Security Register (FTFA_FSEC) */
	uint8_t keyen;
	uint8_t meen;
	uint8_t fslacc;
	uint8_t sec;

	/* Flash Option Register (FTFA_FOPT) */
	uint8_t fopt;

	/* Flash Common Command Object Register (FTFA_FCCOB) */
	uint8_t fccob[12];

	/* Flash Protection Register (FTFA_FPROTn) */
	uint32_t fprot;
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
NAME_(clk)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	int irq;

	irq = cpssp->NAME.ccif & cpssp->NAME.ccie;
	NAME_(irq_set)(cpssp, irq);
}

static void
NAME_(done)(struct cpssp *cpssp)
{
	cpssp->NAME.ccif = 1;
	NAME_(irq_update)(cpssp);
}

static void
NAME_(done_accerr)(struct cpssp *cpssp)
{
	cpssp->NAME.accerr = 1;
	NAME_(done)(cpssp);
}

static void
NAME_(done_fpviol)(struct cpssp *cpssp)
{
	cpssp->NAME.fpviol = 1;
	NAME_(done)(cpssp);
}

static void
NAME_(do_command)(struct cpssp *cpssp)
{
	uint32_t addr;
	uint32_t val0;
	uint32_t val1;

	addr = cpssp->NAME.fccob[1] << 16
		| cpssp->NAME.fccob[2] << 8
		| cpssp->NAME.fccob[3] << 0;
	val0 = cpssp->NAME.fccob[4] << 24
		| cpssp->NAME.fccob[5] << 16
		| cpssp->NAME.fccob[6] << 8
		| cpssp->NAME.fccob[7] << 0;
	val1 = cpssp->NAME.fccob[8] << 24
		| cpssp->NAME.fccob[9] << 16
		| cpssp->NAME.fccob[10] << 8
		| cpssp->NAME.fccob[11] << 0;

#if 0
	fprintf(stderr, "%s: cmd=0x%02x addr=0x%06x val0=0x%08x val1=0x%08x\n",
			__FUNCTION__, cpssp->NAME.fccob[0], addr, val0, val1);
#endif
	switch (cpssp->NAME.fccob[0]) {
	case 0x01:
		/* Read 1s Section */
		switch (cpssp->NAME.fccob[6]) {
		case 0x00:
		case 0x01:
		case 0x02:
			/* ok */
			break;
		default:
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		if (addr & ~(FLASH_SIZE - 1 - 3)) {
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		val0 = cpssp->NAME.fccob[4] << 8
			| cpssp->NAME.fccob[5] << 0;
		if (val0 == 0) {
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		cpssp->NAME.mgstat0 = 0;
		for ( ; 0 < val0; val0--) {
			NAME_(flash_read)(cpssp, addr, &val1);
			if (val1 != 0xffffffff) {
				cpssp->NAME.mgstat0 = 1;
			}
			addr += 4;
		}
		NAME_(done)(cpssp);
		break;
	case 0x02:
		/* Program Check */
		switch (cpssp->NAME.fccob[4]) {
		case 0x01:
		case 0x02:
			/* ok */
			break;
		default:
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		if (addr & ~(FLASH_SIZE - 1 - 3)) {
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		NAME_(flash_read)(cpssp, addr, &val0);
		cpssp->NAME.mgstat0 = (val0 != val1);
		NAME_(done)(cpssp);
	case 0x03:
		/* Read Resource */
		switch (cpssp->NAME.fccob[8]) {
		case 0x00:
			/* Program Flash 0 IFR */
			if (addr & ~0xfc) {
				NAME_(done_accerr)(cpssp);
				goto err;
			}
			NAME_(flash_read)(cpssp, addr, &val0);
			break;
		case 0x01:
			/* Version ID */
			if (addr & ~0x4) {
				NAME_(done_accerr)(cpssp);
				goto err;
			}
			val0 = 0; assert(0); /* FIXME */
			break;
		default:
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		cpssp->NAME.fccob[4] = (val0 >> 24) & 0xff;
		cpssp->NAME.fccob[5] = (val0 >> 16) & 0xff;
		cpssp->NAME.fccob[6] = (val0 >> 8) & 0xff;
		cpssp->NAME.fccob[7] = (val0 >> 0) & 0xff;
		NAME_(done)(cpssp);
		break;
	case 0x06:
		/* Program Longword */
		if (addr & ~(FLASH_SIZE - 1 - 3)) {
			/* Invalid Address */
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		if (! ((cpssp->NAME.fprot >> (addr / (FLASH_SIZE / 32))) & 1)) {
			/* Sector Write Protected */
			NAME_(done_fpviol)(cpssp);
			goto err;
		}
		NAME_(flash_write)(cpssp, addr, val0);
		NAME_(done)(cpssp);
		break;
	case 0x09:
		/* Erase Flash Sector */
		if (! ((cpssp->NAME.fprot >> (addr / (FLASH_SIZE / 32))) & 1)) {
			/* Sector Write Protected */
			NAME_(done_fpviol)(cpssp);
			goto err;
		}
		NAME_(flash_erase)(cpssp, addr, 0x400);
		NAME_(done)(cpssp);
		break;
	case 0x40:
		/* Read 1s All Blocks */
		switch (cpssp->NAME.fccob[1]) {
		case 0x00:
		case 0x01:
		case 0x02:
			/* ok */
			break;
		default:
			/* Invalid Margin */
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		cpssp->NAME.mgstat0 = 0;
		for (addr = 0x100; addr < FLASH_SIZE; addr += 4) {
			NAME_(flash_read)(cpssp, addr, &val1);
			if (val1 != 0xffffffff) {
				cpssp->NAME.mgstat0 = 1;
			}
		}
		NAME_(done)(cpssp);
		break;
	case 0x41:
		/* Read Once */
		addr = cpssp->NAME.fccob[1];
		if (addr & ~0xc) {
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		addr += 0xc0;
		NAME_(flash_read)(cpssp, addr, &val0);
		cpssp->NAME.fccob[4] = (val0 >> 24) & 0xff;
		cpssp->NAME.fccob[5] = (val0 >> 16) & 0xff;
		cpssp->NAME.fccob[6] = (val0 >> 8) & 0xff;
		cpssp->NAME.fccob[7] = (val0 >> 0) & 0xff;
		NAME_(done)(cpssp);
		break;
	case 0x43:
		/* Program Once */
		addr = cpssp->NAME.fccob[1];
		if (addr & ~0xc) {
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		addr += 0xc0;
		NAME_(flash_write)(cpssp, addr, val0);
		NAME_(done)(cpssp);
		break;
	case 0x44:
		/* Erase All Blocks */
		if (cpssp->NAME.fprot != 0xffffffff) {
			NAME_(done_accerr)(cpssp);
			goto err;
		}
		for (addr = 0; addr < FLASH_SIZE; addr += FLASH_SIZE / 32) {
			NAME_(flash_erase)(cpssp, addr, 0x400);
		}
		NAME_(done)(cpssp);
		break;
	case 0x45:
		/* Verify Backdoor Access Key */
		assert(0); /* FIXME */
		break;
	default:
		/* Unknown Command */
		NAME_(done_accerr)(cpssp);
err:	;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x00:
		if ((bs >> 0) & 1) {
			/* Flash Status Register (FTFA_FSTAT) */
			*valp |= cpssp->NAME.ccif << 7;
			*valp |= cpssp->NAME.rdcolerr << 6;
			*valp |= cpssp->NAME.accerr << 5;
			*valp |= cpssp->NAME.fpviol << 4;
			/* 3-1: Reserved */
			*valp |= cpssp->NAME.mgstat0 << 0;
		}
		if ((bs >> 1) & 1) {
			/* Flash Configuration Register (FTFA_FCNFG) */
			*valp |= cpssp->NAME.ccie << (7+8);
			*valp |= cpssp->NAME.rdcollie << (6+8);
			*valp |= cpssp->NAME.ersareq << (5+8);
			*valp |= cpssp->NAME.erssusp << (4+8);
			/* 3-0: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Flash Security Register (FTFA_FSEC) */
			*valp |= cpssp->NAME.keyen << (6+16);
			*valp |= cpssp->NAME.meen << (4+16);
			*valp |= cpssp->NAME.fslacc << (2+16);
			*valp |= cpssp->NAME.sec << (0+16);
		}
		if ((bs >> 3) & 1) {
			/* Flash Option Register (FTFA_FOPT) */
			*valp |= cpssp->NAME.fopt << 24;
		}
		break;
	case 0x04:
	case 0x08:
	case 0x0c:
		/* Flash Common Command Object Register (FTFA_FCCOB) */
		if ((bs >> 0) & 1) {
			*valp |= cpssp->NAME.fccob[addr - 4 + 3] << 0;
		}
		if ((bs >> 1) & 1) {
			*valp |= cpssp->NAME.fccob[addr - 4 + 2] << 8;
		}
		if ((bs >> 2) & 1) {
			*valp |= cpssp->NAME.fccob[addr - 4 + 1] << 16;
		}
		if ((bs >> 3) & 1) {
			*valp |= cpssp->NAME.fccob[addr - 4 + 0] << 24;
		}
		break;
	case 0x10:
		/* Flash Protection Register (FTFA_FPROTn) */
		if ((bs >> 0) & 1) {
			/* FPROT3 */
			*valp |= cpssp->NAME.fprot & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			/* FPROT2 */
			*valp |= cpssp->NAME.fprot & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			/* FPROT1 */
			*valp |= cpssp->NAME.fprot & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			/* FPROT0 */
			*valp |= cpssp->NAME.fprot & (0xff << 24);
		}
		break;
	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		break;
	}
#if 0
	fprintf(stderr, "%s: addr=0x%02x bs=0x%x val=0x%08x\n",
			__FUNCTION__, addr, bs, *valp);
#endif
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x1000 - 1;

#if 0
	fprintf(stderr, "%s: addr=0x%02x bs=0x%x val=0x%08x\n",
			__FUNCTION__, addr, bs, val);
#endif

	switch (addr) {
	case 0x00:
		if ((bs >> 0) & 1) {
			/* Status Register (FTFA_FSTAT) */
			cpssp->NAME.ccif &= ~((val >> 7) & 1);
			cpssp->NAME.rdcolerr &= ~((val >> 6) & 1);
			cpssp->NAME.accerr &= ~((val >> 5) & 1);
			cpssp->NAME.fpviol &= ~((val >> 4) & 1);
			/* Bit 3-1: Reserved */
			/* Bit 0: Read-only */
			NAME_(irq_update)(cpssp);
			if (cpssp->NAME.ccif == 0) {
				NAME_(do_command)(cpssp);
			}
		}
		if ((bs >> 1) & 1) {
			/* Flash Configuration Register (FTFA_FCNFG) */
			cpssp->NAME.ccie = (val >> (7+8)) & 1;
			cpssp->NAME.rdcollie = (val >> (6+8)) & 1;
			/* 5: Read-only */
			cpssp->NAME.erssusp = (val >> (4+8)) & 1;
			/* 3-0: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Flash Security Register (FTFA_FSEC) */
			/* 7-0: Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Flash Option Register (FTFA_FOPT) */
			/* 7-0: Read-only */
		}
		break;
	case 0x04:
	case 0x08:
	case 0x0c:
		/* Flash Common Command Object Register (FTFA_FCCOB) */
		if ((bs >> 0) & 1) {
			cpssp->NAME.fccob[addr - 4 + 3] = (val >> 0) & 0xff;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.fccob[addr - 4 + 2] = (val >> 8) & 0xff;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.fccob[addr - 4 + 1] = (val >> 16) & 0xff;
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.fccob[addr - 4 + 0] = (val >> 24) & 0xff;
		}
		break;
	case 0x10:
		/* Flash Protection Register (FTFA_FPROTn) */
		if ((bs >> 0) & 1) {
			/* FPROT3 */
			cpssp->NAME.fprot &= ~(0xff << 0);
			cpssp->NAME.fprot |= val & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			/* FPROT2 */
			cpssp->NAME.fprot &= ~(0xff << 8);
			cpssp->NAME.fprot |= val & (0xff << 8);
		}
		if ((bs >> 2) & 1) {
			/* FPROT1 */
			cpssp->NAME.fprot &= ~(0xff << 16);
			cpssp->NAME.fprot |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			/* FPROT0 */
			cpssp->NAME.fprot &= ~(0xff << 24);
			cpssp->NAME.fprot |= val & (0xff << 24);
		}
		break;
	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
	/* FIXME */
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	uint32_t val;
	int i;

	/* Flash Status Register (FTFA_FSTAT) */
	cpssp->NAME.ccif = 1;
	cpssp->NAME.rdcolerr = 0;
	cpssp->NAME.accerr = 0;
	cpssp->NAME.fpviol = 0;

	/* Flash Configuration Register (FTFA_FCNFG) */
	cpssp->NAME.ccie = 0;
	cpssp->NAME.rdcollie = 0;
	cpssp->NAME.ersareq = 0;
	cpssp->NAME.erssusp = 0;

	/* Flash Security Register (FTFA_FSEC) */
	/* Flash Option Register (FTFA_FOPT) */
	NAME_(flash_read)(cpssp, 0x40c, &val);

	cpssp->NAME.keyen = (val >> 6) & 0x3;
	cpssp->NAME.meen = (val >> 4) & 0x3;
	cpssp->NAME.fslacc = (val >> 2) & 0x3;
	cpssp->NAME.sec = (val >> 0) & 0x3;

	cpssp->NAME.fopt = (val >> 8) & 0xff;

	/* Flash Common Command Object Register (FTFA_FCCOB) */
	for (i = 0; i < 12; i++) {
		cpssp->NAME.fccob[i] = 0x00;
	}

	/* Flash Protection Register (FTFA_FPROTn) */
	NAME_(flash_read)(cpssp, 0x408, &val);

	cpssp->NAME.fprot = val;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
