;;; Autotools compatible SRFI-64 Scheme unit-test framework
;;; Copyright (C) 2016 gEDA Contributors
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


;;; Refer to the following links for more information:
;;;   https://www.gnu.org/software/automake/manual/html_node/Tests.html
;;;   http://srfi.schemers.org/srfi-64/srfi-64.html
;;;
;;;
;;; This script is launched by autotools like this:
;;; (test.scm --test-name file.scm
;;;           --log-file file.log
;;;           --trs-file file.trs
;;;           --color-tests no
;;;           --enable-hard-errors yes
;;;           --expect-failure no
;;;           -- -L . unit-tests/file.scm)
;;;
;;;
;;; autotools expect the output .trs file contain the info as
;;; follows:
;;;
;;; :test-result: value
;;;   May be used several times, say, for several procedures. Its
;;;   value must be one of: "PASS", "XFAIL", "SKIP", "FAIL",
;;;   "XPASS", "ERROR" and optionally some text after it.
;;;
;;; :recheck: value
;;;   Must be present once or not present at all. If its value is
;;;   "no", 'make recheck' won't run the corresponding test. Since
;;;   I don't know if this would be useful, let's set it to "yes"
;;;   (we could just omit it as well).
;;;
;;; :copy-in-global-log: value
;;;   Must be present once. I believe the best value for it is
;;;   "no", so we won't double log info for each test.
;;;
;;; :test-global-result: value
;;;   Must be present once if several tests are done in one
;;;   script. It should output some summary.


(use-modules (srfi srfi-64)
             (srfi srfi-26)
             (ice-9 getopt-long)
             (ice-9 pretty-print))

(define (report s port)
  (display s port)
  (display s (current-error-port)))

(define (report-runner-properties runner port)
  (let ((detailed-info
         (apply format #f "
Detailed information
====================
File ~ALine ~A

Failed form:
~A

Expected value:
~A
Actual value:
~A

Expected error:
~A
Actual error:
~A
"
                (map (lambda (prop)
                       (with-output-to-string
                         (lambda () (pretty-print
                                (test-result-ref runner prop)))))
                     '(source-file
                       source-line
                       source-form
                       expected-value
                       actual-value
                       expected-error
                       actual-error)))))
    (report detailed-info port)))

(define (custom-test-runner name log-port trs-port colored?
                            enable-hard-errors? expect-failure?)
  (let ((runner (test-runner-null))
        (group-name #f)
        (group-count 0))
    (test-runner-on-group-begin! runner
      (lambda (runner suite-name count)
        (set! group-name suite-name)
        (set! group-count 0)))
    (test-runner-on-test-begin! runner
      (lambda (runner)
        (set! group-count (1+ group-count))))
    (test-runner-on-test-end! runner
      (lambda (runner)
        (let* ((result (test-result-kind runner))
               (result-string (case result
                                ((pass xpass fail xfail skip)
                                 (string-upcase (symbol->string result)))
                                (else "ERROR"))))
          (format trs-port ":test-result: ~A\n" result-string)
          (report (format #f "~A: ~A (~A)\n"
                          result-string group-name group-count)
                  log-port)
          (when (eq? result 'fail)
            (report-runner-properties runner log-port)))))
    (test-runner-on-final! runner
      (lambda (runner) (report "\n" log-port)))
        runner))

(define (main args)
  (define (yes-no s)
    (and (string? s) (not (string=? "no" s))))

  (let* ((option-spec '((test-name (value #t))
                        (log-file (value #t))
                        (trs-file (value #t))
                        (color-tests (value #t))
                        (enable-hard-errors (value #t))
                        (expect-failure (value #t))))
         (options (getopt-long args option-spec))
         (name (option-ref options 'test-name #f))
         (log (option-ref options 'log-file #f))
         (trs (option-ref options 'trs-file #f))
         (string-colored (option-ref options 'color-tests #f))
         (string-enable-hard-errors (option-ref options 'enable-hard-errors #f))
         (string-expect-failure (option-ref options 'expect-failure #f))
         (colored? (and=> string-colored yes-no))
         (enable-hard-errors? (and=> string-enable-hard-errors yes-no))
         (expect-failure? (and=> string-expect-failure yes-no)))
    (if (and (string? log) (string? trs))
        (let* ((log-port (open-output-file log))
               (trs-port (open-output-file trs))
               (runner (custom-test-runner name
                                           log-port
                                           trs-port
                                           colored?
                                           enable-hard-errors?
                                           expect-failure?)))
          (format trs-port ":copy-in-global-log: no\n:recheck: yes\n")
          (test-runner-factory (lambda () runner))
          (let ((captured-stack #f))
            (catch #t
              (lambda ()
                (report (format #f
                                "\nLoading: ~S\n"
                                name)
                        log-port)
                (load-from-path name))
              (lambda (key . args) 
                (report (format #f
                                "ERROR:\n Unexpected exception on loading file ~S:\n~S\n"
                                name
                                (display-backtrace
                                 captured-stack
                                 (current-output-port)
                                 6
                                 5))
                        log-port))
              (lambda (key . args)
                ;; Capture the stack here:
                (set! captured-stack (make-stack #t))))
            (if captured-stack

                (primitive-exit 1)))

          (format trs-port ":test-global-result: ~A\n"
                  (if (> (test-runner-fail-count runner) 0)
                      "FAIL" "PASS"))
          (test-runner-reset runner)
          (close-port log-port)
          (close-port trs-port))
        (display "Use 'make check' to run tests.\n"))))
