#!/usr/bin/python

import Buffy
import unittest

class BuffyTestConfig(unittest.TestCase):
	def setUp(self):
		self.config = Buffy.Config("test-config.txt")

	# Test Config
	def testConfig(self):
		self.failUnless(self.config)

		self.failIf(self.config.view().read())
		self.failIf(self.config.view().empty())
		self.failUnless(self.config.view().important())
		self.assertEqual(self.config.general().interval(), 600)

		locations = self.config.locations()
		self.assertEqual(locations.size(), 4)

		# This one seems not to have been wrapped, and I have no idea why
		programs = self.config.mailPrograms()
		self.assertEqual(programs.size(), 3)

		self.assertEqual(self.config.selectedMailProgram().name(), "mutt")

		self.config.selectMailProgram("Test")
		self.assertEqual(self.config.selectedMailProgram().name(), "Test");

		folder = Buffy.MailFolder.accessFolder("maildir/empty")
		self.failUnless(folder.valid())
		self.config.mailProgram("test").run(folder);

		folder = self.config.folder("foo")
		self.assertEqual(folder.forceview(), True)
		self.assertEqual(folder.forcehide(), False)
		folder.setForceHide(True)
		self.assertEqual(folder.forceview(), False)
		self.assertEqual(folder.forcehide(), True)

		folder = self.config.folder("bar")
		self.assertEqual(folder.forceview(), False)
		self.assertEqual(folder.forcehide(), True)
		folder.setForceView(True)
		self.assertEqual(folder.forceview(), True)
		self.assertEqual(folder.forcehide(), False)

class BuffyTestMaildir(unittest.TestCase):
	# Check an empty maildir
	def testEmpty(self):
		test = Buffy.MailFolder.accessFolder("maildir/empty")
		self.failUnless(test.valid())
		self.assertEqual(test.name(), "empty")
		self.assertEqual(test.path(), "maildir/empty")
		self.assertEqual(test.type(), "Maildir")

		self.assertEqual(test.getMsgTotal(), -1)
		self.assertEqual(test.getMsgUnread(), -1)
		self.assertEqual(test.getMsgNew(), -1)
		self.assertEqual(test.getMsgFlagged(), -1)
		self.failUnless(test.changed())

		test.updateStatistics()
		self.assertEqual(test.getMsgTotal(), 0)
		self.assertEqual(test.getMsgUnread(), 0)
		self.assertEqual(test.getMsgNew(), 0)
		self.assertEqual(test.getMsgFlagged(), 0)
		self.failIf(test.changed())

		folders = Buffy.MailFolder.enumerateFolders("maildir/empty")
		self.assertEqual(len(folders), 1)

	# Check a non empty maildir
	def testNonEmpty(self):
		test = Buffy.MailFolder.accessFolder("maildir/test")
		self.failUnless(test.valid())
		self.assertEqual(test.name(), "test")
		self.assertEqual(test.path(), "maildir/test")
		self.assertEqual(test.type(), "Maildir")

		self.assertEqual(test.getMsgTotal(), -1)
		self.assertEqual(test.getMsgUnread(), -1)
		self.assertEqual(test.getMsgNew(), -1)
		self.assertEqual(test.getMsgFlagged(), -1)
		self.failUnless(test.changed())

		test.updateStatistics()
		self.assertEqual(test.getMsgTotal(), 3)
		self.assertEqual(test.getMsgUnread(), 0)
		self.assertEqual(test.getMsgNew(), 0)
		self.assertEqual(test.getMsgFlagged(), 1)
		self.failIf(test.changed())

		folders = Buffy.MailFolder.enumerateFolders("maildir/test")
		self.assertEqual(len(folders), 1)

	# Check a maildir whose root is a broken symlink
	def test_broken(self):
		test = Buffy.MailFolder.accessFolder("maildir/broken")
		self.failIf(test.valid())

		folders = Buffy.MailFolder.enumerateFolders("maildir/broken")
		self.assertEqual(len(folders), 0)

	# Check a maildir whose cur, new and tmp directories are broken symlinks
	def testBroken1(self):
		test = Buffy.MailFolder.accessFolder("maildir/broken1")
		self.failIf(test.valid())

		folders = Buffy.MailFolder.enumerateFolders("maildir/broken1")
		self.assertEqual(len(folders), 0)

	# Check a maildir whose root is a symlink pointing to a nonexisting directory
	def testLoop(self):
		test = Buffy.MailFolder.accessFolder("maildir/loop")
		self.failIf(test.valid())

		folders = Buffy.MailFolder.enumerateFolders("maildir/loop")
		self.assertEqual(len(folders), 0)

	# Check a maildir which has a submaildir that is a symlink pointing up
	def testLoop1(self):
		test = Buffy.MailFolder.accessFolder("maildir/loop1")
		self.failUnless(test.valid())

		folders = Buffy.MailFolder.enumerateFolders("maildir/loop1")
		self.assertEqual(len(folders), 1)

	# Check maildir enumeration
	def testEnumeration(self):
		folders = Buffy.MailFolder.enumerateFolders("maildir")
		self.assertEqual(len(folders), 4)

class BuffyTestMailbox(unittest.TestCase):
	# Check an empty mailbox
	def test_empty(self):
		test = Buffy.MailFolder.accessFolder("mbox/empty.mbox")
		self.failUnless(test.valid())
		self.assertEqual(test.name(), "empty.mbox")
		self.assertEqual(test.path(), "mbox/empty.mbox")
		self.assertEqual(test.type(), "Mailbox")

		self.assertEqual(test.getMsgTotal(), -1)
		self.assertEqual(test.getMsgUnread(), -1)
		self.assertEqual(test.getMsgNew(), -1)
		self.assertEqual(test.getMsgFlagged(), -1)
		self.failUnless(test.changed())

		test.updateStatistics()
		self.assertEqual(test.getMsgTotal(), 0)
		self.assertEqual(test.getMsgUnread(), 0)
		self.assertEqual(test.getMsgNew(), 0)
		self.assertEqual(test.getMsgFlagged(), 0)
		self.failIf(test.changed())

		folders = Buffy.MailFolder.enumerateFolders("mbox/empty.mbox")
		self.assertEqual(len(folders), 1)

	# Check a non empty mailbox
	def test_nonempty(self):
		test = Buffy.MailFolder.accessFolder("mbox/test.mbox")
		self.failUnless(test.valid())
		self.assertEqual(test.name(), "test.mbox")
		self.assertEqual(test.path(), "mbox/test.mbox")
		self.assertEqual(test.type(), "Mailbox")

		self.assertEqual(test.getMsgTotal(), -1)
		self.assertEqual(test.getMsgUnread(), -1)
		self.assertEqual(test.getMsgNew(), -1)
		self.assertEqual(test.getMsgFlagged(), -1)
		self.failUnless(test.changed())

		test.updateStatistics()
		self.assertEqual(test.getMsgTotal(), 3)
		self.assertEqual(test.getMsgUnread(), 0)
		self.assertEqual(test.getMsgNew(), 0)
		self.assertEqual(test.getMsgFlagged(), 1)
		self.failIf(test.changed())

		folders = Buffy.MailFolder.enumerateFolders("mbox/test.mbox")
		self.assertEqual(len(folders), 1)

	# Check a mailbox whose root is a broken symlink
	def test_broken(self):
		test = Buffy.MailFolder.accessFolder("mbox/broken.mbox")
		self.failIf(test.valid())

		folders = Buffy.MailFolder.enumerateFolders("mbox/broken.mbox")
		self.assertEqual(len(folders), 0)

	# Check enumeration of a directory with mailboxes
	def test_enumeration(self):
		folders = Buffy.MailFolder.enumerateFolders("mbox")
		self.assertEqual(len(folders), 2)

if __name__ == "__main__":
	unittest.main()

## Read configuration
#conf = Buffy.Config()
#
## Decide if a mailbox is to be displayed
## Please note that all configuration methods also come with 'set_*' conterparts
#def view(x):
#	if x.getMsgUnread() > 0:
#		return True
#
#	if conf.view_important().get() and x.getMsgFlagged() > 0:
#		return True
#
#	if conf.view_read().get() and x.getMsgTotal() > 0:
#		return True
#
#	return conf.view_empty().get() or conf.folder(x.path()).get("forceview", "false").get() == "true"
#
#
## Get the folder objects from the paths in the config file
#folders = []
#for path in conf.folder_locations().get():
#	folders += Buffy.MailFolder_enumerateFolders(path)
#
##while True:
#	for f in folders:
##		print "Scanning " + f.name()
#		if f.changed():
#			f.updateStatistics()
#			if view(f):
#				print "%s: %d %d %d %d" % (f.name(), f.getMsgTotal(), f.getMsgUnread(), f.getMsgNew(), f.getMsgFlagged())
#	#time.sleep(conf.update_interval().get())


