/*
 * Copyright (C) 2020 Uniontech Technology Co., Ltd.
 *
 * Author:     xinbo wang <wangxinbo@uniontech.com>
 *
 * Maintainer: xinbo wang <wangxinbo@uniontech.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef DND_SECURITY_H
#define DND_SECURITY_H

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <stdint.h>
#include "dtk_utils.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef enum _Session_Type {
    SecClipboardCopy = 1 << 0
} SessionType;

typedef enum _Permission {
    PermissionAllow = 1 << 0,
    PermissionDeny = 1 << 1,
    PermissionDelay = 1 << 2
} Permission;

// security
typedef int (*InitBackendPtr)();
typedef void (*DestroyBackendPtr)();
typedef int  (*GetSecuritySessionPtr)(SessionType /*types*/);
typedef int  (*DestroySecuritySessionPtr)(int /*session*/);
typedef int  (*DoSecurityVerifyPtr)(void* data, int /*types*/, int /*client*/, int /*target*/);
typedef void (*ReportSecurityVerifiedPtr)(int /*session*/, Permission /*result*/);
typedef struct dtk_array* (*GetSecurityClientsPtr)();

typedef struct _Dnd_Security {
    void *backend;
    char *xdgSessionType;
    bool isWayland;
    void* userData;

    // init and destory
    InitBackendPtr InitBackend;
    DestroyBackendPtr DestroyBackend;

    // dnd security
    GetSecuritySessionPtr GetSecuritySession;
    DestroySecuritySessionPtr DestroySecuritySession;
    DoSecurityVerifyPtr DoSecurityVerify; //callback func
    ReportSecurityVerifiedPtr ReportSecurityVerified;
    GetSecurityClientsPtr GetSecurityClients;
} DndSecurity;

typedef struct _Dnd_Security *DndSecurityPtr;
extern DndSecurityPtr pDndSec;

/**
 * @brief init dtkdisplay content
 */
int InitDtkDisplay();

/**
 * @brief destory dtkdisplay content
 */
void DestoryDtkDisplay();

/**
 * @brief The controlled client refers to the client created
 * the Selection or DRAG SOURCE that can only be accessed
 * by partially trusted clients
 * return session ID.
 * @param types: session type
 * @return session id
 */
int GetSecuritySession(SessionType types);

/**
 * @brief Use session ID to release the controlled client.
 * Cancel the current controlled client. After the
 * cancellation, the shear information sent by the
 * client can be accessed by all programs
 * @param session: session id
 * @return 0: success -1: failed
 */
int DestroySecuritySession(int session);

/**
 * @brief It is generated by the shear safety encryption module.
 * It is a token pair, one for registered control clients, and
 * the other is used to create a special parameter for creating
 * shear plate sources in the protocol, such as (Series or attributes)
 * @param client : client process id
 * @param target : target process id
 * @return 0: success -1: failed
 */
void DoSecurityVerifyCallback(void* data, DoSecurityVerifyPtr func);

/**
 * @brief report security verified result
 * @param session: session id
 * @param result: whether the client is allowed to access
 */
void ReportSecurityVerified(int session, Permission result);

/**
 * @brief get process id of all current control processes
 * @return process id array
 */
struct dtk_array* GetSecurityClients();

#ifdef __cplusplus
}
#endif

#endif
