/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/cauchy/cdf' );
import Cauchy = require( './../../../../../base/dists/cauchy/ctor' );
import entropy = require( './../../../../../base/dists/cauchy/entropy' );
import logcdf = require( './../../../../../base/dists/cauchy/logcdf' );
import logpdf = require( './../../../../../base/dists/cauchy/logpdf' );
import median = require( './../../../../../base/dists/cauchy/median' );
import mode = require( './../../../../../base/dists/cauchy/mode' );
import pdf = require( './../../../../../base/dists/cauchy/pdf' );
import quantile = require( './../../../../../base/dists/cauchy/quantile' );

/**
* Interface describing the `cauchy` namespace.
*/
interface Namespace {
	/**
	* Cauchy distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param x0 - location parameter
	* @param gamma - scale parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 0.0, 1.0 );
	* // returns ~0.852
	*
	* var mycdf = ns.cdf.factory( 1.5, 3.0 );
	*
	* y = mycdf( 1.0 );
	* // returns ~0.447
	*/
	cdf: typeof cdf;

	/**
	* Cauchy distribution.
	*/
	Cauchy: typeof Cauchy;

	/**
	* Returns the differential entropy of a Cauchy distribution.
	*
	* ## Notes
	*
	* -   If provided `gamma <= 0`, the function returns `NaN`.
	*
	* @param x0 - location parameter
	* @param gamma - scale parameter
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 10.0, 5.0 );
	* // returns ~4.14
	*
	* @example
	* var v = ns.entropy( 7.0, 0.1 );
	* // returns ~0.228
	*
	* @example
	* var v = ns.entropy( 10.0, -0.5 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Cauchy distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param x0 - location parameter
	* @param gamma - scale parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 2.0, 0.0, 1.0 );
	* // returns ~-0.16
	*
	* var mylogcdf = ns.logcdf.factory( 1.5, 3.0 );
	*
	* y = mylogcdf( 1.0 );
	* // returns ~-0.805
	*/
	logcdf: typeof logcdf;

	/**
	* Cauchy distribution natural logarithm of probability density function (logPDF).
	*
	* @param x - input value
	* @param x0 - location parameter
	* @param gamma - scale parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.0, 1.0 );
	* // returns ~-2.765
	*
	* var mylogpdf = ns.logpdf.factory( 10.0, 2.0 );
	*
	* y = mylogpdf( 10.0 );
	* // returns ~-1.839
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the median of a Cauchy distribution.
	*
	* ## Notes
	*
	* -   If provided `gamma <= 0`, the function returns `NaN`.
	*
	* @param x0 - location parameter
	* @param gamma - scale parameter
	* @returns median
	*
	* @example
	* var v = ns.median( 10.0, 5.0 );
	* // returns 10.0
	*
	* @example
	* var v = ns.median( 7.0, 0.1 );
	* // returns 7.0
	*
	* @example
	* var v = ns.median( 10.0, -0.5 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Returns the mode of a Cauchy distribution.
	*
	* ## Notes
	*
	* -   If provided `gamma <= 0`, the function returns `NaN`.
	*
	* @param x0 - location parameter
	* @param gamma - scale parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 10.0, 5.0 );
	* // returns 10.0
	*
	* @example
	* var v = ns.mode( 7.0, 0.1 );
	* // returns 7.0
	*
	* @example
	* var v = ns.mode( 10.0, -0.5 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Cauchy distribution probability density function (PDF).
	*
	* @param x - input value
	* @param x0 - location parameter
	* @param gamma - scale parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.0, 1.0 );
	* // returns ~0.063
	*
	* var mypdf = factory( 10.0, 2.0 );
	*
	* y = mypdf( 10.0 );
	* // returns ~0.159
	*/
	pdf: typeof pdf;

	/**
	* Cauchy distribution quantile function.
	*
	* @param p - input value
	* @param x0 - location parameter
	* @param gamma - scale parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0.0, 1.0 );
	* // returns ~1.376
	*
	* var myQuantile = ns.quantile.factory( 10.0, 2.0 );
	*
	* y = myQuantile( 0.5 );
	* // returns 10.0
	*/
	quantile: typeof quantile;
}

/**
* Cauchy distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
