/*
	SRG - Squid Report Generator
	Destination site filtering.
	
	Copyright (C) 2008 Matt Brown <matt@mattb.net.nz>

	This file is part of SRG.

	SRG is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation version 2.

	SRG is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with SRG; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/

#include "srg.h"
#include "prototypes.h"

extern Resolver *dnscache;

/* Global Data Structures */
list<char *> exclude_hostnames;
list<exclude_netblock> exclude_netblocks;

/* Initialise the IP address to user mappings */
void init_sitefilter(void) {

	/* Exit if no site filter file specified */
	if (!srg.sitefilter)
		return;

	/* Read the file */
	Line *iLine = new Line(srg.sitefilter);
	if (iLine->getError() || iLine->eof()) {
		fprintf(stderr, "%s: Could not open the destination site "
				"filter file: %s\n", progname, srg.sitefilter);
		exit(1);
	}

	/* Process each line of the file */
	while(!iLine->eof()) {
		char *excludeLine = iLine->getline();

		/* Check for comment lines */
		if (excludeLine[0] == '\0' || excludeLine[0] == '#')
			continue;

		/* Parse the exclusion type and data */
		char hostname[1024] = {'\0'};
		char network[1024] = {'\0'};
		char netmask[1024] = {'\0'};
		if (sscanf(excludeLine, "host %1023s", hostname) != 1) {
			if (sscanf(excludeLine, "subnet %1023s %1023s",
						network, netmask) != 2) {
				fprintf(stderr, "%s: Invalid line in destination site "
						"filter file: %s\n", progname, srg.sitefilter);
				exit(1);
			}
		}

		/* If a hostname was set, store it and continue */
		if (hostname[0] != '\0') {
			exclude_hostnames.push_back(strdup(hostname));
			continue;
		}

		/* If no hostname was set, try and parse the network/netmask */
		exclude_netblock this_netblock;
		if (inet_aton(network, &this_netblock.network)==0) {
			fprintf(stderr, "Invalid network Address in destination "
					"site filter file: %s\n", network);
			exit(1);
		}
		if (inet_aton(netmask, &this_netblock.netmask)==0) {
			fprintf(stderr, "Invalid netmask in destination site "
					"filter file: %s\n", netmask);
			exit(1);
		}

		/* Add this entry to the list */
		exclude_netblocks.push_back(this_netblock);
	}

	/* Free memory */
	delete iLine;

}

/* Returns true if the specified destination site matches one of the entries in
 * the destination site filter file. */
bool destination_is_excluded(const char *site_hostname) {

	/* Check destination against excluded hostnames first */
	list<char *>::const_iterator iter;
	for (iter=exclude_hostnames.begin(); iter != exclude_hostnames.end();
			iter++) {
		if (strcasecmp((const char *)(*iter), site_hostname) == 0) {
			return true;
		}
	}

	/* Don't continue if there are not netblocks, resolving is slow. */
	if (exclude_netblocks.empty()) {
		return false;
	}

	/* Didn't match any hostnames, resolve to an IP and check netblocks */
	in_addr dest_ip;
	if (dnscache->get_ip(site_hostname, &dest_ip) != 0) {
		/* Could not resolve site_hostname to an IP */
		return false;
	}
	list<exclude_netblock>::const_iterator nb_iter;
	for (nb_iter=exclude_netblocks.begin();
			nb_iter != exclude_netblocks.end();
			nb_iter++) {
		if ((dest_ip.s_addr&(*nb_iter).netmask.s_addr)==
				(*nb_iter).network.s_addr) {
			return true;
		}
	}

	/*  Not found in either list*/
	return false;
}
