
use Test::More tests => 18;
use HTML::Dashboard;

# The tests in this file may look a bit odd. Here is the rationale:
# The only truly meaningful function generatin a "testable" value
# is as_HTML().
#
# The "formal" approach to testing this would be to parse the
# generated HTML into a tree, and then to confirm that the tree 
# has the correct properties (number of children, value of nodes, etc).
# Very painful.
#
# At the same time, the correctness of the generated tables is very
# easy to discern visually, when they are rendered by a browser.
# 
# So, I have generated a set of dashboards, which exercise all of the
# features of this module, verified the correctness of the output
# visually, and then included the visually verified HTML here as
# "test case". This should serve as sufficient basis for regression
# tests.

my $data = [ [ 'A',  0, 'foo', 3.12, 'smith' ],
	     [ 'B',  1, 'bar', 6.22, 'smith' ],
	     [ 'C',  2, 'gzx', 3.16, 'smith' ],
	     [ 'D',  3, 'baz', 7.12, 'allan' ],
	     [ 'E',  4, 'bnk', 3.47, 'allan' ],
	     [ 'F',  5, 'hue', 4.18, 'joedo' ],
	     [ 'G',  6, 'gzi', 3.13, 'joedo' ],
	     [ 'H',  7, 'fct', 7.15, 'joedo' ],
	     [ 'I',  8, 'blo', 2.42, 'joedo' ] ];

my $dash;

# ------------------------------------------------------------

# All Default

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

ok( $dash->as_HTML() eq 
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >A</td><td >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );


# ------------------------------------------------------------
# Striped Reports

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_first_row( '#f0fff0' );
$dash->set_odd_row(   '#fffacd' );
$dash->set_even_row(  style => 'font-style: italic' );
$dash->set_last_row(  '#e0ffff' );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr style="font-style: italic; background-color: #f0fff0" ><td >A</td><td >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="background-color: #fffacd;" ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr style="font-style: italic" ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: #fffacd;" ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr style="font-style: italic" ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: #fffacd;" ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr style="font-style: italic" ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: #fffacd;" ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr style="font-style: italic; background-color: #e0ffff" ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Striped Columns

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_even_col( '#bfefff' );
$dash->set_last_col( style => 'font-size: xx-small; color: red' );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td style="background-color: #bfefff;" >A</td><td >0</td><td style="background-color: #bfefff;" >foo</td><td >3.12</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td style="background-color: #bfefff;" >B</td><td >1</td><td style="background-color: #bfefff;" >bar</td><td >6.22</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >smith</td></tr>
<tr ><td style="background-color: #bfefff;" >C</td><td >2</td><td style="background-color: #bfefff;" >gzx</td><td >3.16</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td style="background-color: #bfefff;" >D</td><td >3</td><td style="background-color: #bfefff;" >baz</td><td >7.12</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >allan</td></tr>
<tr ><td style="background-color: #bfefff;" >E</td><td >4</td><td style="background-color: #bfefff;" >bnk</td><td >3.47</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td style="background-color: #bfefff;" >F</td><td >5</td><td style="background-color: #bfefff;" >hue</td><td >4.18</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >joedo</td></tr>
<tr ><td style="background-color: #bfefff;" >G</td><td >6</td><td style="background-color: #bfefff;" >gzi</td><td >3.13</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td style="background-color: #bfefff;" >H</td><td >7</td><td style="background-color: #bfefff;" >fct</td><td >7.15</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >joedo</td></tr>
<tr ><td style="background-color: #bfefff;" >I</td><td >8</td><td style="background-color: #bfefff;" >blo</td><td >2.42</td><td style="font-size: xx-small; color: red; background-color: #bfefff" >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Cell Triggers (triggered on 3rd column)

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_cell_low( 3, sub { my ( $x ) = @_; $x < 4 }, 'lime' );
$dash->set_cell_med( 3, sub { my ( $x ) = @_; $x >=4 }, 'yellow' );
$dash->set_cell_hi(  3, sub { my ( $x ) = @_; $x > 7 }, 
                     style => 'background-color: red; font-weight: bold' );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >A</td><td >0</td><td >foo</td><td style="background-color: lime;" >3.12</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >B</td><td >1</td><td >bar</td><td style="background-color: yellow;" >6.22</td><td >smith</td></tr>
<tr ><td >C</td><td >2</td><td >gzx</td><td style="background-color: lime;" >3.16</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >D</td><td >3</td><td >baz</td><td style="background-color: red; font-weight: bold" >7.12</td><td >allan</td></tr>
<tr ><td >E</td><td >4</td><td >bnk</td><td style="background-color: lime;" >3.47</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >F</td><td >5</td><td >hue</td><td style="background-color: yellow;" >4.18</td><td >joedo</td></tr>
<tr ><td >G</td><td >6</td><td >gzi</td><td style="background-color: lime;" >3.13</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >H</td><td >7</td><td >fct</td><td style="background-color: red; font-weight: bold" >7.15</td><td >joedo</td></tr>
<tr ><td >I</td><td >8</td><td >blo</td><td style="background-color: lime;" >2.42</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Row Triggers

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_row_low( sub { my ( $r ) = @_; $r->[1]%3 || $r->[3]>4 }, 'lime' );
$dash->set_row_hi(  sub { my ( $r ) = @_; $r->[3]>7 }, 'red' );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >A</td><td >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="background-color: lime;" ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr style="background-color: lime;" ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: red;" ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr style="background-color: lime;" ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: lime;" ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: red;" ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr style="background-color: lime;" ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Column Triggers

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_col_low( 1, sub { my ( $x ) = @_; $x < 4 }, 'lime' );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >A</td><td style="background-color: lime;" >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >B</td><td style="background-color: lime;" >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr ><td >C</td><td style="background-color: lime;" >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >D</td><td style="background-color: lime;" >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >E</td><td style="background-color: lime;" >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >F</td><td style="background-color: lime;" >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr ><td >G</td><td style="background-color: lime;" >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >H</td><td style="background-color: lime;" >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr ><td >I</td><td style="background-color: lime;" >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Triggers blend with stripes

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_row_hi( sub { $_[0]->[1] % 2 == 1 }, style => 'color: red' );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >A</td><td >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="color: red; background-color: #eeeeee" ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="color: red; background-color: #eeeeee" ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="color: red; background-color: #eeeeee" ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="color: red; background-color: #eeeeee" ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# But Triggers Clobber each other

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_row_low( sub { $_[0]->[1]%2 }, 'lime' );
$dash->set_col_med( 1, sub { $_[0] == 8 }, 'yellow' );
$dash->set_cell_hi( 1, sub { $_[0]>=4 && $_[0]<5 }, 'red' );
$dash->set_cell_hi( 3, sub { $_[0]>4 && $_[0]<5 }, 'red' );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >A</td><td style="background-color: yellow;" >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="background-color: lime;" ><td >B</td><td style="background-color: yellow;" >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr ><td >C</td><td style="background-color: yellow;" >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: lime;" ><td >D</td><td style="background-color: yellow;" >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >E</td><td style="background-color: red;" >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: lime;" ><td >F</td><td style="background-color: yellow;" >5</td><td >hue</td><td style="background-color: red;" >4.18</td><td >joedo</td></tr>
<tr ><td >G</td><td style="background-color: yellow;" >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: lime;" ><td >H</td><td style="background-color: yellow;" >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr ><td >I</td><td style="background-color: yellow;" >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Captions

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_captions( 'First Col', 'A Number', 'String', 'Some Float', 'Name' );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >
<tr ><th >First Col</th><th >A Number</th><th >String</th><th >Some Float</th><th >Name</th></tr>

<tr ><td >A</td><td >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# View

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_view( 4, 1, 3 );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >smith</td><td >0</td><td >3.12</td></tr>
<tr style="background-color: #eeeeee;" ><td >smith</td><td >1</td><td >6.22</td></tr>
<tr ><td >smith</td><td >2</td><td >3.16</td></tr>
<tr style="background-color: #eeeeee;" ><td >allan</td><td >3</td><td >7.12</td></tr>
<tr ><td >allan</td><td >4</td><td >3.47</td></tr>
<tr style="background-color: #eeeeee;" ><td >joedo</td><td >5</td><td >4.18</td></tr>
<tr ><td >joedo</td><td >6</td><td >3.13</td></tr>
<tr style="background-color: #eeeeee;" ><td >joedo</td><td >7</td><td >7.15</td></tr>
<tr ><td >joedo</td><td >8</td><td >2.42</td></tr>

</table>

) );

# ------------------------------------------------------------
# Simple Sort (Numeric sort on 3rd column)

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_sort( sub { my ( $x, $y ) = @_; $x->[3] <=> $y->[3] } );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >A</td><td >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Fancy Sort (First on Names, then reverse on numbers)

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_sort( sub { my ( $x, $y ) = @_; 
		       $x->[4] cmp $y->[4] || $y->[1] <=> $x->[1] } );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr ><td >A</td><td >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>

</table>

) );

# ------------------------------------------------------------
# Simple Format (Truncate float to single digit)

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_format( 3, sub { my ( $x ) = @_; sprintf( "%.1f", $x ) } );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >A</td><td >0</td><td >foo</td><td >3.1</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >B</td><td >1</td><td >bar</td><td >6.2</td><td >smith</td></tr>
<tr ><td >C</td><td >2</td><td >gzx</td><td >3.2</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >D</td><td >3</td><td >baz</td><td >7.1</td><td >allan</td></tr>
<tr ><td >E</td><td >4</td><td >bnk</td><td >3.5</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >F</td><td >5</td><td >hue</td><td >4.2</td><td >joedo</td></tr>
<tr ><td >G</td><td >6</td><td >gzi</td><td >3.1</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >H</td><td >7</td><td >fct</td><td >7.2</td><td >joedo</td></tr>
<tr ><td >I</td><td >8</td><td >blo</td><td >2.4</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Collate (Glue column 0 and 1 together, suppress column 0 through view)

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_view( 1, 2, 3, 4 );
$dash->set_collate( 1, sub { my ( @r ) = @_; $r[0] . '-' . $r[1] } );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >

<tr ><td >A-0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >B-1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr ><td >C-2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >D-3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >E-4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >F-5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr ><td >G-6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >H-7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr ><td >I-8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );

# ------------------------------------------------------------
# Pagination

$dash = HTML::Dashboard->new();
$dash->set_data_without_captions( $data );

$dash->set_pagesize( 5 );

$dash->set_captions( 'First Col', 'A Number', 'String', 'Some Float', 'Name' );
$dash->set_table( style => 'margin-bottom: 1em' );

ok( $dash->get_pagecount() == 2 );

ok( $dash->as_HTML(0) eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" style="margin-bottom: 1em" >
<tr ><th >First Col</th><th >A Number</th><th >String</th><th >Some Float</th><th >Name</th></tr>

<tr ><td >A</td><td >0</td><td >foo</td><td >3.12</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>

</table>

) );

ok( $dash->as_HTML(1) eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" style="margin-bottom: 1em" >
<tr ><th >First Col</th><th >A Number</th><th >String</th><th >Some Float</th><th >Name</th></tr>

<tr ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );


# ============================================================
# Some tests which are not part of the gallery...

# ------------------------------------------------------------
# Set with captions

$dash = HTML::Dashboard->new();
$dash->set_data_with_captions( $data );

ok( $dash->as_HTML() eq
q(

<!-- Table generated by HTML::Dashboard - www.cpan.org -->
<table border="1" >
<tr ><th >A</th><th >0</th><th >foo</th><th >3.12</th><th >smith</th></tr>

<tr ><td >B</td><td >1</td><td >bar</td><td >6.22</td><td >smith</td></tr>
<tr style="background-color: #eeeeee;" ><td >C</td><td >2</td><td >gzx</td><td >3.16</td><td >smith</td></tr>
<tr ><td >D</td><td >3</td><td >baz</td><td >7.12</td><td >allan</td></tr>
<tr style="background-color: #eeeeee;" ><td >E</td><td >4</td><td >bnk</td><td >3.47</td><td >allan</td></tr>
<tr ><td >F</td><td >5</td><td >hue</td><td >4.18</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >G</td><td >6</td><td >gzi</td><td >3.13</td><td >joedo</td></tr>
<tr ><td >H</td><td >7</td><td >fct</td><td >7.15</td><td >joedo</td></tr>
<tr style="background-color: #eeeeee;" ><td >I</td><td >8</td><td >blo</td><td >2.42</td><td >joedo</td></tr>

</table>

) );
