// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_AUTOFILL_CORE_BROWSER_AUTOFILL_TRIGGER_DETAILS_H_
#define COMPONENTS_AUTOFILL_CORE_BROWSER_AUTOFILL_TRIGGER_DETAILS_H_

#include "components/autofill/core/browser/autofill_type.h"

namespace autofill {

// Specifies the source that triggered autofilling a form. Differently from
// `AutofillSuggestionTriggerSource` this enum does not mean the action that
// triggered autofill (for example
// AutofillSuggestionTriggerSource::kFormControlElementClicked) but only the
// internal surface that provided the autofill experience (like `kPopup`) or
// internal mechanisms/business logic (like kSelectOptionsChanged). For example
// AutofillSuggestionTriggerSource::kFormControlElementClicked "action" can lead
// autofill to be "surfaced" from `kPopup` or `kFastCheckout`.
enum class AutofillTriggerSource {
  kNone = 0,
  // Autofill was triggered from accepting a suggestion in the Autofill popup.
  kPopup = 1,
  // Autofill was triggered from accepting a suggestion in the keyboard
  // accessory.
  kKeyboardAccessory = 2,
  // Autofill was triggered from accepting a suggestion in the touch to fill for
  // credit cards bottom sheet.
  kTouchToFillCreditCard = 3,
  // Refill was triggered from the forms seen event. This includes cases where a
  // refill was triggered right after a non-refill Autofill invocation - in this
  // case the original trigger source got lost.
  kFormsSeen = 4,
  // Refill was triggered from blink when the selected option of a <select>
  // control is changed.
  kSelectOptionsChanged = 5,
  // Refill was triggered from blink when the input element is in the autofilled
  // state and the value has been changed by JavaScript.
  kJavaScriptChangedAutofilledValue = 6,
  // Autofill was applied after unlocking a server card with the CVC. The
  // original trigger source got lost. This should not happen.
  kCreditCardCvcPopup = 7,
  // Autofill was triggered from a Fast Checkout run.
  kFastCheckout = 8,
  // Filling/preview was triggered through a suggestion generated by manual
  // fallbacks from the Chrome context menu. Contrary to regular address
  // Autofill, Autofill with this trigger source fills both classified and
  // unclassified form fields.
  kManualFallback = 9
};

// Holds the details about a filling operation. The `trigger_source` field
// defines what triggered the autofill experience, i.e kFastCheckout or kPopup.
// The `field_types_to_fill` specifies which fields Autofill is going to take
// into account when filling the form.
struct AutofillTriggerDetails {
  AutofillTriggerSource trigger_source = AutofillTriggerSource::kNone;
  // Default to considering every field type when filling the form.
  FieldTypeSet field_types_to_fill = kAllFieldTypes;
};

}  // namespace autofill

#endif  // COMPONENTS_AUTOFILL_CORE_BROWSER_AUTOFILL_TRIGGER_DETAILS_H_
